/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.ml.mms.archive;

import com.amazonaws.ml.mms.archive.DownloadModelException;
import com.amazonaws.ml.mms.archive.Hex;
import com.amazonaws.ml.mms.archive.InvalidModelException;
import com.amazonaws.ml.mms.archive.LegacyManifest;
import com.amazonaws.ml.mms.archive.Manifest;
import com.amazonaws.ml.mms.archive.ModelException;
import com.amazonaws.ml.mms.archive.ModelNotFoundException;
import com.amazonaws.ml.mms.archive.ZipUtils;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Enumeration;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModelArchive {
    private static final Logger logger = LoggerFactory.getLogger(ModelArchive.class);
    public static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Pattern URL_PATTERN = Pattern.compile("http(s)?://.*", 2);
    private static final String MANIFEST_FILE = "MANIFEST.json";
    private Manifest manifest;
    private String url;
    private File modelDir;
    private boolean extracted;

    public ModelArchive(Manifest manifest, String url, File modelDir, boolean extracted) {
        this.manifest = manifest;
        this.url = url;
        this.modelDir = modelDir;
        this.extracted = extracted;
    }

    public static ModelArchive downloadModel(String modelStore, String url) throws ModelException, IOException {
        if (URL_PATTERN.matcher(url).matches()) {
            File modelDir = ModelArchive.download(url);
            return ModelArchive.load(url, modelDir, true);
        }
        if (url.contains("..")) {
            throw new ModelNotFoundException("Relative path is not allowed in url: " + url);
        }
        if (modelStore == null) {
            throw new ModelNotFoundException("Model store has not been configured.");
        }
        File modelLocation = new File(modelStore, url);
        if (!modelLocation.exists()) {
            throw new ModelNotFoundException("Model not found in model store: " + url);
        }
        if (modelLocation.isFile()) {
            try (FileInputStream is = new FileInputStream(modelLocation);){
                File unzipDir = ModelArchive.unzip(is, null);
                ModelArchive modelArchive = ModelArchive.load(url, unzipDir, true);
                return modelArchive;
            }
        }
        return ModelArchive.load(url, modelLocation, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void migrate(File legacyModelFile, File destination) throws InvalidModelException, IOException {
        boolean failed = true;
        try (ZipFile zip = new ZipFile(legacyModelFile);
             ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(destination));){
            ZipEntry manifestEntry = zip.getEntry(MANIFEST_FILE);
            if (manifestEntry == null) {
                throw new InvalidModelException("Missing manifest file in model archive.");
            }
            JsonParser parser = new JsonParser();
            InputStream is = zip.getInputStream(manifestEntry);
            InputStreamReader reader = new InputStreamReader(is, StandardCharsets.UTF_8);
            JsonObject json = (JsonObject)parser.parse(reader);
            JsonPrimitive version = json.getAsJsonPrimitive("specificationVersion");
            if (version != null && "1.0".equals(version.getAsString())) {
                throw new InvalidModelException("model archive is already in 1.0 version.");
            }
            LegacyManifest legacyManifest = GSON.fromJson((JsonElement)json, LegacyManifest.class);
            Manifest manifest = legacyManifest.migrate();
            zos.putNextEntry(new ZipEntry("MAR-INF/"));
            zos.putNextEntry(new ZipEntry("MAR-INF/MANIFEST.json"));
            zos.write(GSON.toJson(manifest).getBytes(StandardCharsets.UTF_8));
            Enumeration<? extends ZipEntry> en = zip.entries();
            while (en.hasMoreElements()) {
                ZipEntry entry = en.nextElement();
                String name = entry.getName();
                if (MANIFEST_FILE.equalsIgnoreCase(name) || name.startsWith(".")) continue;
                zos.putNextEntry(new ZipEntry(name));
                if (entry.isDirectory()) continue;
                IOUtils.copy(zip.getInputStream(entry), (OutputStream)zos);
            }
            failed = false;
        }
        finally {
            if (failed) {
                FileUtils.deleteQuietly(destination);
            }
        }
    }

    private static File download(String path) throws ModelException, IOException {
        HttpURLConnection conn;
        try {
            URL url = new URL(path);
            conn = (HttpURLConnection)url.openConnection();
            if (conn.getResponseCode() != 200) {
                throw new DownloadModelException("Failed to download model from: " + path + ", code: " + conn.getResponseCode());
            }
        }
        catch (RuntimeException | MalformedURLException e) {
            throw new ModelNotFoundException("Invalid model url: " + path, e);
        }
        catch (IOException e) {
            throw new DownloadModelException("Failed to download model from: " + path, e);
        }
        try {
            String eTag = conn.getHeaderField("ETag");
            File tmpDir = new File(System.getProperty("java.io.tmpdir"));
            File modelDir = new File(tmpDir, "models");
            FileUtils.forceMkdir(modelDir);
            if (eTag != null) {
                File dir;
                if (eTag.startsWith("\"") && eTag.endsWith("\"") && eTag.length() > 2) {
                    eTag = eTag.substring(1, eTag.length() - 1);
                }
                if ((dir = new File(modelDir, eTag)).exists()) {
                    logger.info("model folder already exists: {}", (Object)eTag);
                    return dir;
                }
            }
            return ModelArchive.unzip(conn.getInputStream(), eTag);
        }
        catch (SocketTimeoutException e) {
            throw new DownloadModelException("Download model timeout: " + path, e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ModelArchive load(String url, File dir, boolean extracted) throws InvalidModelException, IOException {
        boolean failed = true;
        try {
            Manifest manifest;
            File manifestFile = new File(dir, "MAR-INF/MANIFEST.json");
            if (manifestFile.exists()) {
                manifest = ModelArchive.readFile(manifestFile, Manifest.class);
            } else {
                manifestFile = new File(dir, MANIFEST_FILE);
                boolean nested = false;
                if (!manifestFile.exists()) {
                    manifestFile = ModelArchive.findFile(dir, MANIFEST_FILE, true);
                    nested = true;
                }
                if (manifestFile == null) {
                    manifest = new Manifest();
                } else {
                    LegacyManifest legacyManifest = ModelArchive.readFile(manifestFile, LegacyManifest.class);
                    manifest = legacyManifest.migrate();
                    File modelDir = manifestFile.getParentFile();
                    if (extracted && nested) {
                        ModelArchive.moveToTopLevel(modelDir, dir);
                    }
                }
            }
            failed = false;
            ModelArchive modelArchive = new ModelArchive(manifest, url, dir, extracted);
            return modelArchive;
        }
        finally {
            if (extracted && failed) {
                FileUtils.deleteQuietly(dir);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static <T> T readFile(File file, Class<T> type) throws InvalidModelException, IOException {
        try (InputStreamReader r = new InputStreamReader((InputStream)new FileInputStream(file), StandardCharsets.UTF_8);){
            T t = GSON.fromJson((Reader)r, type);
            return t;
        }
        catch (JsonParseException e) {
            throw new InvalidModelException("Failed to parse signature.json.", e);
        }
    }

    private static File findFile(File dir, String fileName, boolean recursive) {
        File[] list = dir.listFiles();
        if (list == null) {
            return null;
        }
        for (File file : list) {
            if (recursive && file.isDirectory()) {
                File f = ModelArchive.findFile(file, fileName, false);
                if (f == null) continue;
                return f;
            }
            if (!file.getName().equalsIgnoreCase(fileName)) continue;
            return file;
        }
        return null;
    }

    private static void moveToTopLevel(File from, File to) throws IOException {
        File[] list = from.listFiles();
        if (list != null) {
            for (File file : list) {
                if (file.isDirectory()) {
                    FileUtils.moveDirectoryToDirectory(file, to, false);
                    continue;
                }
                FileUtils.moveFileToDirectory(file, to, false);
            }
        }
    }

    public static File unzip(InputStream is, String eTag) throws IOException {
        File dir;
        MessageDigest md;
        File tmpDir = FileUtils.getTempDirectory();
        File modelDir = new File(tmpDir, "models");
        FileUtils.forceMkdir(modelDir);
        File tmp = File.createTempFile("model", ".download");
        FileUtils.forceDelete(tmp);
        FileUtils.forceMkdir(tmp);
        try {
            md = MessageDigest.getInstance("SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            throw new AssertionError((Object)e);
        }
        ZipUtils.unzip(new DigestInputStream(is, md), tmp);
        if (eTag == null) {
            eTag = Hex.toHexString(md.digest());
        }
        if ((dir = new File(modelDir, eTag)).exists()) {
            FileUtils.deleteDirectory(tmp);
            logger.info("model folder already exists: {}", (Object)eTag);
            return dir;
        }
        FileUtils.moveDirectory(tmp, dir);
        return dir;
    }

    public void validate() throws InvalidModelException {
        Manifest.Model model = this.manifest.getModel();
        try {
            if (model == null) {
                throw new InvalidModelException("Missing Model entry in manifest file.");
            }
            if (model.getModelName() == null) {
                throw new InvalidModelException("Model name is not defined.");
            }
            if (model.getHandler() == null) {
                throw new InvalidModelException("Model handler is not defined.");
            }
            if (this.manifest.getRuntime() == null) {
                throw new InvalidModelException("Runtime is not defined or invalid.");
            }
        }
        catch (InvalidModelException e) {
            this.clean();
            throw e;
        }
    }

    public Manifest getManifest() {
        return this.manifest;
    }

    public String getUrl() {
        return this.url;
    }

    public File getModelDir() {
        return this.modelDir;
    }

    public String getModelName() {
        return this.manifest.getModel().getModelName();
    }

    public void clean() {
        if (this.url != null && this.extracted) {
            FileUtils.deleteQuietly(this.modelDir);
        }
    }
}

