import hashlib
from datetime import datetime
from itertools import chain

from nonebot_plugin_alconna import Image, Text, on_alconna
from nonebot_plugin_localstore import get_cache_dir
from pypinyin import Style, pinyin

from ..config import memes_config
from ..manager import meme_manager
from ..request import MemeKeyWithProperties, render_meme_list
from .utils import UserId

memes_cache_dir = get_cache_dir("nonebot_plugin_memes_api")

help_matcher = on_alconna(
    "表情包制作",
    aliases={"表情列表", "头像表情包", "文字表情包"},
    block=True,
    priority=11,
    use_cmd_start=True,
)


@help_matcher.handle()
async def _(user_id: UserId):
    memes = meme_manager.get_memes()
    list_image_config = memes_config.memes_list_image_config

    sort_by = list_image_config.sort_by
    sort_reverse = list_image_config.sort_reverse
    if sort_by == "key":
        memes = sorted(memes, key=lambda meme: meme.key, reverse=sort_reverse)
    elif sort_by == "keywords":
        memes = sorted(
            memes,
            key=lambda meme: "".join(
                chain.from_iterable(pinyin(meme.keywords[0], style=Style.TONE3))
            ),
            reverse=sort_reverse,
        )
    elif sort_by == "date_created":
        memes = sorted(memes, key=lambda meme: meme.date_created, reverse=sort_reverse)
    elif sort_by == "date_modified":
        memes = sorted(memes, key=lambda meme: meme.date_modified, reverse=sort_reverse)

    label_new_timedelta = list_image_config.label_new_timedelta
    # label_hot_frequency = list_image_config.label_hot_frequency

    meme_list: list[MemeKeyWithProperties] = []
    for meme in memes:
        labels = []
        if datetime.now() - meme.date_created < label_new_timedelta:
            labels.append("new")
        # if await get_meme_frequency(meme.key) >= label_hot_frequency:
        #     labels.append("hot")
        disabled = not meme_manager.check(user_id, meme.key)
        meme_list.append(
            MemeKeyWithProperties(meme_key=meme.key, disabled=disabled, labels=labels)
        )

    # cache rendered meme list
    meme_list_hashable = [
        (
            {
                "key": meme.key,
                "keywords": meme.keywords,
                "shortcuts": [
                    shortcut.humanized or shortcut.key for shortcut in meme.shortcuts
                ],
                "tags": sorted(meme.tags),
            },
            prop,
        )
        for meme, prop in zip(memes, meme_list)
    ]
    meme_list_hash = hashlib.md5(str(meme_list_hashable).encode("utf8")).hexdigest()
    meme_list_cache_file = memes_cache_dir / f"{meme_list_hash}.jpg"
    if not meme_list_cache_file.exists():
        img = await render_meme_list(
            meme_list,
            text_template=list_image_config.text_template,
            add_category_icon=list_image_config.add_category_icon,
        )
        with open(meme_list_cache_file, "wb") as f:
            f.write(img)
    else:
        img = meme_list_cache_file.read_bytes()

    msg = Text(
        "触发方式：“关键词 + 图片/文字”\n"
        "发送 “表情详情 + 关键词” 查看表情参数和预览\n"
        "目前支持的表情列表："
    ) + Image(raw=img)
    await msg.send()
