from logging import INFO
from pathlib import Path
from typing import Dict, Optional, Sequence, Tuple, Union

import torch
from flwr.common.logger import log
from flwr.common.typing import Config, Scalar

from fl4health.checkpointing.client_module import ClientCheckpointModule
from fl4health.clients.basic_client import BasicClient
from fl4health.utils.config import narrow_dict_type
from fl4health.utils.losses import LossMeterType
from fl4health.utils.metrics import Metric


class FlashClient(BasicClient):
    def __init__(
        self,
        data_path: Path,
        metrics: Sequence[Metric],
        device: torch.device,
        loss_meter_type: LossMeterType = LossMeterType.AVERAGE,
        checkpointer: Optional[ClientCheckpointModule] = None,
    ) -> None:
        """
        This client is used to perform client-side training associated with the Flash method described in
        https://proceedings.mlr.press/v202/panchal23a/panchal23a.pdf. Flash is designed to handle statistical
        heterogeneity and concept drift in federated learning through client-side early stopping and server-side
        drift-aware adaptive optimization.

        Args:
            data_path (Path): Path to the data directory.
            metrics (Sequence[Metric]): List of metrics to be used for evaluation.
            device (torch.device): Device to be used for training.
            loss_meter_type (LossMeterType, optional):
                Type of loss meter to be used. Defaults to LossMeterType.AVERAGE.
            checkpointer (Optional[ClientCheckpointModule], optional): Checkpointer module defining when and how to do
                checkpointing during client-side training. No checkpointing is done if not provided. Defaults to None.
        """
        super().__init__(
            data_path=data_path,
            metrics=metrics,
            device=device,
            loss_meter_type=loss_meter_type,
            checkpointer=checkpointer,
        )
        # gamma: Threshold for early stopping based on the change in validation loss.
        self.gamma: Optional[float] = None

    def process_config(self, config: Config) -> Tuple[Union[int, None], Union[int, None], int, bool]:
        local_epochs, local_steps, current_server_round, evaluate_after_fit = super().process_config(config)
        if local_steps is not None:
            raise ValueError(
                "Training by steps is not applicable for FLASH clients.\
                  Please define 'local_epochs' in your config instead"
            )
        return local_epochs, local_steps, current_server_round, evaluate_after_fit

    def train_by_epochs(
        self, epochs: int, current_round: Optional[int] = None
    ) -> Tuple[Dict[str, float], Dict[str, Scalar]]:
        self.model.train()
        local_step = 0
        previous_loss = float("inf")
        report_data: dict = {"round": current_round}
        for local_epoch in range(epochs):
            self.train_metric_manager.clear()
            self.train_loss_meter.clear()
            self._log_header_str(current_round, local_epoch)
            report_data.update({"fit_epoch": local_epoch})
            for input, target in self.train_loader:
                if self.is_empty_batch(input):
                    log(INFO, "Empty batch generated by data loader. Skipping step.")
                    continue

                input = self._move_data_to_device(input)
                target = self._move_data_to_device(target)
                losses, preds = self.train_step(input, target)
                self.train_loss_meter.update(losses)
                self.train_metric_manager.update(preds, target)
                self.update_after_step(local_step, current_round)
                report_data.update({"fit_losses": losses.as_dict(), "fit_step": self.total_steps})
                report_data.update(self.get_client_specific_reports())
                self.reports_manager.report(report_data, current_round, local_epoch, self.total_steps)
                self.total_steps += 1
                local_step += 1

            metrics = self.train_metric_manager.compute()
            loss_dict = self.train_loss_meter.compute().as_dict()
            current_loss, _ = self.validate()

            self._log_results(
                loss_dict,
                metrics,
                current_round=current_round,
                current_epoch=local_epoch,
            )

            if self.gamma is not None and previous_loss - current_loss < self.gamma / (local_epoch + 1):
                log(
                    INFO,
                    f"Early stopping at epoch {local_epoch} with loss change {abs(previous_loss - current_loss)}\
                        and gamma {self.gamma}",
                )
                break

            previous_loss = current_loss

        return loss_dict, metrics

    def setup_client(self, config: Config) -> None:
        super().setup_client(config)
        if "gamma" in config:
            self.gamma = narrow_dict_type(config, "gamma", float)
        else:
            log(INFO, "Gamma not present in config. Early stopping is disabled.")
