"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const package_json_1 = require("delay/package.json");
const fs = require("fs");
const bundling_1 = require("../lib/bundling");
jest.mock('@aws-cdk/aws-lambda');
const writeFileSyncMock = jest.spyOn(fs, 'writeFileSync').mockReturnValue();
const existsSyncOriginal = fs.existsSync;
const existsSyncMock = jest.spyOn(fs, 'existsSync');
beforeEach(() => {
    jest.clearAllMocks();
});
test('Parcel bundling', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: '/cache-dir',
        projectRoot: '/project',
        parcelEnvironment: {
            KEY: 'value',
        },
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            environment: {
                KEY: 'value',
            },
            volumes: [{ containerPath: '/parcel-cache', hostPath: '/cache-dir' }],
            workingDirectory: '/asset-input/folder',
            command: [
                'bash', '-c', 'parcel build /asset-input/folder/entry.ts --target cdk-lambda --cache-dir /parcel-cache',
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        'cdk-lambda': '/asset-output/index.js',
        'targets': {
            'cdk-lambda': {
                context: 'node',
                includeNodeModules: {
                    'aws-sdk': false,
                },
                sourceMap: false,
                minify: false,
                engines: {
                    node: '>= 12',
                },
            },
        },
    }));
});
test('Parcel with Windows paths', () => {
    bundling_1.Bundling.parcel({
        entry: 'C:\\my-project\\lib\\entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: '/cache-dir',
        projectRoot: 'C:\\my-project',
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('C:\\my-project', expect.objectContaining({
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringContaining('/lib/entry.ts'),
            ]),
        }),
    }));
});
test('Parcel bundling with externals and dependencies', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        externalModules: ['abc'],
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                'parcel build /asset-input/folder/entry.ts --target cdk-lambda && mv /asset-input/.package.json /asset-output/package.json && cd /asset-output && npm install',
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: expect.objectContaining({
            'cdk-lambda': expect.objectContaining({
                includeNodeModules: {
                    delay: false,
                    abc: false,
                },
            }),
        }),
    }));
    // Correctly writes dummy package.json
    expect(writeFileSyncMock).toHaveBeenCalledWith('/project/.package.json', JSON.stringify({
        dependencies: {
            delay: package_json_1.version,
        },
    }));
});
test('Detects yarn.lock', () => {
    existsSyncMock.mockImplementation((p) => {
        if (/yarn.lock/.test(p.toString())) {
            return true;
        }
        return existsSyncOriginal(p);
    });
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringMatching(/yarn\.lock.+yarn install/),
            ]),
        }),
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJ1bmRsaW5nLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSxvREFBb0Q7QUFDcEQsd0NBQThDO0FBQzlDLHFEQUE2RDtBQUM3RCx5QkFBeUI7QUFDekIsOENBQTJDO0FBRTNDLElBQUksQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztBQUNqQyxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFFLGVBQWUsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO0FBQzVFLE1BQU0sa0JBQWtCLEdBQUcsRUFBRSxDQUFDLFVBQVUsQ0FBQztBQUN6QyxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxZQUFZLENBQUMsQ0FBQztBQUVwRCxVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO0FBQ3ZCLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsRUFBRTtJQUMzQixtQkFBUSxDQUFDLE1BQU0sQ0FBQztRQUNkLEtBQUssRUFBRSwwQkFBMEI7UUFDakMsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztRQUM1QixRQUFRLEVBQUUsWUFBWTtRQUN0QixXQUFXLEVBQUUsVUFBVTtRQUN2QixpQkFBaUIsRUFBRTtZQUNqQixHQUFHLEVBQUUsT0FBTztTQUNiO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsZ0NBQWdDO0lBQ2hDLE1BQU0sQ0FBQyxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLFVBQVUsRUFBRTtRQUN0RCxhQUFhLEVBQUUsb0JBQWEsQ0FBQyxNQUFNO1FBQ25DLFFBQVEsRUFBRSxNQUFNLENBQUMsZ0JBQWdCLENBQUM7WUFDaEMsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxPQUFPO2FBQ2I7WUFDRCxPQUFPLEVBQUUsQ0FBQyxFQUFFLGFBQWEsRUFBRSxlQUFlLEVBQUUsUUFBUSxFQUFFLFlBQVksRUFBRSxDQUFDO1lBQ3JFLGdCQUFnQixFQUFFLHFCQUFxQjtZQUN2QyxPQUFPLEVBQUU7Z0JBQ1AsTUFBTSxFQUFFLElBQUksRUFBRSx5RkFBeUY7YUFDeEc7U0FDRixDQUFDO0tBQ0gsQ0FBQyxDQUFDO0lBRUgsaUNBQWlDO0lBQ2pDLE1BQU0sSUFBSSxHQUFHLGlCQUFpQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDN0MsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUN4QyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7UUFDMUQsWUFBWSxFQUFFLHdCQUF3QjtRQUN0QyxTQUFTLEVBQUU7WUFDVCxZQUFZLEVBQUU7Z0JBQ1osT0FBTyxFQUFFLE1BQU07Z0JBQ2Ysa0JBQWtCLEVBQUU7b0JBQ2xCLFNBQVMsRUFBRSxLQUFLO2lCQUNqQjtnQkFDRCxTQUFTLEVBQUUsS0FBSztnQkFDaEIsTUFBTSxFQUFFLEtBQUs7Z0JBQ2IsT0FBTyxFQUFFO29CQUNQLElBQUksRUFBRSxPQUFPO2lCQUNkO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO0lBQ3JDLG1CQUFRLENBQUMsTUFBTSxDQUFDO1FBQ2QsS0FBSyxFQUFFLCtCQUErQjtRQUN0QyxPQUFPLEVBQUUsb0JBQU8sQ0FBQyxXQUFXO1FBQzVCLFFBQVEsRUFBRSxZQUFZO1FBQ3RCLFdBQVcsRUFBRSxnQkFBZ0I7S0FDOUIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLGlCQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsb0JBQW9CLENBQUMsZ0JBQWdCLEVBQUUsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1FBQ3BGLFFBQVEsRUFBRSxNQUFNLENBQUMsZ0JBQWdCLENBQUM7WUFDaEMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUM7Z0JBQzlCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7YUFDekMsQ0FBQztTQUNILENBQUM7S0FDSCxDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlEQUFpRCxFQUFFLEdBQUcsRUFBRTtJQUMzRCxtQkFBUSxDQUFDLE1BQU0sQ0FBQztRQUNkLEtBQUssRUFBRSwwQkFBMEI7UUFDakMsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztRQUM1QixXQUFXLEVBQUUsVUFBVTtRQUN2QixlQUFlLEVBQUUsQ0FBQyxLQUFLLENBQUM7UUFDeEIsV0FBVyxFQUFFLENBQUMsT0FBTyxDQUFDO0tBQ3ZCLENBQUMsQ0FBQztJQUVILGdDQUFnQztJQUNoQyxNQUFNLENBQUMsaUJBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLEVBQUU7UUFDdEQsYUFBYSxFQUFFLG9CQUFhLENBQUMsTUFBTTtRQUNuQyxRQUFRLEVBQUUsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ2hDLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsSUFBSTtnQkFDWiw4SkFBOEo7YUFDL0o7U0FDRixDQUFDO0tBQ0gsQ0FBQyxDQUFDO0lBRUgsaUNBQWlDO0lBQ2pDLE1BQU0sSUFBSSxHQUFHLGlCQUFpQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDN0MsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUN4QyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7UUFDMUQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztZQUMvQixZQUFZLEVBQUUsTUFBTSxDQUFDLGdCQUFnQixDQUFDO2dCQUNwQyxrQkFBa0IsRUFBRTtvQkFDbEIsS0FBSyxFQUFFLEtBQUs7b0JBQ1osR0FBRyxFQUFFLEtBQUs7aUJBQ1g7YUFDRixDQUFDO1NBQ0gsQ0FBQztLQUNILENBQUMsQ0FBQyxDQUFDO0lBRUosc0NBQXNDO0lBQ3RDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLHdCQUF3QixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7UUFDdEYsWUFBWSxFQUFFO1lBQ1osS0FBSyxFQUFFLHNCQUFZO1NBQ3BCO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7SUFDN0IsY0FBYyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBYyxFQUFFLEVBQUU7UUFDbkQsSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxFQUFFO1lBQ2xDLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFDRCxPQUFPLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQy9CLENBQUMsQ0FBQyxDQUFDO0lBRUgsbUJBQVEsQ0FBQyxNQUFNLENBQUM7UUFDZCxLQUFLLEVBQUUsMEJBQTBCO1FBQ2pDLE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7UUFDNUIsV0FBVyxFQUFFLFVBQVU7UUFDdkIsV0FBVyxFQUFFLENBQUMsT0FBTyxDQUFDO0tBQ3ZCLENBQUMsQ0FBQztJQUVILGdDQUFnQztJQUNoQyxNQUFNLENBQUMsaUJBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLEVBQUU7UUFDdEQsYUFBYSxFQUFFLG9CQUFhLENBQUMsTUFBTTtRQUNuQyxRQUFRLEVBQUUsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ2hDLE9BQU8sRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDO2dCQUM5QixNQUFNLENBQUMsY0FBYyxDQUFDLDBCQUEwQixDQUFDO2FBQ2xELENBQUM7U0FDSCxDQUFDO0tBQ0gsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJcbmltcG9ydCB7IENvZGUsIFJ1bnRpbWUgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IEFzc2V0SGFzaFR5cGUgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IHZlcnNpb24gYXMgZGVsYXlWZXJzaW9uIH0gZnJvbSAnZGVsYXkvcGFja2FnZS5qc29uJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCB7IEJ1bmRsaW5nIH0gZnJvbSAnLi4vbGliL2J1bmRsaW5nJztcblxuamVzdC5tb2NrKCdAYXdzLWNkay9hd3MtbGFtYmRhJyk7XG5jb25zdCB3cml0ZUZpbGVTeW5jTW9jayA9IGplc3Quc3B5T24oZnMsICd3cml0ZUZpbGVTeW5jJykubW9ja1JldHVyblZhbHVlKCk7XG5jb25zdCBleGlzdHNTeW5jT3JpZ2luYWwgPSBmcy5leGlzdHNTeW5jO1xuY29uc3QgZXhpc3RzU3luY01vY2sgPSBqZXN0LnNweU9uKGZzLCAnZXhpc3RzU3luYycpO1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgamVzdC5jbGVhckFsbE1vY2tzKCk7XG59KTtcblxudGVzdCgnUGFyY2VsIGJ1bmRsaW5nJywgKCkgPT4ge1xuICBCdW5kbGluZy5wYXJjZWwoe1xuICAgIGVudHJ5OiAnL3Byb2plY3QvZm9sZGVyL2VudHJ5LnRzJyxcbiAgICBydW50aW1lOiBSdW50aW1lLk5PREVKU18xMl9YLFxuICAgIGNhY2hlRGlyOiAnL2NhY2hlLWRpcicsXG4gICAgcHJvamVjdFJvb3Q6ICcvcHJvamVjdCcsXG4gICAgcGFyY2VsRW52aXJvbm1lbnQ6IHtcbiAgICAgIEtFWTogJ3ZhbHVlJyxcbiAgICB9LFxuICB9KTtcblxuICAvLyBDb3JyZWN0bHkgYnVuZGxlcyB3aXRoIHBhcmNlbFxuICBleHBlY3QoQ29kZS5mcm9tQXNzZXQpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKCcvcHJvamVjdCcsIHtcbiAgICBhc3NldEhhc2hUeXBlOiBBc3NldEhhc2hUeXBlLkJVTkRMRSxcbiAgICBidW5kbGluZzogZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgS0VZOiAndmFsdWUnLFxuICAgICAgfSxcbiAgICAgIHZvbHVtZXM6IFt7IGNvbnRhaW5lclBhdGg6ICcvcGFyY2VsLWNhY2hlJywgaG9zdFBhdGg6ICcvY2FjaGUtZGlyJyB9XSxcbiAgICAgIHdvcmtpbmdEaXJlY3Rvcnk6ICcvYXNzZXQtaW5wdXQvZm9sZGVyJyxcbiAgICAgIGNvbW1hbmQ6IFtcbiAgICAgICAgJ2Jhc2gnLCAnLWMnLCAncGFyY2VsIGJ1aWxkIC9hc3NldC1pbnB1dC9mb2xkZXIvZW50cnkudHMgLS10YXJnZXQgY2RrLWxhbWJkYSAtLWNhY2hlLWRpciAvcGFyY2VsLWNhY2hlJyxcbiAgICAgIF0sXG4gICAgfSksXG4gIH0pO1xuXG4gIC8vIENvcnJlY3RseSB1cGRhdGVzIHBhY2thZ2UuanNvblxuICBjb25zdCBjYWxsID0gd3JpdGVGaWxlU3luY01vY2subW9jay5jYWxsc1swXTtcbiAgZXhwZWN0KGNhbGxbMF0pLnRvTWF0Y2goJ3BhY2thZ2UuanNvbicpO1xuICBleHBlY3QoSlNPTi5wYXJzZShjYWxsWzFdKSkudG9FcXVhbChleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgJ2Nkay1sYW1iZGEnOiAnL2Fzc2V0LW91dHB1dC9pbmRleC5qcycsXG4gICAgJ3RhcmdldHMnOiB7XG4gICAgICAnY2RrLWxhbWJkYSc6IHtcbiAgICAgICAgY29udGV4dDogJ25vZGUnLFxuICAgICAgICBpbmNsdWRlTm9kZU1vZHVsZXM6IHtcbiAgICAgICAgICAnYXdzLXNkayc6IGZhbHNlLFxuICAgICAgICB9LFxuICAgICAgICBzb3VyY2VNYXA6IGZhbHNlLFxuICAgICAgICBtaW5pZnk6IGZhbHNlLFxuICAgICAgICBlbmdpbmVzOiB7XG4gICAgICAgICAgbm9kZTogJz49IDEyJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ1BhcmNlbCB3aXRoIFdpbmRvd3MgcGF0aHMnLCAoKSA9PiB7XG4gIEJ1bmRsaW5nLnBhcmNlbCh7XG4gICAgZW50cnk6ICdDOlxcXFxteS1wcm9qZWN0XFxcXGxpYlxcXFxlbnRyeS50cycsXG4gICAgcnVudGltZTogUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICBjYWNoZURpcjogJy9jYWNoZS1kaXInLFxuICAgIHByb2plY3RSb290OiAnQzpcXFxcbXktcHJvamVjdCcsXG4gIH0pO1xuXG4gIGV4cGVjdChDb2RlLmZyb21Bc3NldCkudG9IYXZlQmVlbkNhbGxlZFdpdGgoJ0M6XFxcXG15LXByb2plY3QnLCBleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgYnVuZGxpbmc6IGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICAgIGNvbW1hbmQ6IGV4cGVjdC5hcnJheUNvbnRhaW5pbmcoW1xuICAgICAgICBleHBlY3Quc3RyaW5nQ29udGFpbmluZygnL2xpYi9lbnRyeS50cycpLFxuICAgICAgXSksXG4gICAgfSksXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdQYXJjZWwgYnVuZGxpbmcgd2l0aCBleHRlcm5hbHMgYW5kIGRlcGVuZGVuY2llcycsICgpID0+IHtcbiAgQnVuZGxpbmcucGFyY2VsKHtcbiAgICBlbnRyeTogJy9wcm9qZWN0L2ZvbGRlci9lbnRyeS50cycsXG4gICAgcnVudGltZTogUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICBwcm9qZWN0Um9vdDogJy9wcm9qZWN0JyxcbiAgICBleHRlcm5hbE1vZHVsZXM6IFsnYWJjJ10sXG4gICAgbm9kZU1vZHVsZXM6IFsnZGVsYXknXSxcbiAgfSk7XG5cbiAgLy8gQ29ycmVjdGx5IGJ1bmRsZXMgd2l0aCBwYXJjZWxcbiAgZXhwZWN0KENvZGUuZnJvbUFzc2V0KS50b0hhdmVCZWVuQ2FsbGVkV2l0aCgnL3Byb2plY3QnLCB7XG4gICAgYXNzZXRIYXNoVHlwZTogQXNzZXRIYXNoVHlwZS5CVU5ETEUsXG4gICAgYnVuZGxpbmc6IGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICAgIGNvbW1hbmQ6IFtcbiAgICAgICAgJ2Jhc2gnLCAnLWMnLFxuICAgICAgICAncGFyY2VsIGJ1aWxkIC9hc3NldC1pbnB1dC9mb2xkZXIvZW50cnkudHMgLS10YXJnZXQgY2RrLWxhbWJkYSAmJiBtdiAvYXNzZXQtaW5wdXQvLnBhY2thZ2UuanNvbiAvYXNzZXQtb3V0cHV0L3BhY2thZ2UuanNvbiAmJiBjZCAvYXNzZXQtb3V0cHV0ICYmIG5wbSBpbnN0YWxsJyxcbiAgICAgIF0sXG4gICAgfSksXG4gIH0pO1xuXG4gIC8vIENvcnJlY3RseSB1cGRhdGVzIHBhY2thZ2UuanNvblxuICBjb25zdCBjYWxsID0gd3JpdGVGaWxlU3luY01vY2subW9jay5jYWxsc1swXTtcbiAgZXhwZWN0KGNhbGxbMF0pLnRvTWF0Y2goJ3BhY2thZ2UuanNvbicpO1xuICBleHBlY3QoSlNPTi5wYXJzZShjYWxsWzFdKSkudG9FcXVhbChleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgdGFyZ2V0czogZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgJ2Nkay1sYW1iZGEnOiBleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgICAgIGluY2x1ZGVOb2RlTW9kdWxlczoge1xuICAgICAgICAgIGRlbGF5OiBmYWxzZSxcbiAgICAgICAgICBhYmM6IGZhbHNlLFxuICAgICAgICB9LFxuICAgICAgfSksXG4gICAgfSksXG4gIH0pKTtcblxuICAvLyBDb3JyZWN0bHkgd3JpdGVzIGR1bW15IHBhY2thZ2UuanNvblxuICBleHBlY3Qod3JpdGVGaWxlU3luY01vY2spLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKCcvcHJvamVjdC8ucGFja2FnZS5qc29uJywgSlNPTi5zdHJpbmdpZnkoe1xuICAgIGRlcGVuZGVuY2llczoge1xuICAgICAgZGVsYXk6IGRlbGF5VmVyc2lvbixcbiAgICB9LFxuICB9KSk7XG59KTtcblxudGVzdCgnRGV0ZWN0cyB5YXJuLmxvY2snLCAoKSA9PiB7XG4gIGV4aXN0c1N5bmNNb2NrLm1vY2tJbXBsZW1lbnRhdGlvbigocDogZnMuUGF0aExpa2UpID0+IHtcbiAgICBpZiAoL3lhcm4ubG9jay8udGVzdChwLnRvU3RyaW5nKCkpKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG4gICAgcmV0dXJuIGV4aXN0c1N5bmNPcmlnaW5hbChwKTtcbiAgfSk7XG5cbiAgQnVuZGxpbmcucGFyY2VsKHtcbiAgICBlbnRyeTogJy9wcm9qZWN0L2ZvbGRlci9lbnRyeS50cycsXG4gICAgcnVudGltZTogUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICBwcm9qZWN0Um9vdDogJy9wcm9qZWN0JyxcbiAgICBub2RlTW9kdWxlczogWydkZWxheSddLFxuICB9KTtcblxuICAvLyBDb3JyZWN0bHkgYnVuZGxlcyB3aXRoIHBhcmNlbFxuICBleHBlY3QoQ29kZS5mcm9tQXNzZXQpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKCcvcHJvamVjdCcsIHtcbiAgICBhc3NldEhhc2hUeXBlOiBBc3NldEhhc2hUeXBlLkJVTkRMRSxcbiAgICBidW5kbGluZzogZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgY29tbWFuZDogZXhwZWN0LmFycmF5Q29udGFpbmluZyhbXG4gICAgICAgIGV4cGVjdC5zdHJpbmdNYXRjaGluZygveWFyblxcLmxvY2suK3lhcm4gaW5zdGFsbC8pLFxuICAgICAgXSksXG4gICAgfSksXG4gIH0pO1xufSk7XG4iXX0=