"""
Reference counting for Perception (Phase 4).

This module provides functions to count cross-file symbol references.
Analyzes AST to find Name, Call, and Attribute nodes that reference known symbols.

No I/O operations - receives parsed data only.
"""

from __future__ import annotations

import ast
from collections import defaultdict

from deal import post, pre

from invar.core.models import FileInfo, PerceptionMap, SymbolKind, SymbolRefs


@pre(lambda source, known_symbols: isinstance(source, str) and len(source) > 0)
@post(lambda result: isinstance(result, list))
def find_references_in_source(source: str, known_symbols: set[str]) -> list[tuple[str, int]]:
    """
    Find references to known symbols in source code.

    Returns list of (symbol_name, line_number) for each reference found.
    Deduplicates: each (symbol, line) pair counted once.

    Examples:
        >>> refs = find_references_in_source("x = foo()\\nbar(x)", {"foo", "bar"})
        >>> sorted(refs)
        [('bar', 2), ('foo', 1)]
        >>> find_references_in_source("x = unknown()", {"foo"})
        []
    """
    try:
        tree = ast.parse(source)
    except (SyntaxError, TypeError, ValueError):
        return []

    seen: set[tuple[str, int]] = set()

    for node in ast.walk(tree):
        # Count function calls: foo()
        if isinstance(node, ast.Call) and isinstance(node.func, ast.Name):
            name = node.func.id
            if name in known_symbols:
                line = getattr(node, "lineno", 0)
                seen.add((name, line))

    return list(seen)


@pre(lambda file_infos: isinstance(file_infos, list))
@post(lambda result: isinstance(result, dict))
def build_symbol_table(file_infos: list[FileInfo]) -> dict[str, str]:
    """
    Build a mapping of symbol names to their defining file.

    Returns dict of {symbol_name: file_path}.

    Examples:
        >>> from invar.core.models import FileInfo, Symbol, SymbolKind
        >>> sym = Symbol(name="foo", kind=SymbolKind.FUNCTION, line=1, end_line=5)
        >>> info = FileInfo(path="core/calc.py", lines=10, symbols=[sym])
        >>> table = build_symbol_table([info])
        >>> table["foo"]
        'core/calc.py'
    """
    symbol_table: dict[str, str] = {}

    for file_info in file_infos:
        for symbol in file_info.symbols:
            if symbol.kind in (SymbolKind.FUNCTION, SymbolKind.CLASS):
                # Use simple name (may have collisions, that's OK for now)
                symbol_table[symbol.name] = file_info.path

    return symbol_table


@pre(lambda file_infos, sources: isinstance(file_infos, list))
def count_cross_file_references(
    file_infos: list[FileInfo], sources: dict[str, str]
) -> dict[str, int]:
    """
    Count cross-file references for all symbols.

    Returns dict of {"file::symbol": reference_count}.
    Only counts references from OTHER files (excludes self-references).

    Examples:
        >>> from invar.core.models import FileInfo, Symbol, SymbolKind
        >>> sym = Symbol(name="foo", kind=SymbolKind.FUNCTION, line=1, end_line=5)
        >>> info = FileInfo(path="a.py", lines=10, symbols=[sym])
        >>> sources = {"a.py": "def foo(): pass", "b.py": "foo()"}
        >>> info2 = FileInfo(path="b.py", lines=5, symbols=[])
        >>> refs = count_cross_file_references([info, info2], sources)
        >>> refs.get("a.py::foo", 0)
        1
    """
    # Build symbol table: name -> defining file
    symbol_table = build_symbol_table(file_infos)
    known_symbols = set(symbol_table.keys())

    # Count references from each file
    ref_counts: dict[str, int] = defaultdict(int)

    for file_info in file_infos:
        source = sources.get(file_info.path, "")
        if not source:
            continue

        references = find_references_in_source(source, known_symbols)

        for symbol_name, _ in references:
            defining_file = symbol_table.get(symbol_name)
            if defining_file and defining_file != file_info.path:
                # Cross-file reference: increment count
                key = f"{defining_file}::{symbol_name}"
                ref_counts[key] += 1

    return dict(ref_counts)


@pre(lambda file_infos, sources, project_root: (
    isinstance(file_infos, list) and
    isinstance(project_root, str) and len(project_root) > 0
))
def build_perception_map(
    file_infos: list[FileInfo], sources: dict[str, str], project_root: str
) -> PerceptionMap:
    """
    Build complete perception map with reference counts.

    Examples:
        >>> from invar.core.models import FileInfo, Symbol, SymbolKind
        >>> sym = Symbol(name="foo", kind=SymbolKind.FUNCTION, line=1, end_line=5)
        >>> info = FileInfo(path="a.py", lines=10, symbols=[sym])
        >>> pm = build_perception_map([info], {"a.py": "def foo(): pass"}, "/test")
        >>> pm.total_symbols
        1
    """
    ref_counts = count_cross_file_references(file_infos, sources)

    # Build SymbolRefs list
    symbol_refs: list[SymbolRefs] = []
    total_symbols = 0

    for file_info in file_infos:
        for symbol in file_info.symbols:
            if symbol.kind in (SymbolKind.FUNCTION, SymbolKind.CLASS):
                key = f"{file_info.path}::{symbol.name}"
                count = ref_counts.get(key, 0)
                symbol_refs.append(
                    SymbolRefs(
                        symbol=symbol,
                        file_path=file_info.path,
                        ref_count=count,
                    )
                )
                total_symbols += 1

    # Sort by reference count (descending)
    symbol_refs.sort(key=lambda sr: sr.ref_count, reverse=True)

    try:
        return PerceptionMap(
            project_root=project_root,
            total_files=len(file_infos),
            total_symbols=total_symbols,
            symbols=symbol_refs,
        )
    except Exception:
        # Handle CrossHair symbolic value validation failures
        return PerceptionMap(
            project_root="",
            total_files=0,
            total_symbols=0,
            symbols=[],
        )
