from abc import ABC
from sphere.core.schemas.buildings import Buildings
from sphere.core.schemas.abstract_vulnerability_function import AbstractVulnerabilityFunction


class BaseVulnerabilityFunction(AbstractVulnerabilityFunction):
    """
    Base vulnerability function class providing common functionality.
    
    This class provides a concrete implementation of AbstractVulnerabilityFunction
    with basic damage calculation methods that can be extended by specific
    vulnerability function implementations.
    """
    
    def __init__(self, buildings: Buildings):
        """
        Initialize the base vulnerability function.
        
        Args:
            buildings (Buildings): Buildings object containing building data.
        """
        self.buildings = buildings
    
    def calculate_vulnerability(self):
        """
        Calculate vulnerability percentages for buildings.
        
        This base implementation should be overridden by concrete classes
        to provide specific vulnerability calculation logic.
        """
        # Default implementation - sets all damage percentages to 0
        self._initialize_damage_fields()
    
    def apply_damage_percentages(self):
        """
        Apply damage percentages to buildings based on calculated vulnerability.
        
        This base implementation ensures damage percentage fields exist
        and can be extended by concrete classes.
        """
        self._initialize_damage_fields()
    
    def _initialize_damage_fields(self):
        """
        Initialize damage percentage fields in the buildings data.
        """
        fields = self.buildings.fields
        gdf = self.buildings.gdf
        
        # Initialize building damage percentage
        if fields.get_field_name('building_damage_percent') not in gdf.columns:
            gdf[fields.get_field_name('building_damage_percent')] = 0.0
        
        # Initialize content damage percentage  
        if fields.get_field_name('content_damage_percent') not in gdf.columns:
            gdf[fields.get_field_name('content_damage_percent')] = 0.0
        
        # Initialize inventory damage percentage if inventory cost field exists
        if (fields.get_field_name('inventory_cost') in gdf.columns and 
            fields.get_field_name('inventory_damage_percent') not in gdf.columns):
            gdf[fields.get_field_name('inventory_damage_percent')] = 0.0
