"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
const follow_mode_1 = require("./follow-mode");
/**
 * Determines whether a given file should be excluded or not based on given
 * exclusion glob patterns.
 *
 * @param exclude  exclusion patterns
 * @param filePath file path to be assessed against the pattern
 *
 * @returns `true` if the file should be excluded
 */
function shouldExclude(exclude, filePath) {
    let excludeOutput = false;
    for (const pattern of exclude) {
        const negate = pattern.startsWith('!');
        const match = minimatch(filePath, pattern, { matchBase: true, flipNegate: true });
        if (!negate && match) {
            excludeOutput = true;
        }
        if (negate && match) {
            excludeOutput = false;
        }
    }
    return excludeOutput;
}
exports.shouldExclude = shouldExclude;
/**
 * Determines whether a symlink should be followed or not, based on a FollowMode.
 *
 * @param mode       the FollowMode.
 * @param sourceRoot the root of the source tree.
 * @param realPath   the real path of the target of the symlink.
 *
 * @returns true if the link should be followed.
 */
function shouldFollow(mode, sourceRoot, realPath) {
    switch (mode) {
        case follow_mode_1.FollowMode.ALWAYS:
            return fs.existsSync(realPath);
        case follow_mode_1.FollowMode.EXTERNAL:
            return !_isInternal() && fs.existsSync(realPath);
        case follow_mode_1.FollowMode.BLOCK_EXTERNAL:
            return _isInternal() && fs.existsSync(realPath);
        case follow_mode_1.FollowMode.NEVER:
            return false;
        default:
            throw new Error(`Unsupported FollowMode: ${mode}`);
    }
    function _isInternal() {
        return path.resolve(realPath).startsWith(path.resolve(sourceRoot));
    }
}
exports.shouldFollow = shouldFollow;
//# sourceMappingURL=data:application/json;base64,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