import numpy as np

from AnyQt.QtCore import Qt

from Orange.data import Table
from Orange.widgets import widget, gui
from Orange.widgets.utils import itemmodels
from Orange.widgets.utils.sql import check_sql_input
from Orange.widgets.utils.widgetpreview import WidgetPreview
from Orange.widgets.utils.state_summary import format_summary_details, \
    format_multiple_summaries
from Orange.widgets.widget import Input, Output
from Orange.widgets.utils.annotated_data import (create_annotated_table)
from Orange.i18n_config import *


def __(key):
    return i18n.t("widget.data.data.owselectbydataindex." + key)

class OWSelectByDataIndex(widget.OWWidget):
    name = __("name")
    description = __("desc")
    icon = "icons/SelectByDataIndex.svg"
    priority = 1112

    class Inputs:
        data = Input("Data", Table, label=i18n.t("widget.data.data.common.data"))
        data_subset = Input("Data Subset", Table, label=i18n.t("common.general.data_subset"))

    class Outputs:
        matching_data = Output("Matching Data", Table, replaces=["Data"], default=True, label=i18n.t("widget.data.data.common.match_data"))
        non_matching_data = Output("Unmatched Data", Table, label=i18n.t("widget.data.data.common.unmatched_data"))
        # avoiding the default annotated output name (Data), as it was used
        # for Matching Data previously
        annotated_data = Output("Annotated Data", Table, label=i18n.t("widget.data.data.common.annotated_data"))

    want_main_area = False
    resizing_enabled = False

    class Warning(widget.OWWidget.Warning):
        instances_not_matching = widget.Msg(__("msg_table_no_share_instance"))

    def __init__(self):
        super().__init__()

        self.data = None
        self.data_subset = None

        self.model = itemmodels.VariableListModel()
        self.model_unique_with_id = itemmodels.VariableListModel()
        self.extra_model_unique = itemmodels.VariableListModel()
        self.extra_model_unique_with_id = itemmodels.VariableListModel()

        box = gui.hBox(self.controlArea, box=None)
        self.infoBoxData = gui.label(
            box, self, self.data_info_text(None), box=__("box_data"))
        self.infoBoxExtraData = gui.label(
            box, self, self.data_info_text(None), box=i18n.t("common.general.data_subset"))

        self.info.set_input_summary(self.info.NoInput)
        self.info.set_output_summary(self.info.NoOutput)

    @Inputs.data
    @check_sql_input
    def set_data(self, data):
        self.data = data
        self.infoBoxData.setText(self.data_info_text(data))

    @Inputs.data_subset
    @check_sql_input
    def set_data_subset(self, data):
        self.data_subset = data
        self.infoBoxExtraData.setText(self.data_info_text(data))

    def handleNewSignals(self):
        summary, details, kwargs = self.info.NoInput, "", {}
        if self.data or self.data_subset:
            n_data = len(self.data) if self.data else 0
            n_data_subset = len(self.data_subset) if self.data_subset else 0
            summary = f"{self.info.format_number(n_data)}, " \
                      f"{self.info.format_number(n_data_subset)}"
            kwargs = {"format": Qt.RichText}
            details = format_multiple_summaries([
                ("Data", self.data),
                ("Data subset", self.data_subset)
            ])
        self.info.set_input_summary(summary, details, **kwargs)

        self._invalidate()

    @staticmethod
    def data_info_text(data):
        if data is None:
            return i18n.t("common.software.no_data")
        else:
            return __("info_text.number_instance_variable").format(
                data.name, len(data), len(data.domain) + len(data.domain.metas))

    def commit(self):
        self.Warning.instances_not_matching.clear()
        subset_ids = []
        if self.data_subset:
            subset_ids = self.data_subset.ids
        if not self.data:
            matching_output = None
            non_matching_output = None
            annotated_output = None
        else:
            if self.data_subset and \
                    not np.intersect1d(subset_ids, self.data.ids).size:
                self.Warning.instances_not_matching()
            row_sel = np.in1d(self.data.ids, subset_ids)
            matching_output = self.data[row_sel]
            non_matching_output = self.data[~row_sel]
            annotated_output = create_annotated_table(self.data, row_sel)

        summary = self.info.NoOutput if matching_output is None else len(matching_output)
        details = "" if matching_output is None else format_summary_details(matching_output)
        self.info.set_output_summary(summary, details)
        self.Outputs.matching_data.send(matching_output)
        self.Outputs.non_matching_data.send(non_matching_output)
        self.Outputs.annotated_data.send(annotated_output)

    def _invalidate(self):
        self.commit()

    def send_report(self):
        d_text = self.data_info_text(self.data).replace("\n", ", ")
        ds_text = self.data_info_text(self.data_subset).replace("\n", ", ")
        self.report_items("", [(__("report.data"), d_text), (i18n.t("common.general.data_subset"), ds_text)])


if __name__ == "__main__":  # pragma: no cover
    iris = Table("iris.tab")
    WidgetPreview(OWSelectByDataIndex).run(
        set_data=iris,
        set_data_subset=iris[:20])
