# coding=utf-8
# pystray
# Copyright (C) 2016 Moses Palmér
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the Free
# Software Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import io
import signal

import AppKit
import Foundation
import objc
import PIL
import  PyObjCTools.MachSignals

from . import _base


class Icon(_base.Icon):
    #: The selector for the menu item actions
    _MENU_ITEM_SELECTOR = b'activateMenuItem:sender'

    #: The selector for when the menu is starting to be tracked
    _MENU_NEEDS_UPDATE_SELECTOR = b'menuNeedsUpdate:'

    def __init__(self, *args, **kwargs):
        super(Icon, self).__init__(*args, **kwargs)

        #: The NSImage version of the icon
        self._icon_image = None

    def _show(self):
        self._assert_image()
        self._update_title()

        self._status_item.button().setHidden_(False)

    def _hide(self):
        self._status_item.button().setHidden_(True)

    def _update_icon(self):
        self._icon_image = None

        if self.visible:
            self._assert_image()

    def _update_title(self):
        self._status_item.button().setToolTip_(self.title)

    def _run(self):
        # Make sure there is an NSApplication instance
        self._app = AppKit.NSApplication.sharedApplication()

        # Make sure we have a delegate to handle the acttion events
        self._delegate = IconDelegate.alloc().init()
        self._delegate.icon = self

        self._status_bar = AppKit.NSStatusBar.systemStatusBar()
        self._status_item = self._status_bar.statusItemWithLength_(
            AppKit.NSVariableStatusItemLength)

        self._status_item.button().setTarget_(self._delegate)

        self._nsmenu = AppKit.NSMenu.alloc().initWithTitle_(self.name)
        self._nsmenu.setDelegate_(self._delegate)
        self._status_item.setMenu_(self._nsmenu)

        # Notify the setup callback
        self._mark_ready()

        def sigint(*args):
            self._app.terminate_(None)
            if previous_sigint:
                previous_sigint(*args)

        # Make sure that we do not inhibit ctrl+c
        previous_sigint = PyObjCTools.MachSignals.signal(signal.SIGINT, sigint)

        try:
            self._app.run()
        except:
            self._log.error(
                'An error occurred in the main loop', exc_info=True)
        finally:
            if PyObjCTools.MachSignals.getsignal(signal.SIGINT) == sigint:
                PyObjCTools.MachSignals.signal(signal.SIGINT, previous_sigint)
            self._status_bar.removeStatusItem_(self._status_item)

    def _stop(self):
        self._app.stop_(self._app)

        # Post a dummy event; stop_ will only set a flag in NSApp, so it will
        # not terminate until an event has been processed
        event = getattr(
            AppKit.NSEvent,
            'otherEventWithType_'
            'location_'
            'modifierFlags_'
            'timestamp_'
            'windowNumber_'
            'context_'
            'subtype_'
            'data1_'
            'data2_')(
            AppKit.NSApplicationDefined,
            AppKit.NSPoint(0, 0),
            0,
            0.0,
            0,
            None,
            0,
            0,
            0)
        self._app.postEvent_atStart_(event, False)

    def _assert_image(self):
        """Asserts that the cached icon image exists.
        """
        thickness = self._status_bar.thickness()
        size = (int(thickness), int(thickness))
        if self._icon_image and self._icon_image.size() == size:
            return

        if self._icon.size == size:
            source = self._icon
        else:
            source = PIL.Image.new(
                'RGB',
                size)
            source.paste(self._icon.resize(
                size,
                PIL.Image.ANTIALIAS))

        # Convert the PIL image to an NSImage
        b = io.BytesIO()
        source.save(b, 'png')
        data = Foundation.NSData(b.getvalue())

        self._icon_image = AppKit.NSImage.alloc().initWithData_(data)
        self._status_item.button().setImage_(self._icon_image)

    def _update_menu(self):
        """Updates the popup menu.

        If no visible items are present, the menu will be disabled.

        This method yields all descriptors used.
        """
        # Clear any stale menu items
        self._nsmenu.removeAllItems()

        # Generate the menu
        for descriptor in self.menu:
            self._nsmenu.addItem_(self._create_menu_item(descriptor))
            yield descriptor

    def _create_menu_item(self, descriptor):
        """Creates a :class:`AppKit.NSMenuItem` from a :class:`pystray.MenuItem`
        instance.

        :param descriptor: The menu item descriptor.

        :return: a :class:`AppKit.NSMenuItem`
        """
        if descriptor is _base.Menu.SEPARATOR:
            return AppKit.NSMenuItem.separatorItem()

        else:
            menu_item = AppKit.NSMenuItem.alloc() \
                .initWithTitle_action_keyEquivalent_(
                    descriptor.text, self._MENU_ITEM_SELECTOR, '')
            menu_item.setAction_(self._MENU_ITEM_SELECTOR)
            menu_item.setTarget_(self._delegate)
            if descriptor.default:
                menu_item.setAttributedTitle_(
                    Foundation.NSAttributedString.alloc()
                    .initWithString_attributes_(
                        descriptor.text,
                        Foundation.NSDictionary.alloc()
                        .initWithObjectsAndKeys_(
                            AppKit.NSFont.boldSystemFontOfSize_(
                                AppKit.NSFont.menuFontOfSize_(0)
                                .pointSize()),
                            AppKit.NSFontAttributeName)))
            return menu_item


class IconDelegate(Foundation.NSObject):
    @objc.namedSelector(Icon._MENU_ITEM_SELECTOR)
    def activate_menu_item(self, sender):
        index = self.icon._status_item.menu().indexOfItem_(sender)
        self.descriptors[index](self.icon)

    @objc.namedSelector(Icon._MENU_NEEDS_UPDATE_SELECTOR)
    def menu_needs_update(self, sender):
        self.descriptors = list(self.icon._update_menu())
        if not self.icon.menu:
            self.icon()
