from dataclasses import dataclass
from deep_translator import GoogleTranslator
import threading
import subprocess
import ctypes
import os
import time
import math
import glob
import tempfile
import json
import requests
import pydub
import pydub.playback as pb
import ChatAIStreamer as casr

ChatAIStreamer = casr.ChatAIStreamer
streamParams = casr.streamParams
userMessage = casr.userMessage
aiParams = casr.aiParams

TMPFILE_POSTFIX = "_ZundamonAIStreamer"

# Data type of voice used by ZundamonAIStreamer 
@dataclass
class voiceData:
  content: bytes=None

# Concrete VoiceGanarator of ZundamonAIStreamer
# 'generate' is overridden.
class ZundamonGenerator(casr.voiceGenerator):
  def __init__(self, speaker=1, max_retry=20):
    self.__speaker = speaker
    self.__max_retry = max_retry
    super(ZundamonGenerator, self).__init__()
  @property
  def speaker(self):
    return self.__speaker
  @speaker.setter
  def speaker(self, speaker):
    self.__speaker = speaker
  def __generate(self, text):
    query_payload = {"text": text, "speaker": self.__speaker}
    for i in range(self.__max_retry):
      r = requests.post( "http://localhost:50021/audio_query",
                         params=query_payload, timeout=(10.0, 300.0) )
      if r.status_code == 200:
        query_data = r.json()
        break
      time.sleep(1)

    if r.status_code != 200:
      return r

    synth_payload = {"speaker": self.__speaker}
    for i in range(self.__max_retry):
      r = requests.post( "http://localhost:50021/synthesis", params=synth_payload,
                         data=json.dumps(query_data), timeout=(10.0, 300.0) )
      if r.status_code == 200:
        break
      time.sleep(1)

    return r
  def generate(self, text):
    voice = None
    # Translate into Japanese.
    try:
      text_ja = GoogleTranslator(target='ja').translate(text=text)
    except:
      return text, voice

    r = self.__generate(text_ja)

    if r.status_code == 200:
      # Pack in viceData format.
      voice = voiceData(content=bytes(r.content))

    # return voice with translated text.
    return text_ja, voice

# Extend streamerParams and params to hold voiceGenerator instance.
# voice_generator is defaultly intialized with English voiceGenerator.
@dataclass
class streamerParams(casr.streamerParams):
  voice_generator : casr.voiceGenerator = ZundamonGenerator()
  voicevox_path : str = os.getenv("LOCALAPPDATA") + "/" + "Programs/VOICEVOX/VOICEVOX.exe"
@dataclass
class params(casr.params):
  streamer_params : streamerParams = streamerParams()

class InterruptPlaying(Exception):
    pass

# VoicePlayer class which plays the voice generated by ZundamonGenerator.
class VoicePlayer(threading.Thread):
  def __init__(self, voice, get_volume_cb):
    self.__voice = voice
    self.__get_volume_cb = get_volume_cb
    super(VoicePlayer, self).__init__(daemon=True)

  def run(self):
    if self.__voice:
      self.__play(self.__voice)

  def __play(self, voice):
    with tempfile.NamedTemporaryFile() as file:
      file_path = file.name + TMPFILE_POSTFIX
    with open(file_path, "wb") as file:
      file.write(voice.content)

    segment = pydub.AudioSegment.from_wav(file_path)
    segment = segment[10:]
    self.__interrupted = False
    remain_sec = total_sec = segment.duration_seconds
    playng_first_section = True
    while remain_sec > 0.0:
      volume = self.__get_volume_cb()
      if playng_first_section:
        playng_first_section = False
        prev_volume = volume
        tmp_segment = edit_segment(segment, volume , total_sec - remain_sec)
        play_thread = threading.Thread(target=self.play_interruptible, args=[tmp_segment,], daemon=True)
        play_thread.start()
      elif volume != prev_volume:
        self.__interrupted = True
        play_thread.join()
        self.__interrupted = False
        tmp_segment = edit_segment(segment, volume , total_sec - remain_sec)
        play_thread = threading.Thread(target=self.play_interruptible, args=[tmp_segment,], daemon=True)
        play_thread.start()
        remain_sec = remain_sec - 0.1
      prev_volume = volume
      time.sleep(1)
      remain_sec = remain_sec - 1.0
    play_thread.join()

  def __interrupt(self, length_sec, th):
    if hasattr(th, '_thread_id'):
      th_id = th._thread_id
    else:
      for id, thread in threading._active.items():
        if thread is th:
          th_id = id
    for i in range(math.ceil(length_sec / 0.01)):
      if self.__interrupted:
        ctypes.pythonapi.PyThreadState_SetAsyncExc(th_id, ctypes.py_object(InterruptPlaying))
        break
      time.sleep(0.01)

  def __pb_play(self, segment):
    try:
      pb.play(segment)
    except InterruptPlaying:
      pass

  def play_interruptible(self, segment):
    length_sec = segment.duration_seconds

    th = threading.Thread(target=self.__pb_play, args=[segment,], daemon=True)
    th_interrupt = threading.Thread(target=self.__interrupt, args=[length_sec, th])
    th.start()
    th_interrupt.start()
    th_interrupt.join()
    th.join()

def edit_segment(segment, volume, start_sec):
    delta = pydub.utils.ratio_to_db(volume / 100.)
    segment = segment[start_sec * 1000:] + delta
    return segment  

# ZundamonAIStreamer as inheritence from ChatAIStreamer
class ZundamonAIStreamer(casr.ChatAIStreamer):
  def __init__(self, params):
    # Remove garbage of temporary files.
    with tempfile.NamedTemporaryFile() as file:
      tmp_dir_path = os.path.dirname(file.name)
    for file_path in glob.glob(f"{tmp_dir_path}/*{TMPFILE_POSTFIX}*"):
      os.remove(file_path)

    program_path = params.streamer_params.voicevox_path
    if not program_path or program_path == "" or not os.path.exists(program_path):
      program_path = os.getenv("LOCALAPPDATA") + "/" + "Programs/VOICEVOX/VOICEVOX.exe"
    if os.path.exists(program_path):
      subprocess.Popen(program_path)
    time.sleep(1)

    super(ZundamonAIStreamer, self).__init__(params)
