import { PolicyDocument, PolicyStatement } from '@aws-cdk/aws-iam';
import { Construct, IConstruct } from '@aws-cdk/cdk';
import { EncryptionKeyAlias } from './alias';
export interface IEncryptionKey extends IConstruct {
    /**
     * The ARN of the key.
     */
    readonly keyArn: string;
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias: string): EncryptionKeyAlias;
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement: PolicyStatement, allowNoOp?: boolean): void;
    /**
     * Exports this key from the current stack.
     * @returns a key ref which can be used in a call to `EncryptionKey.import(ref)`.
     */
    export(): EncryptionKeyImportProps;
}
export interface EncryptionKeyImportProps {
    /**
     * The ARN of the external KMS key.
     */
    readonly keyArn: string;
}
export declare abstract class EncryptionKeyBase extends Construct implements IEncryptionKey {
    /**
     * The ARN of the key.
     */
    abstract readonly keyArn: string;
    /**
     * Optional policy document that represents the resource policy of this key.
     *
     * If specified, addToResourcePolicy can be used to edit this policy.
     * Otherwise this method will no-op.
     */
    protected abstract readonly policy?: PolicyDocument;
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias: string): EncryptionKeyAlias;
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement: PolicyStatement, allowNoOp?: boolean): void;
    abstract export(): EncryptionKeyImportProps;
}
/**
 * Construction properties for a KMS Key object
 */
export interface EncryptionKeyProps {
    /**
     * A description of the key. Use a description that helps your users decide
     * whether the key is appropriate for a particular task.
     */
    readonly description?: string;
    /**
     * Indicates whether AWS KMS rotates the key.
     * @default false
     */
    readonly enableKeyRotation?: boolean;
    /**
     * Indicates whether the key is available for use.
     * @default Key is enabled
     */
    readonly enabled?: boolean;
    /**
     * Custom policy document to attach to the KMS key.
     *
     * @default A policy document with permissions for the account root to
     * administer the key will be created.
     */
    readonly policy?: PolicyDocument;
    /**
     * Whether the encryption key should be retained when it is removed from the Stack. This is useful when one wants to
     * retain access to data that was encrypted with a key that is being retired.
     *
     * @default true
     */
    readonly retain?: boolean;
}
/**
 * Defines a KMS key.
 */
export declare class EncryptionKey extends EncryptionKeyBase {
    /**
     * Defines an imported encryption key.
     *
     * `ref` can be obtained either via a call to `key.export()` or using
     * literals.
     *
     * For example:
     *
     *   const keyAttr = key.export();
     *   const keyRef1 = EncryptionKey.import(this, 'MyImportedKey1', keyAttr);
     *   const keyRef2 = EncryptionKey.import(this, 'MyImportedKey2', {
     *     keyArn: new KeyArn('arn:aws:kms:...')
     *   });
     *
     * @param scope The parent construct.
     * @param id The name of the construct.
     * @param props The key reference.
     */
    static import(scope: Construct, id: string, props: EncryptionKeyImportProps): IEncryptionKey;
    readonly keyArn: string;
    protected readonly policy?: PolicyDocument;
    constructor(scope: Construct, id: string, props?: EncryptionKeyProps);
    /**
     * Exports this key from the current stack.
     * @returns a key ref which can be used in a call to `EncryptionKey.import(ref)`.
     */
    export(): EncryptionKeyImportProps;
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    private allowAccountToAdmin;
}
