"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const configmap = require("fs");
const path = require("path");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const minimatch_1 = require("minimatch");
const base = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
class ImportedConfigMap extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeConfigMap.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubeConfigMap.GVK.kind;
    }
}
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'configmaps';
        this._binaryData = {};
        this._data = {};
        this.immutable = props.immutable ?? false;
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s_1.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s_1.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
            immutable: this.immutable,
        });
        for (const [k, v] of Object.entries(props.data ?? {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries(props.binaryData ?? {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     */
    static fromConfigMapName(scope, id, name) {
        return new ImportedConfigMap(scope, id, name);
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key ?? path.basename(localFile);
        const value = configmap.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        const exclude = options.exclude ?? [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                const mm = new minimatch_1.Minimatch(pattern);
                if (mm.match(file)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = options.keyPrefix ?? '';
        for (const file of configmap.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (configmap.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-23.ConfigMap", version: "2.0.30" };
//# sourceMappingURL=data:application/json;base64,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