"""Device tracker for MyBMW vehicles."""

from __future__ import annotations

import logging
from typing import Any

from bimmer_connected.vehicle import MyBMWVehicle

from homeassistant.components.device_tracker import TrackerEntity
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddConfigEntryEntitiesCallback

from . import BMWConfigEntry
from .const import ATTR_DIRECTION
from .coordinator import BMWDataUpdateCoordinator
from .entity import BMWBaseEntity

PARALLEL_UPDATES = 0

_LOGGER = logging.getLogger(__name__)


async def async_setup_entry(
    hass: HomeAssistant,
    config_entry: BMWConfigEntry,
    async_add_entities: AddConfigEntryEntitiesCallback,
) -> None:
    """Set up the MyBMW tracker from config entry."""
    coordinator = config_entry.runtime_data
    entities: list[BMWDeviceTracker] = []

    for vehicle in coordinator.account.vehicles:
        entities.append(BMWDeviceTracker(coordinator, vehicle))
        if not vehicle.is_vehicle_tracking_enabled:
            _LOGGER.info(
                (
                    "Tracking is (currently) disabled for vehicle %s (%s), defaulting"
                    " to unknown"
                ),
                vehicle.name,
                vehicle.vin,
            )
    async_add_entities(entities)


class BMWDeviceTracker(BMWBaseEntity, TrackerEntity):
    """MyBMW device tracker."""

    _attr_force_update = False
    _attr_translation_key = "car"
    _attr_name = None

    def __init__(
        self,
        coordinator: BMWDataUpdateCoordinator,
        vehicle: MyBMWVehicle,
    ) -> None:
        """Initialize the Tracker."""
        super().__init__(coordinator, vehicle)
        self._attr_unique_id = vehicle.vin

    @property
    def extra_state_attributes(self) -> dict[str, Any]:
        """Return entity specific state attributes."""
        return {ATTR_DIRECTION: self.vehicle.vehicle_location.heading}

    @property
    def latitude(self) -> float | None:
        """Return latitude value of the device."""
        return (
            self.vehicle.vehicle_location.location[0]
            if self.vehicle.is_vehicle_tracking_enabled
            and self.vehicle.vehicle_location.location
            else None
        )

    @property
    def longitude(self) -> float | None:
        """Return longitude value of the device."""
        return (
            self.vehicle.vehicle_location.location[1]
            if self.vehicle.is_vehicle_tracking_enabled
            and self.vehicle.vehicle_location.location
            else None
        )
