# -*- coding: utf-8 -*-
# © 2012 Nicolas Bessi (Camptocamp SA)
# © 2015 Yannick Vaucher (Camptocamp SA)
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html).
from openerp import models, api, _
from openerp.tools import mod10r
from openerp import exceptions


class AccountInvoice(models.Model):

    _inherit = "account.invoice"

    @api.model
    def _get_reference_type(self):
        selection = super(AccountInvoice, self)._get_reference_type()
        selection.append(('bvr', _('BVR Reference')))
        return selection

    @api.onchange('reference')
    def onchange_reference(self):
        """Identify if the reference entered is of type BVR
        if it does, change reference_type"""
        if len(self.reference or '') == 27:
            try:
                self._is_bvr_reference()
            except exceptions.ValidationError:
                return
            self.reference_type = 'bvr'

    @api.constrains('reference_type')
    def _check_bank_type_for_type_bvr(self):
        for invoice in self:
            if invoice.reference_type == 'bvr':
                bank_acc = invoice.partner_bank_id
                if not (bank_acc.acc_type == 'postal' or
                        bank_acc.acc_type != 'postal' and
                        bank_acc.bank_id.ccp):
                    if invoice.type in ('in_invoice', 'in_refund'):
                        raise exceptions.ValidationError(
                            _('BVR/ESR Reference type needs a postal account'
                              ' number on the customer.')
                        )
                    else:
                        raise exceptions.ValidationError(
                            _('BVR/ESR Reference type needs a postal account'
                              ' number on your company')
                        )
        return True

    @api.multi
    def _is_bvr_reference(self):
        """
        Function to validate a bvr reference like :
        0100054150009>132000000000000000000000014+ 1300132412>
        The validation is based on l10n_ch
        """
        if not self.reference:
            raise exceptions.ValidationError(
                _('BVR/ESR Reference is required')
            )
        # In this case
        # <010001000060190> 052550152684006+ 43435>
        # the reference 052550152684006 do not match modulo 10
        #
        if (mod10r(self.reference[:-1]) != self.reference and
                len(self.reference) == 15):
            return True
        #
        if mod10r(self.reference[:-1]) != self.reference:
            raise exceptions.ValidationError(
                _('Invalid BVR/ESR Number (wrong checksum).')
            )

    @api.constrains('reference')
    def _check_bvr(self):
        """ Do the check only for invoice with reference_type = 'bvr' """
        for invoice in self:
            if invoice.reference_type == 'bvr':
                invoice._is_bvr_reference()
        return True

    @api.model
    def create(self, vals):
        """We override create in order to have customer invoices
        generated by the comercial flow as on change partner is
        not systemtically call"""
        type_defined = vals.get('type') or self.env.context.get('type', False)
        if type_defined == 'out_invoice' and not vals.get('partner_bank_id'):
            user = self.env.user
            bank_ids = user.company_id.partner_id.bank_ids
            if bank_ids:
                vals['partner_bank_id'] = bank_ids[0].id
        return super(AccountInvoice, self).create(vals)
