"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const route53 = require("@aws-cdk/aws-route53");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const targets = require("../lib");
test('use S3 bucket website as record target', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    const bucketWebsite = new s3.Bucket(stack, 'Bucket');
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName: 'test.public' });
    new route53.ARecord(zone, 'Alias', {
        zone,
        recordName: '_foo',
        target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
    });
    // THEN
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        AliasTarget: {
            DNSName: { "Fn::GetAtt": ["Bucket83908E77", "WebsiteURL"] },
            HostedZoneId: "Z3AQBSTGFYJSTF"
        },
    });
});
test('throws if region agnostic', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const bucketWebsite = new s3.Bucket(stack, 'Bucket');
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName: 'test.public' });
    // THEN
    expect(() => {
        new route53.ARecord(zone, 'Alias', {
            zone,
            recordName: '_foo',
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
        });
    }).toThrow(/Cannot use an S3 record alias in region-agnostic stacks/);
});
test('throws if bucket website hosting is unavailable (cn-northwest-1)', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'cn-northwest-1' } });
    const bucketWebsite = new s3.Bucket(stack, 'Bucket');
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName: 'test.public' });
    // THEN
    expect(() => {
        new route53.ARecord(zone, 'Alias', {
            zone,
            recordName: '_foo',
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
        });
    }).toThrow(/Bucket website target is not supported/);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVja2V0LXdlYnNpdGUtdGFyZ2V0LnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidWNrZXQtd2Vic2l0ZS10YXJnZXQudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdDQUE4QjtBQUM5QixnREFBaUQ7QUFDakQsc0NBQXVDO0FBQ3ZDLHdDQUEyQztBQUMzQyxrQ0FBbUM7QUFFbkMsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtJQUNsRCxRQUFRO0lBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsTUFBTSxFQUFFLEVBQUMsR0FBRyxFQUFFLEVBQUMsTUFBTSxFQUFFLFdBQVcsRUFBQyxFQUFDLENBQUMsQ0FBQztJQUVuRSxNQUFNLGFBQWEsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBRXJELE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7SUFDNUYsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7UUFDakMsSUFBSTtRQUNKLFVBQVUsRUFBRSxNQUFNO1FBQ2xCLE1BQU0sRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxhQUFhLENBQUMsQ0FBQztLQUN2RixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyx5QkFBeUIsRUFBRTtRQUN0RCxXQUFXLEVBQUU7WUFDWCxPQUFPLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxnQkFBZ0IsRUFBRSxZQUFZLENBQUMsRUFBRTtZQUM1RCxZQUFZLEVBQUUsZ0JBQWdCO1NBQy9CO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO0lBQ3JDLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sYUFBYSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFFckQsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksT0FBTyxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxRQUFRLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQztJQUU1RixPQUFPO0lBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtRQUNWLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO1lBQ2pDLElBQUk7WUFDSixVQUFVLEVBQUUsTUFBTTtZQUNsQixNQUFNLEVBQUUsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsbUJBQW1CLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDdkYsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLHlEQUF5RCxDQUFDLENBQUM7QUFDeEUsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsa0VBQWtFLEVBQUUsR0FBRyxFQUFFO0lBQzVFLFFBQVE7SUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0lBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxNQUFNLEVBQUUsRUFBQyxHQUFHLEVBQUUsRUFBQyxNQUFNLEVBQUUsZ0JBQWdCLEVBQUMsRUFBQyxDQUFDLENBQUM7SUFFeEUsTUFBTSxhQUFhLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUVyRCxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxFQUFFLFFBQVEsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO0lBRTVGLE9BQU87SUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO1FBQ1YsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDakMsSUFBSTtZQUNKLFVBQVUsRUFBRSxNQUFNO1lBQ2xCLE1BQU0sRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUN2RixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsd0NBQXdDLENBQUMsQ0FBQztBQUN2RCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHJvdXRlNTMgPSByZXF1aXJlKCdAYXdzLWNkay9hd3Mtcm91dGU1MycpO1xuaW1wb3J0IHMzID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLXMzJyk7XG5pbXBvcnQgeyBBcHAsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgdGFyZ2V0cyA9IHJlcXVpcmUoJy4uL2xpYicpO1xuXG50ZXN0KCd1c2UgUzMgYnVja2V0IHdlYnNpdGUgYXMgcmVjb3JkIHRhcmdldCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICd0ZXN0Jywge2Vudjoge3JlZ2lvbjogJ3VzLWVhc3QtMSd9fSk7XG5cbiAgY29uc3QgYnVja2V0V2Vic2l0ZSA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdCdWNrZXQnKTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHpvbmUgPSBuZXcgcm91dGU1My5QdWJsaWNIb3N0ZWRab25lKHN0YWNrLCAnSG9zdGVkWm9uZScsIHsgem9uZU5hbWU6ICd0ZXN0LnB1YmxpYycgfSk7XG4gIG5ldyByb3V0ZTUzLkFSZWNvcmQoem9uZSwgJ0FsaWFzJywge1xuICAgIHpvbmUsXG4gICAgcmVjb3JkTmFtZTogJ19mb28nLFxuICAgIHRhcmdldDogcm91dGU1My5SZWNvcmRUYXJnZXQuZnJvbUFsaWFzKG5ldyB0YXJnZXRzLkJ1Y2tldFdlYnNpdGVUYXJnZXQoYnVja2V0V2Vic2l0ZSkpXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpSb3V0ZTUzOjpSZWNvcmRTZXQnLCB7XG4gICAgQWxpYXNUYXJnZXQ6IHtcbiAgICAgIEROU05hbWU6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJCdWNrZXQ4MzkwOEU3N1wiLCBcIldlYnNpdGVVUkxcIl0gfSxcbiAgICAgIEhvc3RlZFpvbmVJZDogXCJaM0FRQlNUR0ZZSlNURlwiXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgndGhyb3dzIGlmIHJlZ2lvbiBhZ25vc3RpYycsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBidWNrZXRXZWJzaXRlID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0J1Y2tldCcpO1xuXG4gIC8vIFdIRU5cbiAgY29uc3Qgem9uZSA9IG5ldyByb3V0ZTUzLlB1YmxpY0hvc3RlZFpvbmUoc3RhY2ssICdIb3N0ZWRab25lJywgeyB6b25lTmFtZTogJ3Rlc3QucHVibGljJyB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdCgoKSA9PiB7XG4gICAgbmV3IHJvdXRlNTMuQVJlY29yZCh6b25lLCAnQWxpYXMnLCB7XG4gICAgICB6b25lLFxuICAgICAgcmVjb3JkTmFtZTogJ19mb28nLFxuICAgICAgdGFyZ2V0OiByb3V0ZTUzLlJlY29yZFRhcmdldC5mcm9tQWxpYXMobmV3IHRhcmdldHMuQnVja2V0V2Vic2l0ZVRhcmdldChidWNrZXRXZWJzaXRlKSlcbiAgICB9KTtcbiAgfSkudG9UaHJvdygvQ2Fubm90IHVzZSBhbiBTMyByZWNvcmQgYWxpYXMgaW4gcmVnaW9uLWFnbm9zdGljIHN0YWNrcy8pO1xufSk7XG5cbnRlc3QoJ3Rocm93cyBpZiBidWNrZXQgd2Vic2l0ZSBob3N0aW5nIGlzIHVuYXZhaWxhYmxlIChjbi1ub3J0aHdlc3QtMSknLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAndGVzdCcsIHtlbnY6IHtyZWdpb246ICdjbi1ub3J0aHdlc3QtMSd9fSk7XG5cbiAgY29uc3QgYnVja2V0V2Vic2l0ZSA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdCdWNrZXQnKTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHpvbmUgPSBuZXcgcm91dGU1My5QdWJsaWNIb3N0ZWRab25lKHN0YWNrLCAnSG9zdGVkWm9uZScsIHsgem9uZU5hbWU6ICd0ZXN0LnB1YmxpYycgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QoKCkgPT4ge1xuICAgIG5ldyByb3V0ZTUzLkFSZWNvcmQoem9uZSwgJ0FsaWFzJywge1xuICAgICAgem9uZSxcbiAgICAgIHJlY29yZE5hbWU6ICdfZm9vJyxcbiAgICAgIHRhcmdldDogcm91dGU1My5SZWNvcmRUYXJnZXQuZnJvbUFsaWFzKG5ldyB0YXJnZXRzLkJ1Y2tldFdlYnNpdGVUYXJnZXQoYnVja2V0V2Vic2l0ZSkpXG4gICAgfSk7XG4gIH0pLnRvVGhyb3coL0J1Y2tldCB3ZWJzaXRlIHRhcmdldCBpcyBub3Qgc3VwcG9ydGVkLyk7XG59KTtcbiJdfQ==