import hashlib
import json
import logging
import sys

import six

log = logging.getLogger(__name__)

SENSITIVE_HEADERS = ['X-Token']


def parse_headers(headers):
    result = {}
    for header in headers:
        if ':' in header:
            header = header.replace(' ', '')
            parts = header.split(':')
            result[parts[0]] = parts[1]
    return result


def handle_http_error(func):
    def wrap(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as err:
            log.error(err)
            sys.exit(2)

    return wrap


def resource_filter(func):
    """This decorator allows to you filter answer from RESOURCE.list() by
    project_id and region. 
    Both params are optional and may be used separately.
    
    Example:
        selvpc --debug floatingip list
        selvpc --debug floatingip list --project=UUID
        selvpc --debug floatingip list --region=REGION
        selvpc --debug floatingip list --project=UUID --region=REGION
        
        client.subnets.list(project=UUID)
        client.subnets.list(region=REGION)
        client.subnets.list(project=UUID, region=REGION)
    """

    def wrap(*args, **kwargs):
        project_id = kwargs.pop("project_id", None)
        region = kwargs.pop("region", None)
        resources = func(*args, **kwargs)

        if project_id:
            resources = [r for r in resources if r["project_id"] == project_id]

        if region:
            resources = [r for r in resources if r["region"] == region]

        return resources

    return wrap


def add_resource_filter_arguments(parser, add_region=True):
    parser.add_argument(
        '-p',
        '--project',
        dest="project_id",
        required=False,
        default=None,
        type=str,
    )
    if add_region:
        parser.add_argument(
            '-r',
            '--region',
            required=False,
            default=None,
            type=str,
        )


def confirm_action(action):
    """Func must be a take_action func."""

    def wrap(func):
        def wrap(*args, **kwargs):
            if not hasattr(args[1], "yes_i_really_want_to_" + action):
                log.error("Please add confirm argument into parser.")
                sys.exit(-1)

            accept = getattr(args[1], "yes_i_really_want_to_" + action)
            if not accept:
                log.warning("Confirm action by --yes-i-really-want-to-" +
                            action)
                sys.exit(-1)

            return func(*args, **kwargs)

        return wrap

    return wrap


def get_item_properties(item, fields, mixed_case_fields=(), formatters=None):
    """Return a tuple containing the item properties.

    :param item: a single item resource (e.g. Server, Tenant, etc)
    :param fields: tuple of strings with the desired field names
    :param mixed_case_fields: tuple of field names to preserve case
    :param formatters: dictionary mapping field names to callables
       to format the values
    """
    if formatters is None:
        formatters = {}

    row = []

    for field in fields:
        if field in formatters:
            row.append(formatters[field](item))
        else:
            if field in mixed_case_fields:
                field_name = field.replace(' ', '_')
            else:
                field_name = field.lower().replace(' ', '_')
            if not hasattr(item, field_name) and isinstance(item, dict):
                data = item[field_name]
            else:
                data = getattr(item, field_name, '')
            if data is None:
                data = ''
            row.append(data)
    return tuple(row)


def sort_list_of_dicts(list_, dict_key):
    """Sort list of dicts by dict key

    :param list list_: List of dicts,
    :param string dict_key: Dict key for sorting.
    :rtype: list
    """

    # NOTE: Python 3 introduced new rules for ordering comparisons:
    # See detailed here (chapter ordering-comparisons)
    # https://docs.python.org/release/3.0.1/whatsnew/3.0.html
    items = []
    for item in list_:
        if item[dict_key] is None:
            item[dict_key] = str()
        items.append(item)

    return sorted(items, key=lambda item: item[dict_key])


def build_url(*args):
    """Build URL by provided parts of url.
    Also this method strip all right slashes.

    :param args: Parts of url.
    :rtype: str
    """
    return "/".join([part.rstrip('/') for part in args])


def update_json_error_message(content):
    """Converts and capitalize JSON error to normal message.

    :param str content: JSON-answer from server.
    :rtype: str
    """
    if 'error' in content:
        try:
            message = json.loads(content)['error']
            return message.capitalize().replace('_', ' ')
        except:
            return content


def try_parse_json(json_):
    """Converts the string representation of JSON to JSON.

    :param str json_: JSON in str representation.
    :rtype: :class:`dict` if converted successfully, otherwise False.
    """
    if not json_:
        return False

    try:
        return json.loads(json_)
    except ValueError as e:
        return False


def make_curl(url, method, data):
    string_parts = ['curl -i', ' -X{} "{}"'.format(method, url)]

    for (key, value) in six.iteritems(data.get('headers', {})):
        if key in SENSITIVE_HEADERS:
            v = str()
            if value:
                v = value.encode('utf-8')
            h = hashlib.sha1(v)
            d = h.hexdigest()
            value = "{SHA1}%s" % d
        header = ' -H "%s: %s"' % (key, value)
        string_parts.append(header)

    if data.get('json', None):
        string_parts.append(" -d '%s'" % (json.dumps(data['json'])))
    return "".join(string_parts)


def process_partial_quotas(resp_ok):
    result = {"quotas": {}}
    for item in resp_ok:
        if item["resource"] not in result["quotas"]:
            result["quotas"][item["resource"]] = [{
                k: item[k] for k in item if k != "resource"
            }]
        else:
            result["quotas"][item["resource"]].append({
                k: item[k] for k in item if k != "resource"
            })
    return result
