from typing import Callable, TypeVar, Dict, List, Any

T = TypeVar('T')
K = TypeVar('K')
V = TypeVar('V')
R = TypeVar('R')


def keys_len(operators: list[dict]):
    return sum([len(operator["keys"]) for operator in operators])


def merge_keys(lists: list[list[dict]]):
    first_operators = lists[0]
    merged_operators = []

    for index, operator in enumerate(first_operators):
        merged_operator_keys = [key for operators in lists for key in operators[index]["keys"]]

        merged_operators.append({
            **operator,
            "keys": merged_operator_keys
        })

    return merged_operators


def filter_keys(operators: list[dict], callback):
    return [{**operator, "keys": [key for key in operator["keys"] if callback(key)]} for operator in operators]


def pick_by(obj: Dict[K, T], predicate: Callable[[T, K], bool]) -> Dict[K, T]:
    """
    Creates an dict composed of the dict properties predicate returns truthy for.
    The predicate is invoked with two arguments: ``(value, key)``.
    :param obj: (dict): Dictionary to pick from.
    :param predicate: (lambda(value, key)): Lambda used to determine which properties to pick.
    Returns:
        dict: Dictionary containing only picked properties.
    Example:
        >>> some_dict = {'a': 1, 'b': '2', 'c': 3 }
        >>> assert pick_by(some_dict, lambda v: isinstance(v, int)) == {'a': 1, 'c': 3}
    """

    result = {}

    for key, value in obj.items():
        if predicate(value, key):
            result[key] = value

    return result


def group_by(collection: List[T], iteratee: Callable[[T], K]) -> Dict[K, List[T]]:
    """
    Creates an object composed of keys generated from the results of running each element of collection thru iteratee.
    The order of grouped values is determined by the order they occur in collection.
    The corresponding value of each key is an array of elements responsible for generating the key.
    The iteratee is invoked with one argument: (value).

    :param collection: (list): The collection to iterate over.
    :param iteratee: (lambda(value)): The iteratee returning keys for aggregate object.
    Returns:
        dict: Returns the composed aggregate object.
    Example:
        >>> ret = group_by([6.1, 4.2, 6.3], lambda value: math.floor(value))
        >>> assert ret == {'4': [4.2], '6': [6.1, 6.3]}
    """
    result = {}
    for item in collection:
        key = iteratee(item)
        if key not in result:
            result[key] = []
        result[key].append(key)

    return result


def map_values(obj: Dict[K, T], iteratee: Callable[[T, K, Dict[K, T]], R]) -> Dict[K, R]:
    """
    Creates an object with the same keys as object and values generated by running each own enumerable
    string keyed property of object thru iteratee. The iteratee is invoked with three arguments:
    (value, key, object).
    :param obj: (dict): The object to iterate over.
    :param iteratee: (lambda(value, key, obj)): The function invoked per iteration.
    Returns:
       dict: Returns the new mapped object.
    Example:
       >>> ret = map_values({'a': 2, 'b': 1, 'c': 3}, lambda value, key, obj: value * 2)
       >>> assert ret == {'a': 4, 'b': 2, 'c': 6}
    """

    return {key: iteratee(value, key, obj) for key, value in obj.items()}


def count_by(collection: List[T], iteratee: Callable[[T], K]) -> Dict[K, int]:
    """
    Creates an object composed of keys generated from the results of running each element of collection thru iteratee.
    The corresponding value of each key is the number of times the key was returned by iteratee.
    The iteratee is invoked with one argument: (value).
    :param collection: (list): The collection to iterate over.
    :param iteratee: (lambda(value) -> str): The iteratee returning keys for aggregate object
    Returns:
        dict: Returns the composed aggregate dictionary.
    Example:
        >>> ret = count_by(['a', 'b', 'c', 'a'], lambda v: v)
        >>> assert ret == {'a': 2, 'b': 1, 'c': 1}
    """
    aggregated = group_by(collection, iteratee)

    return map_values(aggregated, lambda value, key, obj: len(value))
