import datetime
import io
import json
import unittest
import unittest.mock
from pathlib import Path

import botocore.exceptions as be
import botocore.response as br
from parameterized import parameterized

from insightconnect_plugin_runtime import Input, Output, Connection
from insightconnect_plugin_runtime.clients.aws_client import AWSAction, ActionHelper
from insightconnect_plugin_runtime.exceptions import PluginException


def raise_attribiute_error(arg1, arg2):
    raise AttributeError()


class Boto3Stub:
    def assume_role(self, *args, **kwargs):
        return {
            "Credentials": {
                "AccessKeyId": "123",
                "SecretAccessKey": "456",
                "SessionToken": "token",
            }
        }


class TestAwsAction(unittest.TestCase):
    def setUp(self) -> None:
        self.auth_params = {
            "aws_access_key_id": "123",
            "aws_secret_access_key": "321",
        }
        self.region = "us-east"

        self.aws_action = AWSAction(
            "NewAction", "Description", None, None, "ec2", "service"
        )
        self.aws_action.connection = unittest.mock.create_autospec(Connection)
        self.aws_action.connection.helper = unittest.mock.create_autospec(ActionHelper)
        self.aws_action.input = Input({})
        self.aws_action.output = Output({})

    @unittest.mock.patch("botocore.session.Session", return_value=unittest.mock.Mock())
    @unittest.mock.patch("boto3.client", return_value=Boto3Stub())
    def test_assume_role(self, mock_session, mock_sts_client):
        assume_role_params = {
            "role_arn": "test_role",
            "external_id": "test_id",
            "region": "test-region",
        }
        aws_session = unittest.mock.Mock()

        AWSAction.try_to_assume_role("ec2", assume_role_params, self.auth_params)

        mock_sts_client.assert_called_once()

    @unittest.mock.patch("botocore.session.Session", return_value=unittest.mock.Mock())
    @unittest.mock.patch("boto3.client", return_value=Boto3Stub())
    def test_assume_role_without_role_arn(self, mock_session, mock_sts_client):
        assume_role_params = {"role_arn": "", "external_id": "", "region": ""}
        aws_session = unittest.mock.Mock()

        if assume_role_params.get("role_arn"):
            AWSAction.try_to_assume_role("ec2", assume_role_params, self.auth_params)

        mock_sts_client.assert_not_called()

    @unittest.mock.patch("botocore.session.Session", return_value=unittest.mock.Mock())
    @unittest.mock.patch("boto3.client", return_value=Boto3Stub())
    def test_run_action(self, mock_session, mock_sts_client):
        client_function = getattr(mock_session, "service")
        self.aws_action.handle_rest_call(client_function, {})

        client_function.assert_called_once()
        self.aws_action.connection.helper.format_input.assert_called_once()
        self.aws_action.connection.helper.format_output.assert_called_once()

    def _mock_call_raise_endpoint_connection_error(self):
        raise be.EndpointConnectionError(**{"endpoint_url": "test_url"})

    def test_handle_rest_call_endpoint_connection_error(self):
        with self.assertRaises(PluginException):
            mock_call = self._mock_call_raise_endpoint_connection_error
            self.aws_action.handle_rest_call(mock_call, {})

    def _mock_call_raise_param_validation_error(self):
        raise be.ParamValidationError(**{"endpoint_url": "test_url"})

    def test_handle_rest_call_param_validation_error(self):
        with self.assertRaises(PluginException):
            mock_call = self._mock_call_raise_param_validation_error
            self.aws_action.handle_rest_call(mock_call, {})

    def _mock_call_raise_client_error(self):
        raise be.ClientError(**{"endpoint_url": "test_url"})

    def test_handle_rest_call_client_error(self):
        mock_call = self._mock_call_raise_client_error
        with self.assertRaises(PluginException):
            self.aws_action.handle_rest_call(mock_call, {})

    def mocked_requests_get(*args, **kwargs):
        class MockResponse:
            def __init__(self, json_data, status_code):
                self.json_data = json_data
                self.status_code = status_code
                self.ok = True

        return MockResponse(None, 200)

    @unittest.mock.patch("requests.get", side_effect=mocked_requests_get)
    @unittest.mock.patch("botocore.session.Session", return_value=unittest.mock.Mock())
    @unittest.mock.patch("boto3.client", return_value=Boto3Stub())
    def test_test(self, mock_get, mock_session, mock_sts_client):
        self.aws_action.connection.client = mock_session
        self.aws_action.test()
        self.aws_action.connection.helper.format_output.assert_called_once()


class Test(unittest.TestCase):
    def setUp(self) -> None:
        return super().setUp()

    @parameterized.expand(
        [
            ("snake_str_1", "SnakeStr1"),
            ("sssnake31_#test", "Sssnake31#Test"),
            ("s__o", "SO"),
            ("_private_variable", "PrivateVariable"),
        ],
    )
    def test_to_upper_camel_case(self, input_str, output_str):
        camel_case_str = ActionHelper.to_upper_camel_case(input_str)
        self.assertEqual(camel_case_str, output_str)

    def test_get_empty_input(self):
        formatted_params = ActionHelper.format_input(
            {"$param1": {}, "$param2": {}, "$param3": [], "$param4": "test"}
        )
        self.assertEqual(formatted_params, {"Param4": "test"})

    def test_get_empty_output(self):
        path = Path(__file__).parent / f"payloads/output_schema.json"
        with open(path) as file:
            output_schema = json.load(file)
        empty_output = ActionHelper.get_empty_output(output_schema)
        self.assertEqual(
            empty_output,
            {
                "reservations": [],
                "response_metadata": {"http_status_code": 0, "request_id": ""},
            },
        )

    @parameterized.expand(
        [
            (datetime.datetime(2022, 9, 4), "2022-09-04T00:00:00"),
            (2.467, "2.467"),
            (b"a", "YQ=="),
            (br.StreamingBody(io.BytesIO(b"\x01\x02\x03\x04"), 4), "AQIDBA=="),
            ("test string", "test string"),
            ([3.14, 2.71], ["3.14", "2.71"]),
            ([{"key1": 3.14}, {"key2": 2.71}], [{"key1": "3.14"}, {"key2": "2.71"}]),
        ]
    )
    def test_fix_output_types(self, input_type, output_type):
        ah = ActionHelper()
        date = ah.fix_output_types(input_type)
        self.assertEqual(output_type, date)

    @parameterized.expand(
        [
            ({"UpperCamel": "OutputValue"}, {"upper_camel": "OutputValue"}),
            ([{"UpperCamel": "OutputValue"}], [{"upper_camel": "OutputValue"}]),
        ]
    )
    def test_convert_all_to_snake_case(self, test_input, test_output):
        converted = ActionHelper.convert_all_to_snake_case(test_input)
        self.assertEqual(converted, test_output)

    def test_format_output(self):
        ah = ActionHelper()
        path = Path(__file__).parent / f"payloads/output_schema.json"
        with open(path) as file:
            output_schema = json.load(file)
        test_input = {"Reservations": [{"key1": 1}]}
        output = ah.format_output(output_schema, test_input)
        correct_output = {
            "reservations": [{"key1": 1}],
            "response_metadata": {"http_status_code": 0, "request_id": ""},
        }
        self.assertEqual(output, correct_output)

    @parameterized.expand(
        [
            ({"snake_case": "OutputValue"}, {"SnakeCase": "OutputValue"}),
            ([{"snake_case": "OutputValue"}], [{"SnakeCase": "OutputValue"}]),
        ]
    )
    def test_convert_all_to_upper_camel_case(self, test_input, test_output):
        converted = ActionHelper.convert_all_to_upper_camel_case(test_input)
        self.assertEqual(test_output, converted)
