"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamMode = exports.StreamEncryption = exports.Stream = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const kinesis_fixed_canned_metrics_1 = require("./kinesis-fixed-canned-metrics");
const kinesis_generated_1 = require("./kinesis.generated");
const READ_OPERATIONS = [
    'kinesis:DescribeStreamSummary',
    'kinesis:GetRecords',
    'kinesis:GetShardIterator',
    'kinesis:ListShards',
    'kinesis:SubscribeToShard',
    'kinesis:DescribeStream',
    'kinesis:ListStreams',
];
const WRITE_OPERATIONS = [
    'kinesis:ListShards',
    'kinesis:PutRecord',
    'kinesis:PutRecords',
];
/**
 * Represents a Kinesis Stream.
 */
class StreamBase extends core_1.Resource {
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, ...READ_OPERATIONS);
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to encrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee) {
        var _b;
        const ret = this.grant(grantee, ...WRITE_OPERATIONS);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grantEncrypt(grantee);
        return ret;
    }
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee) {
        var _b;
        const ret = this.grant(grantee, ...Array.from(new Set([...READ_OPERATIONS, ...WRITE_OPERATIONS])));
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grantEncryptDecrypt(grantee);
        return ret;
    }
    /**
     * Grant the indicated permissions on this stream to the given IAM principal (Role/Group/User).
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
    /**
     * Return stream metric based from its metric name
     *
     * @param metricName name of the stream metric
     * @param props properties of the metric
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName,
            dimensionsMap: {
                StreamName: this.streamName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of bytes retrieved from the Kinesis stream, measured over the specified time period. Minimum, Maximum,
     * and Average statistics represent the bytes in a single GetRecords operation for the stream in the specified time
     * period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsBytesAverage, props);
    }
    /**
     * The age of the last record in all GetRecords calls made against a Kinesis stream, measured over the specified time
     * period. Age is the difference between the current time and when the last record of the GetRecords call was written
     * to the stream. The Minimum and Maximum statistics can be used to track the progress of Kinesis consumer
     * applications. A value of zero indicates that the records being read are completely caught up with the stream.
     *
     * The metric defaults to maximum over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsIteratorAgeMilliseconds(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsIteratorAgeMillisecondsMaximum, props);
    }
    /**
     * The number of successful GetRecords operations per stream, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsSuccess(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsSuccessAverage, props);
    }
    /**
     * The number of records retrieved from the shard, measured over the specified time period. Minimum, Maximum, and
     * Average statistics represent the records in a single GetRecords operation for the stream in the specified time
     * period.
     *
     * average
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsRecordsAverage, props);
    }
    /**
     * The number of successful GetRecords operations per stream, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsLatency(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsLatencyAverage, props);
    }
    /**
     * The number of bytes put to the Kinesis stream using the PutRecord operation over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordBytesAverage, props);
    }
    /**
     * The time taken per PutRecord operation, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordLatency(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordLatencyAverage, props);
    }
    /**
     * The number of successful PutRecord operations per Kinesis stream, measured over the specified time period. Average
     * reflects the percentage of successful writes to a stream.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordSuccess(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordSuccessAverage, props);
    }
    /**
     * The number of bytes put to the Kinesis stream using the PutRecords operation over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsBytesAverage, props);
    }
    /**
     * The time taken per PutRecords operation, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsLatency(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsLatencyAverage, props);
    }
    /**
     *  The number of PutRecords operations where at least one record succeeded, per Kinesis stream, measured over the
     *  specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsSuccess(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsSuccessAverage, props);
    }
    /**
     * The total number of records sent in a PutRecords operation per Kinesis data stream, measured over the specified
     * time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsTotalRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsTotalRecordsAverage, props);
    }
    /**
     * The number of successful records in a PutRecords operation per Kinesis data stream, measured over the specified
     * time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsSuccessfulRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsSuccessfulRecordsAverage, props);
    }
    /**
     * The number of records rejected due to internal failures in a PutRecords operation per Kinesis data stream,
     * measured over the specified time period. Occasional internal failures are to be expected and should be retried.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsFailedRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsFailedRecordsAverage, props);
    }
    /**
     * The number of records rejected due to throttling in a PutRecords operation per Kinesis data stream, measured over
     * the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsThrottledRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsThrottledRecordsAverage, props);
    }
    /**
     * The number of bytes successfully put to the Kinesis stream over the specified time period. This metric includes
     * bytes from PutRecord and PutRecords operations. Minimum, Maximum, and Average statistics represent the bytes in a
     * single put operation for the stream in the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricIncomingBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.incomingBytesAverage, props);
    }
    /**
     * The number of records successfully put to the Kinesis stream over the specified time period. This metric includes
     * record counts from PutRecord and PutRecords operations. Minimum, Maximum, and Average statistics represent the
     * records in a single put operation for the stream in the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricIncomingRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.incomingRecordsAverage, props);
    }
    /**
     * The number of GetRecords calls throttled for the stream over the specified time period. The most commonly used
     * statistic for this metric is Average.
     *
     * When the Minimum statistic has a value of 1, all records were throttled for the stream during the specified time
     * period.
     *
     * When the Maximum statistic has a value of 0 (zero), no records were throttled for the stream during the specified
     * time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties
     *
     * @param props properties of the metric
     *
     */
    metricReadProvisionedThroughputExceeded(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.readProvisionedThroughputExceededAverage, props);
    }
    /**
     * The number of records rejected due to throttling for the stream over the specified time period. This metric
     * includes throttling from PutRecord and PutRecords operations.
     *
     * When the Minimum statistic has a non-zero value, records were being throttled for the stream during the specified
     * time period.
     *
     * When the Maximum statistic has a value of 0 (zero), no records were being throttled for the stream during the
     * specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricWriteProvisionedThroughputExceeded(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.writeProvisionedThroughputExceededAverage, props);
    }
    // create metrics based on generated KinesisMetrics static methods
    metricFromCannedFunction(createCannedProps, props) {
        return new cloudwatch.Metric({
            ...createCannedProps({ StreamName: this.streamName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
class Stream extends StreamBase {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id, {
            physicalName: props.streamName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_kinesis_StreamProps(props);
        let shardCount = props.shardCount;
        const streamMode = (_b = props.streamMode) !== null && _b !== void 0 ? _b : StreamMode.PROVISIONED;
        if (streamMode === StreamMode.ON_DEMAND && shardCount !== undefined) {
            throw new Error(`streamMode must be set to ${StreamMode.PROVISIONED} (default) when specifying shardCount`);
        }
        if (streamMode === StreamMode.PROVISIONED && shardCount === undefined) {
            shardCount = 1;
        }
        const retentionPeriodHours = (_d = (_c = props.retentionPeriod) === null || _c === void 0 ? void 0 : _c.toHours()) !== null && _d !== void 0 ? _d : 24;
        if (!core_1.Token.isUnresolved(retentionPeriodHours)) {
            if (retentionPeriodHours < 24 || retentionPeriodHours > 8760) {
                throw new Error(`retentionPeriod must be between 24 and 8760 hours. Received ${retentionPeriodHours}`);
            }
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, 'Resource', {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption,
            streamModeDetails: streamMode ? { streamMode } : undefined,
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    /**
     * Import an existing Kinesis Stream provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param streamArn Stream ARN (i.e. arn:aws:kinesis:<region>:<account-id>:stream/Foo)
     */
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_kinesis_StreamAttributes(attrs);
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).splitArn(attrs.streamArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _b;
        // if encryption properties are not set, default to KMS in regions where KMS is available
        if (!props.encryption && !props.encryptionKey) {
            const conditionName = 'AwsCdkKinesisEncryptedStreamsUnsupportedRegions';
            const existing = core_1.Stack.of(this).node.tryFindChild(conditionName);
            // create a single condition for the Stack
            if (!existing) {
                new core_1.CfnCondition(core_1.Stack.of(this), conditionName, {
                    expression: core_1.Fn.conditionOr(core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-north-1'), core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-northwest-1')),
                });
            }
            return {
                streamEncryption: core_1.Fn.conditionIf(conditionName, core_1.Aws.NO_VALUE, { EncryptionType: 'KMS', KeyId: 'alias/aws/kinesis' }),
            };
        }
        // default based on whether encryption key is specified
        const encryptionType = (_b = props.encryption) !== null && _b !== void 0 ? _b : (props.encryptionKey ? StreamEncryption.KMS : StreamEncryption.UNENCRYPTED);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return {};
        }
        if (encryptionType === StreamEncryption.MANAGED) {
            const encryption = { encryptionType: 'KMS', keyId: 'alias/aws/kinesis' };
            return { streamEncryption: encryption };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn,
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
_a = JSII_RTTI_SYMBOL_1;
Stream[_a] = { fqn: "@aws-cdk/aws-kinesis.Stream", version: "1.147.0" };
/**
 * What kind of server-side encryption to apply to this stream
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Records in the stream are not encrypted.
     */
    StreamEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    StreamEncryption["KMS"] = "KMS";
    /**
     * Server-side encryption with a master key managed by Amazon Kinesis
     */
    StreamEncryption["MANAGED"] = "MANAGED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
/**
 * Specifies the capacity mode to apply to this stream.
 */
var StreamMode;
(function (StreamMode) {
    /**
     * Specify the provisioned capacity mode. The stream will have `shardCount` shards unless
     * modified and will be billed according to the provisioned capacity.
     */
    StreamMode["PROVISIONED"] = "PROVISIONED";
    /**
     * Specify the on-demand capacity mode. The stream will autoscale and be billed according to the
     * volume of data ingested and retrieved.
     */
    StreamMode["ON_DEMAND"] = "ON_DEMAND";
})(StreamMode = exports.StreamMode || (exports.StreamMode = {}));
//# sourceMappingURL=data:application/json;base64,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