"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToTranslate = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToTranslate class.
 */
class LambdaToTranslate extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToTranslate class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToTranslateProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // All our tests are based upon this behavior being on, so we're setting
        // context here rather than assuming the client will set it
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        // Check props
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        defaults.CheckTranslateProps(props);
        // Setup VPC if required
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.TRANSLATE);
            if (props.asyncJobs) {
                defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.S3);
            }
        }
        const lambdaEnvironmentVariables = [];
        const translateConfiguration = defaults.ConfigureTranslateSupport(this, id, props);
        if (translateConfiguration.sourceBucket) {
            // Incorporate all the configuration created (to support async jobs)
            this.sourceBucket = translateConfiguration.sourceBucket.bucket;
            this.sourceLoggingBucket = translateConfiguration.sourceBucket.loggingBucket;
            this.sourceBucketInterface = translateConfiguration.sourceBucket.bucketInterface;
            this.destinationBucket = translateConfiguration.destinationBucket?.bucket;
            this.destinationLoggingBucket = translateConfiguration.destinationBucket?.loggingBucket;
            this.destinationBucketInterface = translateConfiguration.destinationBucket?.bucketInterface;
            lambdaEnvironmentVariables.push({
                defaultName: "SOURCE_BUCKET_NAME",
                clientNameOverride: props.sourceBucketEnvironmentVariableName,
                value: this.sourceBucketInterface.bucketName
            });
            lambdaEnvironmentVariables.push({
                defaultName: "DESTINATION_BUCKET_NAME",
                clientNameOverride: props.destinationBucketEnvironmentVariableName,
                value: this.destinationBucketInterface?.bucketName
            });
            lambdaEnvironmentVariables.push({
                defaultName: "DATA_ACCESS_ROLE_ARN",
                clientNameOverride: props.dataAccessRoleArnEnvironmentVariableName,
                value: translateConfiguration.translateRole?.roleArn
            });
        }
        // Now we know everything the Lambda Function needs, we can configure it
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        // Add all actions from translate configuration and client to the Lambda function
        // PassRole is handled separately, because it must specify role being passed as the resource
        const lambdaFunctionRoleActions = [];
        translateConfiguration.lambdaIamActionsRequired.forEach(action => {
            lambdaFunctionRoleActions.push(action);
        });
        if (props.additionalPermissions) {
            props.additionalPermissions.forEach(permission => {
                if (!lambdaFunctionRoleActions.includes(permission)) {
                    lambdaFunctionRoleActions.push(permission);
                }
            });
        }
        this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: lambdaFunctionRoleActions,
            resources: ['*']
        }));
        // Add PassRole in it's own statement
        if (translateConfiguration.translateRole) {
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ["iam:PassRole"],
                resources: [translateConfiguration.translateRole.roleArn]
            }));
        }
        // Configure environment variables
        lambdaEnvironmentVariables.forEach(variable => {
            const varName = variable.clientNameOverride || variable.defaultName;
            this.lambdaFunction.addEnvironment(varName, variable.value);
        });
    }
}
exports.LambdaToTranslate = LambdaToTranslate;
_a = JSII_RTTI_SYMBOL_1;
LambdaToTranslate[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-translate.LambdaToTranslate", version: "2.97.0" };
//# sourceMappingURL=data:application/json;base64,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