"""
SurrealDB helpers
"""

import atexit
import base64
import re
import os
import signal
import sys
import time


from subprocess import Popen, PIPE, TimeoutExpired

from glom import glom, Iter
from agptools.containers import flatten
from ..crud import parse_duri, DEFAULT_DATABASE, DEFAULT_NAMESPACE, tf

from ..storage import Surrealist
from ..definitions import MONOTONIC_KEY

# from ..syncmodels import SyncModel


def to64(x):
    x = f"{x}"
    x = bytes(x, "utf-8")
    x = base64.b64encode(x)
    x = x.decode("utf-8")
    x = x.replace("=", "_")

    # y = from64(x)
    return x


def from64(x):
    x = x.replace("_", "=")
    x = base64.b64decode(x)
    x = x.decode("utf-8")
    return x


# ----------------------------------------------------------
# Surreal Server
# ----------------------------------------------------------
class SurrealServer:
    """A helper to launch a local surrealDB server"""

    def __init__(self, path, bind="0.0.0.0:9000", daemon=False):
        self.path = path
        self.bind = bind
        self.daemon = daemon
        self.proc = None
        self.pid = None

    def cmd(self):
        return [
            "surreal",
            "start",
            "--allow-all",
            "--bind",
            f"{self.bind}",
            "--user",
            "root",
            "--pass",
            "root",
            f"file://{self.path}",
        ]

    def start(self):
        """starts surreal process and register a callback is anything goes wrong"""
        os.makedirs(self.path, exist_ok=True)

        extra = {}
        self.pid = os.getpid()

        def launch():
            # launch server
            self.proc = Popen(
                self.cmd(),
                stdout=PIPE,
                stderr=PIPE,
                **extra,
            )
            # give sometime to communicate with process
            # so server will be ready or we get some error feedback
            try:
                stdout, stderr = self.proc.communicate(timeout=0.5)
                print(stdout)
                print(stderr)
                return False
                # raise RuntimeError()  # something was wrong

            except TimeoutExpired:
                pass

            # print(f"Server pid: {self.pid}")
            if self.daemon:
                # with open(f"{self.url}/pid", "w", encoding="utf-8") as f:
                # f.write(f"{self.pid}")
                pass
            else:
                # stop process when parent process may die
                atexit.register(self.stop)

            return True

        result = False
        if self.daemon:
            try:
                print("forking process ...")
                pid = os.fork()
                self.pid = os.getpid()
                if pid:
                    result = True
                else:
                    print(f"child launch server")
                    result = launch()
                    # detach
                    print(f"child detach fds")
                    sys.stdin.close()
                    sys.stdout.close()
                    sys.stderr.close()
            except OSError as why:
                print(why)
                os._exit(1)
        else:
            result = launch()

        return result

    def stop(self):
        """stops child process and unregister callback"""
        if self.daemon:
            # find process that match the launching arguments
            cmd = "\0".join(self.cmd())
            for root, folders, _ in os.walk("/proc"):
                for pid in folders:
                    if re.match(r"\d+", pid):
                        try:
                            cmdline = open(
                                f"{root}/{pid}/cmdline",
                                "r",
                                encoding="utf-8",
                            ).read()
                            if cmd in cmdline:
                                print(
                                    f"Stopping: {pid} : {' '.join(self.cmd())}"
                                )
                                os.kill(int(pid), signal.SIGTERM)
                                return True
                        except Exception:
                            pass
            else:
                cmd = " ".join(self.cmd())
                print(f"can't find a surreal server such: '{cmd}'")
                return False
        else:
            self.proc.terminate()
            atexit.unregister(self.stop)
        return True

    @property
    def url(self):
        url = f"http://localhost:{self.bind.split(':')[-1]}"
        return url


# ----------------------------------------------------------
# SurrealStatsfrom .crud import parse_duri, DEFAULT_DATABASE, DEFAULT_NAMESPACE, tf


# ----------------------------------------------------------
class SurrealStats:
    def __init__(self, url):
        self.url = url
        self.inventory = {}

    def collect(
        self,
        include_ns=[],
        skip_ns=['system'],
    ):
        surreal = Surrealist(self.url)
        for _ in range(20):
            if surreal.is_ready():
                break
            time.sleep(0.1)

        con = surreal.connect()

        inventory = self.inventory
        namespaces = {}
        inventory['ns'] = namespaces

        if include_ns:
            NS = set(include_ns)
        else:
            inventory['root'] = root = con.root_info(structured=True).result
            NS = set(glom(root, 'namespaces.*.name'))

        for ns in NS.difference(skip_ns):
            ns_holder = namespaces.setdefault(ns, {})
            con.use(ns, '')
            ns_info = con.ns_info(structured=True).result

            for db in glom(ns_info, 'databases.*.name'):
                db_holder = ns_holder.setdefault(db, {})
                r = con.use(ns, db)
                db_info = con.db_info(structured=True).result
                for tb in glom(db_info, 'tables.*.name'):
                    tb_holder = db_holder.setdefault(tb, {})
                    tb_info = con.table_info(tb, structured=True).result
                    tb_holder.update(tb_info)
                    tb_holder['count'] = con.count(tb).result

        # stats
        stats = inventory['stats'] = {}
        stats['ns'] = glom(inventory, ('ns', len))
        stats['db'] = glom(inventory, ('ns.*', Iter().flatten().all(), len))
        stats['tb'] = glom(inventory, ('ns.*.*', flatten, Iter().all(), len))
        stats['records'] = sum(glom(inventory, ('ns.*.*.*.count', flatten)))

        return self.inventory

    def samples(self, n=2):
        if not self.inventory:
            self.collect()

        surreal = Surrealist(self.url)
        con = surreal.connect()
        universe = self.inventory['ns']
        samples = self.inventory['samples'] = {}
        tubes = self.inventory['tubes'] = []
        for ns, databases in universe.items():
            for db, tables in databases.items():
                con.use(ns, db)
                for table in tables:
                    sql = f"SELECT * FROM {table} ORDER BY id DESC LIMIT {n}"
                    res = con.query(sql)

                    tubename = f"{ns}://{db}/{table}"
                    tubes.append(tubename)
                    samples[tubename] = res.result

        tubes.sort()
        return samples

    def check(self, uri):
        _uri = parse_duri(uri)
        namespace = _uri['fscheme']
        database = _uri['host']
        table = _uri['table']

        surreal = Surrealist(self.url, namespace, database)
        for _ in range(20):
            if surreal.is_ready():
                break
            time.sleep(0.1)
        con = surreal.connect()
        res = con.query(f"SELECT * from {table} ORDER BY {MONOTONIC_KEY}")
        for record in res.result:
            foo = 1
        foo = 1
