"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = void 0;
const fs = require("fs");
const path = require("path");
const assets = require("@aws-cdk/assets");
const ecr = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const minimatch = require("minimatch");
/**
 * (experimental) An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 *
 * @experimental
 */
class DockerImageAsset extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.relativeStagedPath(stack),
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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