#!/usr/bin/python
# (C) Copyright 2006 Olivier Grisel
# Author: Olivier Grisel <olivier.grisel@ensta.org>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.
#
# $Id$
"""trace2html.py [options]

Utility to generate HTML coverage reports

trace2html uses coverage data collected with the `trace` module of the
standard python library and turn into a set of human readable HTML documents
showing sortable summary and annotated source files.

Examples
--------

Generate coverage data in the 'counts' file with trace.py::

  $ /usr/lib/python2.4/trace.py -mc -C coverage_dir -f counts my_testrunner.py

Write a report in directory 'coverage_dir' from data collected in 'counts'::

  $ trace2html.py -f counts -o coverage_dir
  $ firefox coverage_dir/index.html

Load data from several files (the report is written to 'coverage_dir' by
default)::

  $ trace2html.py -f file1 -f2 file2
  $ firefox coverage_dir/index.html
"""

import cgi
import datetime
import linecache
import optparse
import os
import sys
import trace
import urllib

try:
    # Extracted resources from the trace2html data even if it's packaged as
    # an egg
    from pkg_resources import resource_string
    TRACE2HTML_CSS = resource_string('trace2htmldata', 'trace2html.css')
    SORTABLETABLE_JS = resource_string('trace2htmldata', 'sortabletable.js')
except ImportError:
    # distutils backward compatibility
    import trace2htmldata
    resources = trace2htmldata.__path__[0]
    TRACE2HTML_CSS = file(os.path.join(resources, 'trace2html.css')).read()
    SORTABLETABLE_JS = file(os.path.join(resources, 'sortabletable.js')).read()

#
# First some HTML templates
#

HTML_PAGE = """\
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
  "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=UTF-8"/>
  <title>%(title)s</title>
  <link title="Style" type="text/css" rel="stylesheet" href="%(css_file)s"/>
  <script type="text/javascript" src="sortabletable.js"></script>
</head>

<body>
<h1>%(title)s</h1>

%(body)s

<p class="footer">
Report generated by <a href="http://cheeseshop.python.org/pypi/trace2html"
>trace2html</a> on %(date)s.
</p>

</body>

</html>
"""

SUMMARY = """\
<table id="summary" class="summary">
<thead>
  <tr><td>Module</td><td>Coverage %%</td></tr>
</thead>
<tbody>
%(summary_lines)s
</tbody>
</table>
<script type="text/javascript">
var packageTable = new SortableTable(document.getElementById("summary"),
    ["String", "Percentage"]);
packageTable.sort(0);
</script>
"""

SUMMARY_LINE = """\
<tr class=%(row_class)s>
  <td class="moduleName">
    <a href="%(module_link)s">%(modulename)s</a></td>
  <td class="percent">
    <div class="coveragePercent">
      %(percent)d%%
    </div>
    <div class="coverageBar">
      <div class="inCoverageBar" style="width: %(percent)d%%">
      </div>
    </div>
  </td>
</tr>
"""

ANNOTATED_SOURCE_FILE = """\
%(summary)s

<table class="srcCode">
<tbody>
%(src_lines)s
</tbody>
</table>
"""

ANNOTATED_SOURCE_LINE = """\
<tr class="%(coverage_class)s">
  <td class="nbLine">%(line_number)s</td>
  <td class="nbHits">%(line_hits)s</td>
  <td class="src">%(src_line)s</td>
</tr>
"""

#
# Command line options parsing
#

def parse_args():
    """Use optparse to extract options from the argv list

    >>> sys.argv[1:] = '--coverage-file=/tmp/some_file -f another_file'.split()
    >>> options, _ = parse_args()
    >>> options.coverage_files
    ['/tmp/some_file', 'another_file']
    >>> options.report_dir
    'coverage_dir'

    >>> sys.argv[1:] = '-f another_file -o /tmp/report'.split()
    >>> options, _ = parse_args()
    >>> options.coverage_files
    ['another_file']
    >>> options.report_dir
    '/tmp/report'
    """
    parser = optparse.OptionParser(usage=__doc__)

    parser.add_option('-f', '--coverage-file',
                      action='append',
                      dest='coverage_files',
                      default=[],
                      help="Use the content of a trace file")

    parser.add_option('-o', '--output-dir',
                      action='store',
                      dest='report_dir',
                      default='coverage_dir',
                      help="Directory to store the generated HTML report. "
                           "Defaults to 'coverage_dir'")

    # TODO: use it!
    parser.add_option('-s', '--with-css-stylesheet',
                      action='store',
                      dest='css',
                      default=None,
                      help="Use an alternative CSS stylesheet")

    parser.add_option('-t', '--self-test',
                      action='store_true',
                      dest='selftest',
                      default=False,
                      help="Run the tests for trace2html")

    parser.add_option('-v', '--verbose',
                      action='count',
                      default=0,
                      help="Set verbose mode (cumulative), default is quiet")

    return parser.parse_args()

class CoverageHtmlResults(trace.CoverageResults):
    """Extend the coverage results class to add the ability to compute coverage
    summary data and write it down as html reports
    """
    _page_pattern = HTML_PAGE
    _summary_pattern = SUMMARY
    _summary_line_pattern = SUMMARY_LINE
    _annotated_src_file_pattern = ANNOTATED_SOURCE_FILE
    _annotated_src_line_pattern = ANNOTATED_SOURCE_LINE

    _default_css_filename = 'trace2html.css'
    _default_js_filename = 'sortabletable.js'

    def writeHtmlReport(self, report_dir, css_filename=None):
        """Write the report of the collected data to report_dir after creating
        it if not existing

        This is highly 'inspirated' by how trace.CoverageResults writes it's
        reports.

        >>> import tempfile, shutil, os
        >>> report_dir = tempfile.mkdtemp()
        >>> reporter = CoverageHtmlResults()
        >>> reporter.writeHtmlReport(report_dir) # doctest: +ELLIPSIS
        '...index.html'
        >>> os.listdir(report_dir)
        ['trace2html.css', 'sortabletable.js', 'index.html']

        Cleaning the test directory
        >>> shutil.rmtree(report_dir)
        """

        # write the directory
        if not os.path.exists(report_dir):
            os.mkdir(report_dir)

        # write the css file
        if css_filename is not None:
            css_data = file(css_filename).read()
        else:
            css_filename = self._default_css_filename
            css_data = TRACE2HTML_CSS
        file(os.path.join(report_dir, css_filename), 'w').write(css_data)

        # write the js file for summary lines sorting
        js_filename = self._default_js_filename
        js_data = SORTABLETABLE_JS
        file(os.path.join(report_dir, js_filename), 'w').write(js_data)

        # turn the counts data ("(filename, lineno) = count") into something
        # accessible on a per-file basis
        per_file_counts = {}
        for (filename, lineno), count in self.counts.iteritems():
            lines_hit = per_file_counts.setdefault(filename, {})
            lines_hit[lineno] = count

        # accumulate summary info
        sums = {}

        for filename, count in per_file_counts.iteritems():
            # skip some "files" we don't care about...
            if filename == "<string>":
                continue

            if filename.endswith(".pyc") or filename.endswith(".pyo"):
                filename = filename[:-1]

            # Get a list of the line numbers which represent executable content
            # (returned as a dict for better lookup speed)
            lnotab = trace.find_executable_linenos(filename)

            source = linecache.getlines(filename)
            modulename = trace.fullmodname(filename)
            percent = self._writeAnnotatedSourceCodePage(
                report_dir, modulename, source, lnotab, count, css_filename)

            sums[modulename] = percent

        # write the summary
        index_filename = os.path.join(report_dir, 'index.html')
        self._writePage(index_filename, 'Coverage Report - All Modules',
                        self._summary(sums), css_filename)

        return os.path.abspath(index_filename)

    def _writeAnnotatedSourceCodePage(self, report_dir, modulename, lines,
                                      lnotab, lines_hit, css_filename=None):
        """Write an annotated html version of the source code of the module

        This is highly inspirated by the CoverageResults.write_results_file
        method.
        """

        filename = os.path.join(report_dir, modulename + ".html")

        # counters for the summary
        n_hits, n_lines = 0, 0

        annotated_lines = ''
        for i, line in enumerate(lines):
            lineno = i + 1
            if line.strip():
                src_line = '<pre>%s</pre>' % cgi.escape(line[:-1])
            else:
                src_line = ''

            params = {
                'coverage_class': '',
                'line_number': str(lineno),
                'line_hits': '',
                'src_line': src_line,
            }
            # do the blank/comment match to try to mark more lines
            # (help the reader find stuff that hasn't been covered)
            if lineno in lines_hit:
                params['line_hits'] = str(lines_hit[lineno])
                n_hits += 1
                n_lines += 1
            elif not trace.rx_blank.match(line):
                # lines preceded by no marks weren't hit
                # Highlight them if so indicated, unless the line contains
                # #pragma: NO COVER
                if lineno in lnotab and not trace.PRAGMA_NOCOVER in lines[i]:
                    n_lines += 1
                    params['line_hits'] = '0'
                    params['coverage_class'] = 'uncovered'
            annotated_lines += ANNOTATED_SOURCE_LINE % params

        if not n_lines:
            n_lines, n_hits = 1, 1
        percent = int(100 * n_hits / n_lines)
        summary = self._summary({modulename: percent})

        body = ANNOTATED_SOURCE_FILE % {
            'module_path': modulename,
            'summary': summary,
            'src_lines': annotated_lines,
        }
        title = "Coverage Report - %s" % modulename
        self._writePage(filename, title, body, css_filename)
        return percent

    def _summary(self, stats):
        summary_lines = ''
        for i, (modulename, percent) in enumerate(sorted(stats.iteritems())):
            summary_lines += SUMMARY_LINE % {
                'modulename': cgi.escape(modulename),
                'module_link': urllib.quote(modulename + '.html'),
                'percent': percent,
                'row_class': i % 2 is 0 and 'even' or 'odd',
            }
        return SUMMARY % {'summary_lines': summary_lines}

    def _writePage(self, filename, title, body, css_filename=None):
        """Generate an html page a write it to filename"""
        data = {
            'title': title,
            'body': body,
            'css_file': css_filename or self._default_css_filename,
            'date': str(datetime.datetime.now()),
        }
        file(filename, 'w').write(self._page_pattern % data)

#
# selftest support
#

def _test(verbose=0):
    import doctest
    return doctest.testmod(verbose=verbose)

def test_suite():
    """setuptools testrunner integration"""
    import trace2html, doctest
    return doctest.DocTestSuite(trace2html)

#
# Then comes the logic
#

def main(options, args):
    """Write a report according to the collected options"""
    # all args should be formatted as options
    if args:
        # TODO: use args to trace some program
        print "unregnised arguments %r, please run trace2html.py --help" % (
            args[0])
        sys.exit(1)

    # self testing
    if options.selftest:
        # returns the number of failures as return code
        sys.exit(_test(options.verbose)[0])

    if not options.coverage_files:
        # avoid writing a report for nothing
        sys.exit(0)

    # collecting
    reporter = CoverageHtmlResults()
    for coverage_file in options.coverage_files:
        reporter.update(trace.CoverageResults(infile=coverage_file))

    # writing the report
    try:
        report_path = reporter.writeHtmlReport(options.report_dir, options.css)
        if options.verbose:
            print "Report written to: %s" % report_path
    except IOError, err:
        print >> sys.stderr, "Could not write the report, cause: %s" % err
        sys.exit(1)

# Run away!
if __name__ == "__main__": #pragma NO COVER
    main(*parse_args())


