"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
const testPort = 12321;
const testFunctionName = "something-unique";
const testClusterName = "something-else";
test("When provided a VPC, does not create a second VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
});
test("When provided an existingCache, does not create a second cache", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc, testPort);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingCache,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ElastiCache::CacheCluster", 1);
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        Port: testPort,
    });
});
test("When provided an existingFunction, does not create a second function", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: ".handler",
        vpc: existingVpc,
        functionName: testFunctionName,
    });
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingLambdaObj: existingFunction,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test custom environment variable name", () => {
    const stack = new cdk.Stack();
    const testEnvironmentVariableName = "CUSTOM_CLUSTER_NAME";
    new lib_1.LambdaToElasticachememcached(stack, "test-construct", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        cacheEndpointEnvironmentVariableName: testEnvironmentVariableName,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CUSTOM_CLUSTER_NAME: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
    });
});
test("Test setting custom function properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
            functionName: testFunctionName,
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test setting custom cache properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        cacheProps: {
            clusterName: testClusterName,
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        ClusterName: testClusterName,
    });
});
test("Test setting custom VPC properties", () => {
    const stack = new cdk.Stack();
    const testCidrBlock = "192.168.0.0/16";
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        vpcProps: {
            ipAddresses: ec2.IpAddresses.cidr(testCidrBlock),
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidrBlock,
    });
});
test("Test all default values", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.resourceCountIs("AWS::ElastiCache::CacheCluster", 1);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CACHE_ENDPOINT: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
        Handler: ".handler",
        Runtime: "nodejs16.x",
    });
    // All values taken from elasticache-defaults.ts
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        CacheNodeType: "cache.t3.medium",
        Engine: "memcached",
        NumCacheNodes: 2,
        Port: 11222,
        AZMode: "cross-az",
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Test for the proper self referencing security group', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        cacheProps: {
            port: 22223
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: 22223,
        ToPort: 22223,
        GroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
    });
});
// test('', () => {});
test("Test error from existingCache and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existing function and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: ".handler",
        vpc: existingVpc,
    });
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingLambdaObj: existingFunction,
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existingCache and cacheProps", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            existingVpc,
            cacheProps: {
                numCacheNodes: 4,
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("Cannot specify existingCache and cacheProps");
});
test("Test error from trying to launch Redis", () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            cacheProps: {
                numCacheNodes: 4,
                engine: "redis",
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("This construct can only launch memcached clusters");
});
//# sourceMappingURL=data:application/json;base64,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