import datetime
import enum
import os
import pathlib
import types
from collections.abc import Callable, Iterable, Iterator, Mapping, Sequence
from typing import Annotated, Any, Generic, Literal, Protocol, TypeAlias, TypeVar, overload

import dimod.sampleset
import numpy
import numpy.typing
from typing_extensions import Buffer

from . import _backward as _backward
from ._backward import BinaryIntMatrix as BinaryIntMatrix
from ._backward import BinaryIntPoly as BinaryIntPoly
from ._backward import BinaryIntPolyArray as BinaryIntPolyArray
from ._backward import BinaryIntQuadraticModel as BinaryIntQuadraticModel
from ._backward import BinaryIntSymbolGenerator as BinaryIntSymbolGenerator
from ._backward import BinaryMatrix as BinaryMatrix
from ._backward import BinaryPoly as BinaryPoly
from ._backward import BinaryPolyArray as BinaryPolyArray
from ._backward import BinaryQuadraticModel as BinaryQuadraticModel
from ._backward import BinarySymbolGenerator as BinarySymbolGenerator
from ._backward import InequalityFormulation as InequalityFormulation
from ._backward import IsingIntMatrix as IsingIntMatrix
from ._backward import IsingIntPoly as IsingIntPoly
from ._backward import IsingIntPolyArray as IsingIntPolyArray
from ._backward import IsingIntQuadraticModel as IsingIntQuadraticModel
from ._backward import IsingIntSymbolGenerator as IsingIntSymbolGenerator
from ._backward import IsingMatrix as IsingMatrix
from ._backward import IsingPoly as IsingPoly
from ._backward import IsingPolyArray as IsingPolyArray
from ._backward import IsingQuadraticModel as IsingQuadraticModel
from ._backward import IsingSymbolGenerator as IsingSymbolGenerator
from ._backward import QuadraticModel as QuadraticModel
from ._backward import Solver as Solver
from ._backward import SolverResult as SolverResult
from ._backward import SolverSolution as SolverSolution
from ._backward import SymbolGenerator as SymbolGenerator
from ._backward import convert_to_matrix as convert_to_matrix
from ._backward import decode_solution as decode_solution
from ._backward import gen_symbols as gen_symbols
from ._backward import intersection as intersection
from ._backward import pair_sum as pair_sum
from ._backward import product as product
from ._backward import replace_all as replace_all
from ._backward import sum_poly as sum_poly
from ._backward import symmetric_difference as symmetric_difference
from ._backward import union as union

class DLPackBuffer(Protocol):
    def __dlpack__(self) -> object: ...

NDArray: TypeAlias = Buffer | DLPackBuffer

class AcceptableDegrees:
    def __init__(
        self,
        objective: dict[
            VariableType | Literal["Binary", "Ising", "Integer", "Real"],
            Degree | Literal["Zero", "Linear", "Quadratic", "Cubic", "Quartic", "HighOrder"] | int,
        ] = ...,
        equality_constraints: dict[
            VariableType | Literal["Binary", "Ising", "Integer", "Real"],
            Degree | Literal["Zero", "Linear", "Quadratic", "Cubic", "Quartic", "HighOrder"] | int,
        ] = ...,
        inequality_constraints: dict[
            VariableType | Literal["Binary", "Ising", "Integer", "Real"],
            Degree | Literal["Zero", "Linear", "Quadratic", "Cubic", "Quartic", "HighOrder"] | int,
        ] = ...,
    ) -> None: ...
    @property
    def objective(self) -> dict[VariableType, Degree]:
        """objective

        Returns:
            dict[amplify.VariableType, amplify.Degree]:
        """

    @property
    def equality_constraints(self) -> dict[VariableType, Degree]:
        """equality_constraints

        Returns:
            dict[amplify.VariableType, amplify.Degree]:
        """

    @property
    def inequality_constraints(self) -> dict[VariableType, Degree]:
        """inequality_constraints

        Returns:
            dict[amplify.VariableType, amplify.Degree]:
        """

class AmplifyClient:
    """Base class of all clients in the Amplify SDK"""

class Constraint:
    @overload
    def __init__(
        self, left: Poly, bounds: tuple[float | None, float | None], penalty: Poly, label: str = ""
    ) -> None: ...
    @overload
    def __init__(self, left: Poly, eq: float, penalty: Poly, label: str = "") -> None: ...
    @overload
    def __init__(self, left: Poly, le: float, penalty: Poly, label: str = "") -> None: ...
    @overload
    def __init__(self, left: Poly, ge: float, penalty: Poly, label: str = "") -> None: ...
    @property
    def weight(self) -> float:
        """weight

        Returns:
            float:
        """

    @weight.setter
    def weight(self, arg: float, /) -> None:
        """
        Args:
            arg (float)
        """

    @property
    def label(self) -> str:
        """label

        Returns:
            str:
        """

    @label.setter
    def label(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    def __eq__(self, arg: object, /) -> bool: ...
    def __ne__(self, arg: object, /) -> bool: ...
    def __mul__(self, arg: float, /) -> Constraint: ...
    def __rmul__(self, arg: float, /) -> Constraint: ...
    def __imul__(self, arg: float, /) -> Constraint: ...
    def __truediv__(self, arg: float, /) -> Constraint: ...
    def __itruediv__(self, arg: float, /) -> Constraint: ...
    def __add__(self, arg: Constraint, /) -> ConstraintList: ...
    def __radd__(self, arg: int, /) -> Constraint: ...
    @property
    def penalty(self) -> Poly:
        """penalty

        Returns:
            amplify.Poly:
        """

    @property
    def conditional(self) -> tuple[Poly, Literal["EQ", "GE", "GT", "LE", "LT", "BW"], float | tuple[float, float]]:
        """conditional

        Returns:
            tuple[amplify.Poly, typing.Literal['EQ', 'GE', 'GT', 'LE', 'LT', 'BW'], float | tuple[float, float]]:
        """

    @overload
    def is_satisfied(self, values: Values) -> bool: ...
    @overload
    def is_satisfied(self, mapping: Mapping[Poly, float] | Mapping[Variable, float]) -> bool:
        """is_satisfied

        .. admonition:: Overload 1.

            Args:
                * values (amplify.Values)

            Returns:
                bool:

        .. admonition:: Overload 2.

            Args:
                * mapping (collections.abc.Mapping[amplify.Poly, float] | collections.abc.Mapping[amplify.Variable, float])

            Returns:
                bool:
        """

class ConstraintList:
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, arg: Sequence[Constraint]) -> None: ...
    @overload
    def __init__(self, arg: Iterable[Constraint]) -> None: ...
    def __eq__(self, arg: object, /) -> bool: ...
    def __ne__(self, arg: object, /) -> bool: ...
    @overload
    def __add__(self, arg: ConstraintList, /) -> ConstraintList: ...
    @overload
    def __add__(self, arg: Constraint, /) -> ConstraintList: ...
    @overload
    def __radd__(self, arg: Constraint, /) -> ConstraintList: ...
    @overload
    def __radd__(self, arg: int, /) -> ConstraintList: ...
    @overload
    def __iadd__(self, arg: ConstraintList, /) -> ConstraintList: ...
    @overload
    def __iadd__(self, arg: Constraint, /) -> ConstraintList: ...
    def __mul__(self, arg: float, /) -> ConstraintList: ...
    def __rmul__(self, arg: float, /) -> ConstraintList: ...
    def __imul__(self, arg: float, /) -> ConstraintList: ...
    def __truediv__(self, arg: float, /) -> ConstraintList: ...
    def __itruediv__(self, arg: float, /) -> ConstraintList: ...
    def __len__(self) -> int: ...
    def __getitem__(self, key: int) -> Constraint: ...
    def __setitem__(self, key: int, value: Constraint) -> None: ...
    def __copy__(self) -> ConstraintList: ...
    def __deepcopy__(self, memo: object | None = None) -> ConstraintList: ...

    class Iterator:
        def __iter__(self) -> ConstraintList.Iterator: ...
        def __next__(self) -> Constraint: ...

    def __iter__(self) -> ConstraintList.Iterator: ...
    def append(self, value: Constraint) -> None:
        """append

        Args:
            value (amplify.Constraint)
        """

    def clear(self) -> None:
        """clear"""

    def extend(self, sequence: Sequence[Constraint]) -> None:
        """extend

        Args:
            sequence (collections.abc.Sequence[amplify.Constraint])
        """

    def insert(self, index: int, value: Constraint) -> None:
        """insert

        Args:
            index (int)
            value (amplify.Constraint)
        """

    def remove(self, value: Constraint) -> None:
        """remove

        Args:
            value (amplify.Constraint)
        """

    def pop(self, index: int | None = None) -> Constraint:
        """pop

        Args:
            index (int | None): Defaults to ``None``.

        Returns:
            amplify.Constraint:
        """

    def count(self, value: Constraint) -> int:
        """count

        Args:
            value (amplify.Constraint)

        Returns:
            int:
        """

    @overload
    def index(self, value: Constraint) -> int: ...
    @overload
    def index(self, value: Constraint, start: int) -> int: ...
    @overload
    def index(self, value: Constraint, start: int, end: int) -> int:
        """index

        .. admonition:: Overload 1.

            Args:
                * value (amplify.Constraint)

            Returns:
                int:

        .. admonition:: Overload 2.

            Args:
                * value (amplify.Constraint)
                * start (int)

            Returns:
                int:

        .. admonition:: Overload 3.

            Args:
                * value (amplify.Constraint)
                * start (int)
                * end (int)

            Returns:
                int:
        """

    def reverse(self) -> None:
        """reverse"""

    def copy(self) -> ConstraintList:
        """copy

        Returns:
            amplify.ConstraintList:
        """

class DASClient(AmplifyClient):
    def __init__(self) -> None: ...

    class Parameters:
        class Schedule:
            @property
            def algorithm(self) -> DASClient.Algorithm | None:
                """algorithm

                Returns:
                    amplify.DASClient.Algorithm | None:
                """

            @algorithm.setter
            def algorithm(self, arg: Literal["ALL", "SA", "QMC", "PT"] | DASClient.Algorithm | None) -> None:
                """
                Args:
                    arg (typing.Literal['ALL', 'SA', 'QMC', 'PT'] | amplify.DASClient.Algorithm | None)
                """

            @property
            def anneal_time(self) -> int | None:
                """anneal_time

                Returns:
                    int | None:
                """

            @anneal_time.setter
            def anneal_time(self, arg: int | None) -> None:
                """
                Args:
                    arg (int | None)
                """

            @property
            def temper(self) -> list[float] | None:
                """temper

                Returns:
                    list[float] | None:
                """

            @temper.setter
            def temper(self, arg: Sequence[float] | None) -> None:
                """
                Args:
                    arg (collections.abc.Sequence[float] | None)
                """

            @property
            def gamma(self) -> list[float] | None:
                """gamma

                Returns:
                    list[float] | None:
                """

            @gamma.setter
            def gamma(self, arg: Sequence[float] | None) -> None:
                """
                Args:
                    arg (collections.abc.Sequence[float] | None)
                """

        @property
        def auto(self) -> bool | None:
            """auto

            Returns:
                bool | None:
            """

        @auto.setter
        def auto(self, arg: bool | None) -> None:
            """
            Args:
                arg (bool | None)
            """

        @property
        def algorithm(self) -> DASClient.Algorithm | None:
            """algorithm

            Returns:
                amplify.DASClient.Algorithm | None:
            """

        @algorithm.setter
        def algorithm(self, arg: str | DASClient.Algorithm | None) -> None:
            """
            Args:
                arg (str | amplify.DASClient.Algorithm | None)
            """

        @property
        def trotter(self) -> int | None:
            """trotter

            Returns:
                int | None:
            """

        @trotter.setter
        def trotter(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def anneal_time(self) -> int | None:
            """anneal_time

            Returns:
                int | None:
            """

        @anneal_time.setter
        def anneal_time(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def repetition(self) -> int | None:
            """repetition

            Returns:
                int | None:
            """

        @repetition.setter
        def repetition(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def proc(self) -> DASClient.Proc | None:
            """proc

            Returns:
                amplify.DASClient.Proc | None:
            """

        @proc.setter
        def proc(self, arg: Literal["CPU", "GPU"] | DASClient.Proc | None) -> None:
            """
            Args:
                arg (typing.Literal['CPU', 'GPU'] | amplify.DASClient.Proc | None)
            """

        @property
        def schedule_type(self) -> DASClient.ScheduleType | None:
            """schedule_type

            Returns:
                amplify.DASClient.ScheduleType | None:
            """

        @schedule_type.setter
        def schedule_type(self, arg: Literal["Linear", "Power", "Expo"] | DASClient.ScheduleType | None) -> None:
            """
            Args:
                arg (typing.Literal['Linear', 'Power', 'Expo'] | amplify.DASClient.ScheduleType | None)
            """

        @property
        def sa_temper(self) -> float | None:
            """sa_temper

            Returns:
                float | None:
            """

        @sa_temper.setter
        def sa_temper(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def qmc_gamma(self) -> float | None:
            """qmc_gamma

            Returns:
                float | None:
            """

        @qmc_gamma.setter
        def qmc_gamma(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def qmc_temper(self) -> float | None:
            """qmc_temper

            Returns:
                float | None:
            """

        @qmc_temper.setter
        def qmc_temper(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def decrease_factor(self) -> float | None:
            """decrease_factor

            Returns:
                float | None:
            """

        @decrease_factor.setter
        def decrease_factor(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def pt_temper(self) -> float | None:
            """pt_temper

            Returns:
                float | None:
            """

        @pt_temper.setter
        def pt_temper(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def pt_factor(self) -> float | None:
            """pt_factor

            Returns:
                float | None:
            """

        @pt_factor.setter
        def pt_factor(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def schedule(self) -> DASClient.Parameters.Schedule:
            """schedule

            Returns:
                amplify.DASClient.Parameters.Schedule:
            """

    class Algorithm(enum.Enum):
        ALL = 0
        SA = 1
        QMC = 2
        PT = 3

    class ScheduleType(enum.Enum):
        Linear = 0
        Power = 1
        Expo = 2

    class Proc(enum.Enum):
        CPU = 0
        GPU = 1

    class Result:
        @property
        def execution_time(self) -> datetime.timedelta:
            """execution_time

            Returns:
                datetime.timedelta:
            """

        @property
        def energy(self) -> float:
            """energy

            Returns:
                float:
            """

        @property
        def spins(self) -> list[int]:
            """spins

            Returns:
                list[int]:
            """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def library_path(self) -> str | None:
        """library_path

        Returns:
            str | None:
        """

    @property
    def parameters(self) -> DASClient.Parameters:
        """parameters

        Returns:
            amplify.DASClient.Parameters:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> DASClient.Result: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.DASClient.Result:

        .. admonition:: Overload 2.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])
        """

class DWaveClientParameters:
    def __init__(self) -> None: ...
    def __getattr__(self, arg: object, /) -> Any: ...
    def __setattr__(self, arg0: object, arg1: object, /) -> None: ...

class DWaveSamplerClient(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", solver: str = "", url: str = "", proxy: str | None = None) -> None: ...
    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def solver(self) -> str:
        """solver

        Returns:
            str:
        """

    @solver.setter
    def solver(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def compress_qpu_problem_data(self) -> bool | None:
        """compress_qpu_problem_data

        Returns:
            bool | None:
        """

    @compress_qpu_problem_data.setter
    def compress_qpu_problem_data(self, arg: bool | None) -> None:
        """
        Args:
            arg (bool | None)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def parameters(self) -> DWaveClientParameters:
        """parameters

        Returns:
            amplify.DWaveClientParameters:
        """

    @property
    def solver_names(self) -> list[str]:
        """solver_names

        Returns:
            list[str]:
        """

    @property
    def properties(self) -> dict[str, Any]:
        """properties

        Returns:
            dict[str, typing.Any]:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly, dry_run: Literal[False] = False) -> dimod.sampleset.SampleSet: ...
    @overload
    def solve(self, objective: Poly, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                dimod.sampleset.SampleSet:

        .. admonition:: Overload 2.

            Args:
                * objective (amplify.Poly)
                * dry_run (typing.Literal[True])
        """

    @property
    def graph(self) -> Graph:
        """graph

        Returns:
            amplify.Graph:
        """

class Degree(enum.Enum):
    Zero = 0
    Linear = 1
    Quadratic = 2
    Cubic = 3
    Quartic = 4
    HighOrder = 5

class Dim:
    """Base class of dimension type of PolyArray"""

class Dim0(Dim):
    """0D PolyArray"""

class Dim1(Dim):
    """1D PolyArray"""

class Dim2(Dim):
    """2D PolyArray"""

class Dim3(Dim):
    """3D PolyArray"""

class Dim4(Dim):
    """4D PolyArray"""

class EmbeddingMethod(enum.Enum):
    Default = 0
    Minor = 1
    Clique = 2
    Parallel = 3

class FixstarsClient(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", url: str = "", proxy: str | None = None) -> None: ...

    class Parameters:
        class Outputs:
            @property
            def spins(self) -> bool | None:
                """spins

                Returns:
                    bool | None:
                """

            @spins.setter
            def spins(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def energies(self) -> bool | None:
                """energies

                Returns:
                    bool | None:
                """

            @energies.setter
            def energies(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def sort(self) -> bool | None:
                """sort

                Returns:
                    bool | None:
                """

            @sort.setter
            def sort(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def duplicate(self) -> bool | None:
                """duplicate

                Returns:
                    bool | None:
                """

            @duplicate.setter
            def duplicate(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def feasibilities(self) -> bool | None:
                """feasibilities

                Returns:
                    bool | None:
                """

            @feasibilities.setter
            def feasibilities(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def num_outputs(self) -> int | None:
                """num_outputs

                Returns:
                    int | None:
                """

            @num_outputs.setter
            def num_outputs(self, arg: int | None) -> None:
                """
                Args:
                    arg (int | None)
                """

        @property
        def timeout(self) -> datetime.timedelta | None:
            """timeout

            Returns:
                datetime.timedelta | None:
            """

        @timeout.setter
        def timeout(self, arg: int | datetime.timedelta | None) -> None:
            """
            Args:
                arg (int | datetime.timedelta | None)
            """

        @property
        def num_gpus(self) -> int | None:
            """num_gpus

            Returns:
                int | None:
            """

        @num_gpus.setter
        def num_gpus(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def penalty_calibration(self) -> bool | None:
            """penalty_calibration

            Returns:
                bool | None:
            """

        @penalty_calibration.setter
        def penalty_calibration(self, arg: bool | None) -> None:
            """
            Args:
                arg (bool | None)
            """

        @property
        def outputs(self) -> FixstarsClient.Parameters.Outputs:
            """outputs

            Returns:
                amplify.FixstarsClient.Parameters.Outputs:
            """

    class Result:
        class ExecutionTime:
            @property
            def annealing_time(self) -> datetime.timedelta:
                """annealing_time

                Returns:
                    datetime.timedelta:
                """

            @property
            def queue_time(self) -> datetime.timedelta:
                """queue_time

                Returns:
                    datetime.timedelta:
                """

            @property
            def cpu_time(self) -> datetime.timedelta:
                """cpu_time

                Returns:
                    datetime.timedelta:
                """

            @property
            def time_stamps(self) -> list[datetime.timedelta]:
                """time_stamps

                Returns:
                    list[datetime.timedelta]:
                """

        class ExecutionParameters:
            @property
            def num_gpus(self) -> int:
                """num_gpus

                Returns:
                    int:
                """

            @property
            def timeout(self) -> datetime.timedelta:
                """timeout

                Returns:
                    datetime.timedelta:
                """

            @property
            def num_iterations(self) -> int:
                """num_iterations

                Returns:
                    int:
                """

            @property
            def penalty_calibration(self) -> bool:
                """penalty_calibration

                Returns:
                    bool:
                """

            @property
            def penalty_multipliers(
                self,
            ) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64", "writable": False}]:
                """penalty_multipliers

                Returns:
                    numpy.ndarray:
                """

            @property
            def version(self) -> str:
                """version

                Returns:
                    str:
                """

        @property
        def execution_time(self) -> FixstarsClient.Result.ExecutionTime:
            """execution_time

            Returns:
                amplify.FixstarsClient.Result.ExecutionTime:
            """

        @property
        def energies(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64", "writable": False}]:
            """energies

            Returns:
                numpy.ndarray:
            """

        @property
        def spins(self) -> list[Annotated[numpy.typing.NDArray[numpy.int8], {"dtype": "int8", "writable": False}]]:
            """spins

            Returns:
                list[numpy.ndarray[dtype=int8, writable=False]]:
            """

        @property
        def feasibilities(self) -> Annotated[numpy.typing.NDArray[numpy.bool_], {"dtype": "bool", "writable": False}]:
            """feasibilities

            Returns:
                numpy.ndarray:
            """

        @property
        def execution_parameters(self) -> FixstarsClient.Result.ExecutionParameters:
            """execution_parameters

            Returns:
                amplify.FixstarsClient.Result.ExecutionParameters:
            """

        @property
        def message(self) -> str:
            """message

            Returns:
                str:
            """

    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def compression(self) -> bool:
        """compression

        Returns:
            bool:
        """

    @compression.setter
    def compression(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def parameters(self) -> FixstarsClient.Parameters:
        """parameters

        Returns:
            amplify.FixstarsClient.Parameters:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> FixstarsClient.Result: ...
    @overload
    def solve(
        self, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> FixstarsClient.Result: ...
    @overload
    def solve(
        self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> FixstarsClient.Result: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.FixstarsClient.Result:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.FixstarsClient.Result:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.FixstarsClient.Result:

        .. admonition:: Overload 4.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 5.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 6.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
        """

class FujitsuDA3cClient(FujitsuDA4Client):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(
        self, token: str = "", type_id: str | None = None, url: str = "", proxy: str | None = None
    ) -> None: ...

class FujitsuDA4Client(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(
        self, token: str = "", type_id: str | None = None, url: str = "", proxy: str | None = None
    ) -> None: ...

    class Progress:
        @property
        def energy(self) -> float:
            """energy

            Returns:
                float:
            """

        @property
        def penalty_energy(self) -> float:
            """penalty_energy

            Returns:
                float:
            """

        @property
        def time(self) -> datetime.timedelta:
            """time

            Returns:
                datetime.timedelta:
            """

    class QuboSolution:
        @property
        def energy(self) -> float:
            """energy

            Returns:
                float:
            """

        @property
        def penalty_energy(self) -> float:
            """penalty_energy

            Returns:
                float:
            """

        @property
        def frequency(self) -> int:
            """frequency

            Returns:
                int:
            """

        @property
        def configuration(self) -> dict[str, bool]:
            """configuration

            Returns:
                dict[str, bool]:
            """

    class SolverTiming:
        @property
        def solve_time(self) -> datetime.timedelta:
            """solve_time

            Returns:
                datetime.timedelta:
            """

        @property
        def total_elapsed_time(self) -> datetime.timedelta:
            """total_elapsed_time

            Returns:
                datetime.timedelta:
            """

    class QuboSolutionList:
        @property
        def progress(self) -> list[FujitsuDA4Client.Progress]:
            """progress

            Returns:
                list[amplify.FujitsuDA4Client.Progress]:
            """

        @property
        def result_status(self) -> bool:
            """result_status

            Returns:
                bool:
            """

        @property
        def solutions(self) -> list[FujitsuDA4Client.QuboSolution]:
            """solutions

            Returns:
                list[amplify.FujitsuDA4Client.QuboSolution]:
            """

        @property
        def timing(self) -> FujitsuDA4Client.SolverTiming:
            """timing

            Returns:
                amplify.FujitsuDA4Client.SolverTiming:
            """

    class JobStatus(enum.Enum):
        Done = 0
        Running = 1
        Waiting = 2
        Canceled = 3
        Error = 4
        Deleted = 5

    class Result:
        @property
        def qubo_solution(self) -> FujitsuDA4Client.QuboSolutionList:
            """qubo_solution

            Returns:
                amplify.FujitsuDA4Client.QuboSolutionList:
            """

        @property
        def status(self) -> FujitsuDA4Client.JobStatus:
            """status

            Returns:
                amplify.FujitsuDA4Client.JobStatus:
            """

        @property
        def message(self) -> str:
            """message

            Returns:
                str:
            """

    QuboResponse: TypeAlias = FujitsuDA4Client.Result

    class CommonError:
        @property
        def code(self) -> int:
            """code

            Returns:
                int:
            """

        @property
        def title(self) -> str:
            """title

            Returns:
                str:
            """

        @property
        def message(self) -> str:
            """message

            Returns:
                str:
            """

    class JobStatusInfo:
        @property
        def job_id(self) -> str:
            """job_id

            Returns:
                str:
            """

        @property
        def job_status(self) -> FujitsuDA4Client.JobStatus:
            """job_status

            Returns:
                amplify.FujitsuDA4Client.JobStatus:
            """

        @property
        def start_time(self) -> datetime.datetime:
            """start_time

            Returns:
                datetime.datetime:
            """

    class Parameters:
        @property
        def time_limit_sec(self) -> datetime.timedelta | None:
            """time_limit_sec

            Returns:
                datetime.timedelta | None:
            """

        @time_limit_sec.setter
        def time_limit_sec(self, arg: int | datetime.timedelta | None) -> None:
            """
            Args:
                arg (int | datetime.timedelta | None)
            """

        @property
        def target_energy(self) -> float | None:
            """target_energy

            Returns:
                float | None:
            """

        @target_energy.setter
        def target_energy(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def num_run(self) -> int | None:
            """num_run

            Returns:
                int | None:
            """

        @num_run.setter
        def num_run(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def num_group(self) -> int | None:
            """num_group

            Returns:
                int | None:
            """

        @num_group.setter
        def num_group(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def num_output_solution(self) -> int | None:
            """num_output_solution

            Returns:
                int | None:
            """

        @num_output_solution.setter
        def num_output_solution(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def gs_level(self) -> int | None:
            """gs_level

            Returns:
                int | None:
            """

        @gs_level.setter
        def gs_level(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def gs_cutoff(self) -> int | None:
            """gs_cutoff

            Returns:
                int | None:
            """

        @gs_cutoff.setter
        def gs_cutoff(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def one_hot_level(self) -> int | None:
            """one_hot_level

            Returns:
                int | None:
            """

        @one_hot_level.setter
        def one_hot_level(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def one_hot_cutoff(self) -> int | None:
            """one_hot_cutoff

            Returns:
                int | None:
            """

        @one_hot_cutoff.setter
        def one_hot_cutoff(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def internal_penalty(self) -> int | None:
            """internal_penalty

            Returns:
                int | None:
            """

        @internal_penalty.setter
        def internal_penalty(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def penalty_auto_mode(self) -> int | None:
            """penalty_auto_mode

            Returns:
                int | None:
            """

        @penalty_auto_mode.setter
        def penalty_auto_mode(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def penalty_coef(self) -> int | None:
            """penalty_coef

            Returns:
                int | None:
            """

        @penalty_coef.setter
        def penalty_coef(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def penalty_inc_rate(self) -> int | None:
            """penalty_inc_rate

            Returns:
                int | None:
            """

        @penalty_inc_rate.setter
        def penalty_inc_rate(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def max_penalty_coef(self) -> int | None:
            """max_penalty_coef

            Returns:
                int | None:
            """

        @max_penalty_coef.setter
        def max_penalty_coef(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def type_id(self) -> str | None:
        """type_id

        Returns:
            str | None:
        """

    @type_id.setter
    def type_id(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def set_penalty_binary_polynomial(self) -> bool:
        """set_penalty_binary_polynomial

        Returns:
            bool:
        """

    @set_penalty_binary_polynomial.setter
    def set_penalty_binary_polynomial(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_one_way_one_hot_groups(self) -> bool:
        """set_one_way_one_hot_groups

        Returns:
            bool:
        """

    @set_one_way_one_hot_groups.setter
    def set_one_way_one_hot_groups(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_two_way_one_hot_groups(self) -> bool:
        """set_two_way_one_hot_groups

        Returns:
            bool:
        """

    @set_two_way_one_hot_groups.setter
    def set_two_way_one_hot_groups(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_inequalities(self) -> bool:
        """set_inequalities

        Returns:
            bool:
        """

    @set_inequalities.setter
    def set_inequalities(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def parameters(self) -> FujitsuDA4Client.Parameters:
        """parameters

        Returns:
            amplify.FujitsuDA4Client.Parameters:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    def health_check(self) -> FujitsuDA4Client.CommonError | None:
        """health_check

        Returns:
            amplify.FujitsuDA4Client.CommonError | None:
        """

    def get_jobs(self) -> list[FujitsuDA4Client.JobStatusInfo]:
        """get_jobs

        Returns:
            list[amplify.FujitsuDA4Client.JobStatusInfo]:
        """

    def get_job_result(self, job_id: str) -> FujitsuDA4Client.Result:
        """get_job_result

        Args:
            job_id (str)

        Returns:
            amplify.FujitsuDA4Client.Result:
        """

    def delete_job_result(self, job_id: str) -> FujitsuDA4Client.Result:
        """delete_job_result

        Args:
            job_id (str)

        Returns:
            amplify.FujitsuDA4Client.Result:
        """

    def cancel_job(self, job_id: str) -> FujitsuDA4Client.JobStatus:
        """cancel_job

        Args:
            job_id (str)

        Returns:
            amplify.FujitsuDA4Client.JobStatus:
        """

    @overload
    def async_solve(self, objective: Poly) -> str: ...
    @overload
    def async_solve(self, constraint: Constraint | ConstraintList) -> str: ...
    @overload
    def async_solve(self, objective: Poly, constraint: Constraint | ConstraintList) -> str:
        """async_solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly)

            Returns:
                str:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)

            Returns:
                str:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly)
                * constraint (amplify.Constraint | amplify.ConstraintList)

            Returns:
                str:
        """

    @overload
    def solve(
        self,
        objective: Poly | Matrix,
        dry_run: Literal[False] = False,
        retry_after_delete: bool = True,
        delete_after_solve: bool = True,
    ) -> FujitsuDA4Client.Result: ...
    @overload
    def solve(
        self,
        constraint: Constraint | ConstraintList,
        dry_run: Literal[False] = False,
        retry_after_delete: bool = True,
        delete_after_solve: bool = True,
    ) -> FujitsuDA4Client.Result: ...
    @overload
    def solve(
        self,
        objective: Poly | Matrix,
        constraint: Constraint | ConstraintList,
        dry_run: Literal[False] = False,
        retry_after_delete: bool = True,
        delete_after_solve: bool = True,
    ) -> FujitsuDA4Client.Result: ...
    @overload
    def solve(
        self,
        objective: Poly | Matrix,
        dry_run: Literal[True],
        retry_after_delete: bool = True,
        delete_after_solve: bool = True,
    ) -> None: ...
    @overload
    def solve(
        self,
        constraint: Constraint | ConstraintList,
        dry_run: Literal[True],
        retry_after_delete: bool = True,
        delete_after_solve: bool = True,
    ) -> None: ...
    @overload
    def solve(
        self,
        objective: Poly | Matrix,
        constraint: Constraint | ConstraintList,
        dry_run: Literal[True],
        retry_after_delete: bool = True,
        delete_after_solve: bool = True,
    ) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.
                * retry_after_delete (bool): Defaults to ``True``.
                * delete_after_solve (bool): Defaults to ``True``.

            Returns:
                amplify.FujitsuDA4Client.Result:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.
                * retry_after_delete (bool): Defaults to ``True``.
                * delete_after_solve (bool): Defaults to ``True``.

            Returns:
                amplify.FujitsuDA4Client.Result:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.
                * retry_after_delete (bool): Defaults to ``True``.
                * delete_after_solve (bool): Defaults to ``True``.

            Returns:
                amplify.FujitsuDA4Client.Result:

        .. admonition:: Overload 4.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])
                * retry_after_delete (bool): Defaults to ``True``.
                * delete_after_solve (bool): Defaults to ``True``.

        .. admonition:: Overload 5.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
                * retry_after_delete (bool): Defaults to ``True``.
                * delete_after_solve (bool): Defaults to ``True``.

        .. admonition:: Overload 6.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
                * retry_after_delete (bool): Defaults to ``True``.
                * delete_after_solve (bool): Defaults to ``True``.
        """

class Graph:
    @property
    def type(self) -> str:
        """type

        Returns:
            str:
        """

    @property
    def shape(self) -> list[int]:
        """shape

        Returns:
            list[int]:
        """

    @property
    def nodes(self) -> list[int]:
        """nodes

        Returns:
            list[int]:
        """

    @property
    def edges(self) -> list[tuple[int, int]]:
        """edges

        Returns:
            list[tuple[int, int]]:
        """

    @property
    def adjacency(self) -> list[list[int]]:
        """adjacency

        Returns:
            list[list[int]]:
        """

class GurobiClient(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, library_path: str) -> None: ...

    class Parameters:
        @property
        def log_to_console(self) -> bool | None:
            """log_to_console

            Returns:
                bool | None:
            """

        @log_to_console.setter
        def log_to_console(self, arg: bool | None) -> None:
            """
            Args:
                arg (bool | None)
            """

        @property
        def agg_fill(self) -> int | None:
            """agg_fill

            Returns:
                int | None:
            """

        @agg_fill.setter
        def agg_fill(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def aggregate(self) -> int | None:
            """aggregate

            Returns:
                int | None:
            """

        @aggregate.setter
        def aggregate(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def bar_conv_tol(self) -> float | None:
            """bar_conv_tol

            Returns:
                float | None:
            """

        @bar_conv_tol.setter
        def bar_conv_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def bar_correctors(self) -> int | None:
            """bar_correctors

            Returns:
                int | None:
            """

        @bar_correctors.setter
        def bar_correctors(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def bar_homogeneous(self) -> int | None:
            """bar_homogeneous

            Returns:
                int | None:
            """

        @bar_homogeneous.setter
        def bar_homogeneous(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def bar_iter_limit(self) -> int | None:
            """bar_iter_limit

            Returns:
                int | None:
            """

        @bar_iter_limit.setter
        def bar_iter_limit(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def bar_order(self) -> int | None:
            """bar_order

            Returns:
                int | None:
            """

        @bar_order.setter
        def bar_order(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def bar_qcp_conv_tol(self) -> float | None:
            """bar_qcp_conv_tol

            Returns:
                float | None:
            """

        @bar_qcp_conv_tol.setter
        def bar_qcp_conv_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def best_bd_stop(self) -> float | None:
            """best_bd_stop

            Returns:
                float | None:
            """

        @best_bd_stop.setter
        def best_bd_stop(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def best_obj_stop(self) -> float | None:
            """best_obj_stop

            Returns:
                float | None:
            """

        @best_obj_stop.setter
        def best_obj_stop(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def bqp_cuts(self) -> int | None:
            """bqp_cuts

            Returns:
                int | None:
            """

        @bqp_cuts.setter
        def bqp_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def branch_dir(self) -> int | None:
            """branch_dir

            Returns:
                int | None:
            """

        @branch_dir.setter
        def branch_dir(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def clique_cuts(self) -> int | None:
            """clique_cuts

            Returns:
                int | None:
            """

        @clique_cuts.setter
        def clique_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cloud_access_id(self) -> str | None:
            """cloud_access_id

            Returns:
                str | None:
            """

        @cloud_access_id.setter
        def cloud_access_id(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cloud_host(self) -> str | None:
            """cloud_host

            Returns:
                str | None:
            """

        @cloud_host.setter
        def cloud_host(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cloud_pool(self) -> str | None:
            """cloud_pool

            Returns:
                str | None:
            """

        @cloud_pool.setter
        def cloud_pool(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cloud_secret_key(self) -> str | None:
            """cloud_secret_key

            Returns:
                str | None:
            """

        @cloud_secret_key.setter
        def cloud_secret_key(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def compute_server(self) -> str | None:
            """compute_server

            Returns:
                str | None:
            """

        @compute_server.setter
        def compute_server(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def concurrent_jobs(self) -> int | None:
            """concurrent_jobs

            Returns:
                int | None:
            """

        @concurrent_jobs.setter
        def concurrent_jobs(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def concurrent_method(self) -> int | None:
            """concurrent_method

            Returns:
                int | None:
            """

        @concurrent_method.setter
        def concurrent_method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def concurrent_mip(self) -> int | None:
            """concurrent_mip

            Returns:
                int | None:
            """

        @concurrent_mip.setter
        def concurrent_mip(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cover_cuts(self) -> int | None:
            """cover_cuts

            Returns:
                int | None:
            """

        @cover_cuts.setter
        def cover_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def crossover(self) -> int | None:
            """crossover

            Returns:
                int | None:
            """

        @crossover.setter
        def crossover(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def crossover_basis(self) -> int | None:
            """crossover_basis

            Returns:
                int | None:
            """

        @crossover_basis.setter
        def crossover_basis(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cs_app_name(self) -> str | None:
            """cs_app_name

            Returns:
                str | None:
            """

        @cs_app_name.setter
        def cs_app_name(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cs_auth_token(self) -> str | None:
            """cs_auth_token

            Returns:
                str | None:
            """

        @cs_auth_token.setter
        def cs_auth_token(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cs_batch_mode(self) -> int | None:
            """cs_batch_mode

            Returns:
                int | None:
            """

        @cs_batch_mode.setter
        def cs_batch_mode(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cs_client_log(self) -> int | None:
            """cs_client_log

            Returns:
                int | None:
            """

        @cs_client_log.setter
        def cs_client_log(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cs_group(self) -> str | None:
            """cs_group

            Returns:
                str | None:
            """

        @cs_group.setter
        def cs_group(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cs_idle_timeout(self) -> int | None:
            """cs_idle_timeout

            Returns:
                int | None:
            """

        @cs_idle_timeout.setter
        def cs_idle_timeout(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cs_manager(self) -> str | None:
            """cs_manager

            Returns:
                str | None:
            """

        @cs_manager.setter
        def cs_manager(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cs_priority(self) -> int | None:
            """cs_priority

            Returns:
                int | None:
            """

        @cs_priority.setter
        def cs_priority(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cs_queue_timeout(self) -> float | None:
            """cs_queue_timeout

            Returns:
                float | None:
            """

        @cs_queue_timeout.setter
        def cs_queue_timeout(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def cs_router(self) -> str | None:
            """cs_router

            Returns:
                str | None:
            """

        @cs_router.setter
        def cs_router(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def csapi_access_id(self) -> str | None:
            """csapi_access_id

            Returns:
                str | None:
            """

        @csapi_access_id.setter
        def csapi_access_id(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def csapi_secret(self) -> str | None:
            """csapi_secret

            Returns:
                str | None:
            """

        @csapi_secret.setter
        def csapi_secret(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def cstls_insecure(self) -> int | None:
            """cstls_insecure

            Returns:
                int | None:
            """

        @cstls_insecure.setter
        def cstls_insecure(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cut_agg_passes(self) -> int | None:
            """cut_agg_passes

            Returns:
                int | None:
            """

        @cut_agg_passes.setter
        def cut_agg_passes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cut_passes(self) -> int | None:
            """cut_passes

            Returns:
                int | None:
            """

        @cut_passes.setter
        def cut_passes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def cutoff(self) -> float | None:
            """cutoff

            Returns:
                float | None:
            """

        @cutoff.setter
        def cutoff(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def cuts(self) -> int | None:
            """cuts

            Returns:
                int | None:
            """

        @cuts.setter
        def cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def degen_moves(self) -> int | None:
            """degen_moves

            Returns:
                int | None:
            """

        @degen_moves.setter
        def degen_moves(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def disconnected(self) -> int | None:
            """disconnected

            Returns:
                int | None:
            """

        @disconnected.setter
        def disconnected(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def display_interval(self) -> int | None:
            """display_interval

            Returns:
                int | None:
            """

        @display_interval.setter
        def display_interval(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def distributed_mip_jobs(self) -> int | None:
            """distributed_mip_jobs

            Returns:
                int | None:
            """

        @distributed_mip_jobs.setter
        def distributed_mip_jobs(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def dual_reductions(self) -> int | None:
            """dual_reductions

            Returns:
                int | None:
            """

        @dual_reductions.setter
        def dual_reductions(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def feas_relax_big_m(self) -> float | None:
            """feas_relax_big_m

            Returns:
                float | None:
            """

        @feas_relax_big_m.setter
        def feas_relax_big_m(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def feasibility_tol(self) -> float | None:
            """feasibility_tol

            Returns:
                float | None:
            """

        @feasibility_tol.setter
        def feasibility_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def flow_cover_cuts(self) -> int | None:
            """flow_cover_cuts

            Returns:
                int | None:
            """

        @flow_cover_cuts.setter
        def flow_cover_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def flow_path_cuts(self) -> int | None:
            """flow_path_cuts

            Returns:
                int | None:
            """

        @flow_path_cuts.setter
        def flow_path_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def func_max_val(self) -> float | None:
            """func_max_val

            Returns:
                float | None:
            """

        @func_max_val.setter
        def func_max_val(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def func_nonlinear(self) -> int | None:
            """func_nonlinear

            Returns:
                int | None:
            """

        @func_nonlinear.setter
        def func_nonlinear(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def func_piece_error(self) -> float | None:
            """func_piece_error

            Returns:
                float | None:
            """

        @func_piece_error.setter
        def func_piece_error(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def func_piece_length(self) -> float | None:
            """func_piece_length

            Returns:
                float | None:
            """

        @func_piece_length.setter
        def func_piece_length(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def func_piece_ratio(self) -> float | None:
            """func_piece_ratio

            Returns:
                float | None:
            """

        @func_piece_ratio.setter
        def func_piece_ratio(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def func_pieces(self) -> int | None:
            """func_pieces

            Returns:
                int | None:
            """

        @func_pieces.setter
        def func_pieces(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def gomory_passes(self) -> int | None:
            """gomory_passes

            Returns:
                int | None:
            """

        @gomory_passes.setter
        def gomory_passes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def gub_cover_cuts(self) -> int | None:
            """gub_cover_cuts

            Returns:
                int | None:
            """

        @gub_cover_cuts.setter
        def gub_cover_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def heuristics(self) -> float | None:
            """heuristics

            Returns:
                float | None:
            """

        @heuristics.setter
        def heuristics(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def ignore_names(self) -> int | None:
            """ignore_names

            Returns:
                int | None:
            """

        @ignore_names.setter
        def ignore_names(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def iis_method(self) -> int | None:
            """iis_method

            Returns:
                int | None:
            """

        @iis_method.setter
        def iis_method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def implied_cuts(self) -> int | None:
            """implied_cuts

            Returns:
                int | None:
            """

        @implied_cuts.setter
        def implied_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def improve_start_gap(self) -> float | None:
            """improve_start_gap

            Returns:
                float | None:
            """

        @improve_start_gap.setter
        def improve_start_gap(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def improve_start_nodes(self) -> float | None:
            """improve_start_nodes

            Returns:
                float | None:
            """

        @improve_start_nodes.setter
        def improve_start_nodes(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def improve_start_time(self) -> float | None:
            """improve_start_time

            Returns:
                float | None:
            """

        @improve_start_time.setter
        def improve_start_time(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def inf_proof_cuts(self) -> int | None:
            """inf_proof_cuts

            Returns:
                int | None:
            """

        @inf_proof_cuts.setter
        def inf_proof_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def inf_unbd_info(self) -> int | None:
            """inf_unbd_info

            Returns:
                int | None:
            """

        @inf_unbd_info.setter
        def inf_unbd_info(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def int_feas_tol(self) -> float | None:
            """int_feas_tol

            Returns:
                float | None:
            """

        @int_feas_tol.setter
        def int_feas_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def integrality_focus(self) -> int | None:
            """integrality_focus

            Returns:
                int | None:
            """

        @integrality_focus.setter
        def integrality_focus(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def iteration_limit(self) -> float | None:
            """iteration_limit

            Returns:
                float | None:
            """

        @iteration_limit.setter
        def iteration_limit(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def job_id(self) -> str | None:
            """job_id

            Returns:
                str | None:
            """

        @job_id.setter
        def job_id(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def json_sol_detail(self) -> int | None:
            """json_sol_detail

            Returns:
                int | None:
            """

        @json_sol_detail.setter
        def json_sol_detail(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def lazy_constraints(self) -> int | None:
            """lazy_constraints

            Returns:
                int | None:
            """

        @lazy_constraints.setter
        def lazy_constraints(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def license_id(self) -> int | None:
            """license_id

            Returns:
                int | None:
            """

        @license_id.setter
        def license_id(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def lift_project_cuts(self) -> int | None:
            """lift_project_cuts

            Returns:
                int | None:
            """

        @lift_project_cuts.setter
        def lift_project_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def log_file(self) -> pathlib.Path | None:
            """log_file

            Returns:
                pathlib.Path | None:
            """

        @log_file.setter
        def log_file(self, arg: str | os.PathLike | None) -> None:
            """
            Args:
                arg (str | os.PathLike | None)
            """

        @property
        def lp_warm_start(self) -> int | None:
            """lp_warm_start

            Returns:
                int | None:
            """

        @lp_warm_start.setter
        def lp_warm_start(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def markowitz_tol(self) -> float | None:
            """markowitz_tol

            Returns:
                float | None:
            """

        @markowitz_tol.setter
        def markowitz_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def mem_limit(self) -> float | None:
            """mem_limit

            Returns:
                float | None:
            """

        @mem_limit.setter
        def mem_limit(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def method(self) -> int | None:
            """method

            Returns:
                int | None:
            """

        @method.setter
        def method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def min_rel_nodes(self) -> int | None:
            """min_rel_nodes

            Returns:
                int | None:
            """

        @min_rel_nodes.setter
        def min_rel_nodes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def mip_focus(self) -> int | None:
            """mip_focus

            Returns:
                int | None:
            """

        @mip_focus.setter
        def mip_focus(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def mip_gap(self) -> float | None:
            """mip_gap

            Returns:
                float | None:
            """

        @mip_gap.setter
        def mip_gap(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def mip_gap_abs(self) -> float | None:
            """mip_gap_abs

            Returns:
                float | None:
            """

        @mip_gap_abs.setter
        def mip_gap_abs(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def mip_sep_cuts(self) -> int | None:
            """mip_sep_cuts

            Returns:
                int | None:
            """

        @mip_sep_cuts.setter
        def mip_sep_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def miqcp_method(self) -> int | None:
            """miqcp_method

            Returns:
                int | None:
            """

        @miqcp_method.setter
        def miqcp_method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def mir_cuts(self) -> int | None:
            """mir_cuts

            Returns:
                int | None:
            """

        @mir_cuts.setter
        def mir_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def mixing_cuts(self) -> int | None:
            """mixing_cuts

            Returns:
                int | None:
            """

        @mixing_cuts.setter
        def mixing_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def mod_k_cuts(self) -> int | None:
            """mod_k_cuts

            Returns:
                int | None:
            """

        @mod_k_cuts.setter
        def mod_k_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def multi_obj_method(self) -> int | None:
            """multi_obj_method

            Returns:
                int | None:
            """

        @multi_obj_method.setter
        def multi_obj_method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def multi_obj_pre(self) -> int | None:
            """multi_obj_pre

            Returns:
                int | None:
            """

        @multi_obj_pre.setter
        def multi_obj_pre(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def network_alg(self) -> int | None:
            """network_alg

            Returns:
                int | None:
            """

        @network_alg.setter
        def network_alg(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def network_cuts(self) -> int | None:
            """network_cuts

            Returns:
                int | None:
            """

        @network_cuts.setter
        def network_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def nlp_heur(self) -> int | None:
            """nlp_heur

            Returns:
                int | None:
            """

        @nlp_heur.setter
        def nlp_heur(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def no_rel_heur_time(self) -> float | None:
            """no_rel_heur_time

            Returns:
                float | None:
            """

        @no_rel_heur_time.setter
        def no_rel_heur_time(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def no_rel_heur_work(self) -> float | None:
            """no_rel_heur_work

            Returns:
                float | None:
            """

        @no_rel_heur_work.setter
        def no_rel_heur_work(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def no_rel_heuristic(self) -> int | None:
            """no_rel_heuristic

            Returns:
                int | None:
            """

        @no_rel_heuristic.setter
        def no_rel_heuristic(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def node_limit(self) -> float | None:
            """node_limit

            Returns:
                float | None:
            """

        @node_limit.setter
        def node_limit(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def node_method(self) -> int | None:
            """node_method

            Returns:
                int | None:
            """

        @node_method.setter
        def node_method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def nodefile_dir(self) -> pathlib.Path | None:
            """nodefile_dir

            Returns:
                pathlib.Path | None:
            """

        @nodefile_dir.setter
        def nodefile_dir(self, arg: str | os.PathLike | None) -> None:
            """
            Args:
                arg (str | os.PathLike | None)
            """

        @property
        def nodefile_start(self) -> float | None:
            """nodefile_start

            Returns:
                float | None:
            """

        @nodefile_start.setter
        def nodefile_start(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def non_convex(self) -> int | None:
            """non_convex

            Returns:
                int | None:
            """

        @non_convex.setter
        def non_convex(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def norm_adjust(self) -> int | None:
            """norm_adjust

            Returns:
                int | None:
            """

        @norm_adjust.setter
        def norm_adjust(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def numeric_focus(self) -> int | None:
            """numeric_focus

            Returns:
                int | None:
            """

        @numeric_focus.setter
        def numeric_focus(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def obbt(self) -> int | None:
            """obbt

            Returns:
                int | None:
            """

        @obbt.setter
        def obbt(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def obj_number(self) -> int | None:
            """obj_number

            Returns:
                int | None:
            """

        @obj_number.setter
        def obj_number(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def obj_scale(self) -> float | None:
            """obj_scale

            Returns:
                float | None:
            """

        @obj_scale.setter
        def obj_scale(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def optimality_tol(self) -> float | None:
            """optimality_tol

            Returns:
                float | None:
            """

        @optimality_tol.setter
        def optimality_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def output_flag(self) -> int | None:
            """output_flag

            Returns:
                int | None:
            """

        @output_flag.setter
        def output_flag(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def partition_place(self) -> int | None:
            """partition_place

            Returns:
                int | None:
            """

        @partition_place.setter
        def partition_place(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def perturb_value(self) -> float | None:
            """perturb_value

            Returns:
                float | None:
            """

        @perturb_value.setter
        def perturb_value(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def pool_gap(self) -> float | None:
            """pool_gap

            Returns:
                float | None:
            """

        @pool_gap.setter
        def pool_gap(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def pool_gap_abs(self) -> float | None:
            """pool_gap_abs

            Returns:
                float | None:
            """

        @pool_gap_abs.setter
        def pool_gap_abs(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def pool_search_mode(self) -> int | None:
            """pool_search_mode

            Returns:
                int | None:
            """

        @pool_search_mode.setter
        def pool_search_mode(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pool_solutions(self) -> int | None:
            """pool_solutions

            Returns:
                int | None:
            """

        @pool_solutions.setter
        def pool_solutions(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_crush(self) -> int | None:
            """pre_crush

            Returns:
                int | None:
            """

        @pre_crush.setter
        def pre_crush(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_dep_row(self) -> int | None:
            """pre_dep_row

            Returns:
                int | None:
            """

        @pre_dep_row.setter
        def pre_dep_row(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_dual(self) -> int | None:
            """pre_dual

            Returns:
                int | None:
            """

        @pre_dual.setter
        def pre_dual(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_miqcp_form(self) -> int | None:
            """pre_miqcp_form

            Returns:
                int | None:
            """

        @pre_miqcp_form.setter
        def pre_miqcp_form(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_passes(self) -> int | None:
            """pre_passes

            Returns:
                int | None:
            """

        @pre_passes.setter
        def pre_passes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_q_linearize(self) -> int | None:
            """pre_q_linearize

            Returns:
                int | None:
            """

        @pre_q_linearize.setter
        def pre_q_linearize(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def pre_sparsify(self) -> int | None:
            """pre_sparsify

            Returns:
                int | None:
            """

        @pre_sparsify.setter
        def pre_sparsify(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def presolve(self) -> int | None:
            """presolve

            Returns:
                int | None:
            """

        @presolve.setter
        def presolve(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def proj_implied_cuts(self) -> int | None:
            """proj_implied_cuts

            Returns:
                int | None:
            """

        @proj_implied_cuts.setter
        def proj_implied_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def psd_cuts(self) -> int | None:
            """psd_cuts

            Returns:
                int | None:
            """

        @psd_cuts.setter
        def psd_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def psd_tol(self) -> float | None:
            """psd_tol

            Returns:
                float | None:
            """

        @psd_tol.setter
        def psd_tol(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def pump_passes(self) -> int | None:
            """pump_passes

            Returns:
                int | None:
            """

        @pump_passes.setter
        def pump_passes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def qcp_dual(self) -> int | None:
            """qcp_dual

            Returns:
                int | None:
            """

        @qcp_dual.setter
        def qcp_dual(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def quad(self) -> int | None:
            """quad

            Returns:
                int | None:
            """

        @quad.setter
        def quad(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def record(self) -> int | None:
            """record

            Returns:
                int | None:
            """

        @record.setter
        def record(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def relax_lift_cuts(self) -> int | None:
            """relax_lift_cuts

            Returns:
                int | None:
            """

        @relax_lift_cuts.setter
        def relax_lift_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def result_file(self) -> pathlib.Path | None:
            """result_file

            Returns:
                pathlib.Path | None:
            """

        @result_file.setter
        def result_file(self, arg: str | os.PathLike | None) -> None:
            """
            Args:
                arg (str | os.PathLike | None)
            """

        @property
        def rins(self) -> int | None:
            """rins

            Returns:
                int | None:
            """

        @rins.setter
        def rins(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def rlt_cuts(self) -> int | None:
            """rlt_cuts

            Returns:
                int | None:
            """

        @rlt_cuts.setter
        def rlt_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def scale_flag(self) -> int | None:
            """scale_flag

            Returns:
                int | None:
            """

        @scale_flag.setter
        def scale_flag(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def scenario_number(self) -> int | None:
            """scenario_number

            Returns:
                int | None:
            """

        @scenario_number.setter
        def scenario_number(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def seed(self) -> int | None:
            """seed

            Returns:
                int | None:
            """

        @seed.setter
        def seed(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def server_password(self) -> str | None:
            """server_password

            Returns:
                str | None:
            """

        @server_password.setter
        def server_password(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def server_timeout(self) -> int | None:
            """server_timeout

            Returns:
                int | None:
            """

        @server_timeout.setter
        def server_timeout(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def sift_method(self) -> int | None:
            """sift_method

            Returns:
                int | None:
            """

        @sift_method.setter
        def sift_method(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def sifting(self) -> int | None:
            """sifting

            Returns:
                int | None:
            """

        @sifting.setter
        def sifting(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def simplex_pricing(self) -> int | None:
            """simplex_pricing

            Returns:
                int | None:
            """

        @simplex_pricing.setter
        def simplex_pricing(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def soft_mem_limit(self) -> float | None:
            """soft_mem_limit

            Returns:
                float | None:
            """

        @soft_mem_limit.setter
        def soft_mem_limit(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def sol_files(self) -> str | None:
            """sol_files

            Returns:
                str | None:
            """

        @sol_files.setter
        def sol_files(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def solution_limit(self) -> int | None:
            """solution_limit

            Returns:
                int | None:
            """

        @solution_limit.setter
        def solution_limit(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def solution_number(self) -> int | None:
            """solution_number

            Returns:
                int | None:
            """

        @solution_number.setter
        def solution_number(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def solution_target(self) -> int | None:
            """solution_target

            Returns:
                int | None:
            """

        @solution_target.setter
        def solution_target(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def start_node_limit(self) -> int | None:
            """start_node_limit

            Returns:
                int | None:
            """

        @start_node_limit.setter
        def start_node_limit(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def start_number(self) -> int | None:
            """start_number

            Returns:
                int | None:
            """

        @start_number.setter
        def start_number(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def strong_cg_cuts(self) -> int | None:
            """strong_cg_cuts

            Returns:
                int | None:
            """

        @strong_cg_cuts.setter
        def strong_cg_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def sub_mip_cuts(self) -> int | None:
            """sub_mip_cuts

            Returns:
                int | None:
            """

        @sub_mip_cuts.setter
        def sub_mip_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def sub_mip_nodes(self) -> int | None:
            """sub_mip_nodes

            Returns:
                int | None:
            """

        @sub_mip_nodes.setter
        def sub_mip_nodes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def symmetry(self) -> int | None:
            """symmetry

            Returns:
                int | None:
            """

        @symmetry.setter
        def symmetry(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def threads(self) -> int | None:
            """threads

            Returns:
                int | None:
            """

        @threads.setter
        def threads(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def time_limit(self) -> datetime.timedelta | None:
            """time_limit

            Returns:
                datetime.timedelta | None:
            """

        @time_limit.setter
        def time_limit(self, arg: float | datetime.timedelta | None) -> None:
            """
            Args:
                arg (float | datetime.timedelta | None)
            """

        @property
        def token_server(self) -> str | None:
            """token_server

            Returns:
                str | None:
            """

        @token_server.setter
        def token_server(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def ts_port(self) -> int | None:
            """ts_port

            Returns:
                int | None:
            """

        @ts_port.setter
        def ts_port(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_cleanup(self) -> float | None:
            """tune_cleanup

            Returns:
                float | None:
            """

        @tune_cleanup.setter
        def tune_cleanup(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def tune_criterion(self) -> int | None:
            """tune_criterion

            Returns:
                int | None:
            """

        @tune_criterion.setter
        def tune_criterion(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_dynamic_jobs(self) -> int | None:
            """tune_dynamic_jobs

            Returns:
                int | None:
            """

        @tune_dynamic_jobs.setter
        def tune_dynamic_jobs(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_jobs(self) -> int | None:
            """tune_jobs

            Returns:
                int | None:
            """

        @tune_jobs.setter
        def tune_jobs(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_metric(self) -> int | None:
            """tune_metric

            Returns:
                int | None:
            """

        @tune_metric.setter
        def tune_metric(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_output(self) -> int | None:
            """tune_output

            Returns:
                int | None:
            """

        @tune_output.setter
        def tune_output(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_results(self) -> int | None:
            """tune_results

            Returns:
                int | None:
            """

        @tune_results.setter
        def tune_results(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def tune_target_mip_gap(self) -> float | None:
            """tune_target_mip_gap

            Returns:
                float | None:
            """

        @tune_target_mip_gap.setter
        def tune_target_mip_gap(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def tune_target_time(self) -> float | None:
            """tune_target_time

            Returns:
                float | None:
            """

        @tune_target_time.setter
        def tune_target_time(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def tune_time_limit(self) -> datetime.timedelta | None:
            """tune_time_limit

            Returns:
                datetime.timedelta | None:
            """

        @tune_time_limit.setter
        def tune_time_limit(self, arg: float | datetime.timedelta | None) -> None:
            """
            Args:
                arg (float | datetime.timedelta | None)
            """

        @property
        def tune_trials(self) -> int | None:
            """tune_trials

            Returns:
                int | None:
            """

        @tune_trials.setter
        def tune_trials(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def update_mode(self) -> int | None:
            """update_mode

            Returns:
                int | None:
            """

        @update_mode.setter
        def update_mode(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def username(self) -> str | None:
            """username

            Returns:
                str | None:
            """

        @username.setter
        def username(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def var_branch(self) -> int | None:
            """var_branch

            Returns:
                int | None:
            """

        @var_branch.setter
        def var_branch(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def wls_access_id(self) -> str | None:
            """wls_access_id

            Returns:
                str | None:
            """

        @wls_access_id.setter
        def wls_access_id(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def wls_secret(self) -> str | None:
            """wls_secret

            Returns:
                str | None:
            """

        @wls_secret.setter
        def wls_secret(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def wls_token(self) -> str | None:
            """wls_token

            Returns:
                str | None:
            """

        @wls_token.setter
        def wls_token(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def wls_token_duration(self) -> int | None:
            """wls_token_duration

            Returns:
                int | None:
            """

        @wls_token_duration.setter
        def wls_token_duration(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def wls_token_refresh(self) -> float | None:
            """wls_token_refresh

            Returns:
                float | None:
            """

        @wls_token_refresh.setter
        def wls_token_refresh(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def work_limit(self) -> float | None:
            """work_limit

            Returns:
                float | None:
            """

        @work_limit.setter
        def work_limit(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def worker_password(self) -> str | None:
            """worker_password

            Returns:
                str | None:
            """

        @worker_password.setter
        def worker_password(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def worker_pool(self) -> str | None:
            """worker_pool

            Returns:
                str | None:
            """

        @worker_pool.setter
        def worker_pool(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def zero_half_cuts(self) -> int | None:
            """zero_half_cuts

            Returns:
                int | None:
            """

        @zero_half_cuts.setter
        def zero_half_cuts(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def zero_obj_nodes(self) -> int | None:
            """zero_obj_nodes

            Returns:
                int | None:
            """

        @zero_obj_nodes.setter
        def zero_obj_nodes(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

    class Result:
        class OptimizationStatusCode(enum.Enum):
            Unknown = 0
            Loaded = 1
            Optimal = 2
            Infeasible = 3
            InfOrUnbd = 4
            Unbounded = 5
            Cutoff = 6
            IterationLimit = 7
            NodeLimit = 8
            TimeLimit = 9
            SolutionLimit = 10
            Interrupted = 11
            Numeric = 12
            Suboptimal = 13
            Inprogress = 14
            UserObjLimit = 15
            WorkLimit = 16
            MemLimit = 17

        @property
        def status(self) -> GurobiClient.Result.OptimizationStatusCode:
            """status

            Returns:
                amplify.GurobiClient.Result.OptimizationStatusCode:
            """

        @property
        def runtime(self) -> datetime.timedelta:
            """runtime

            Returns:
                datetime.timedelta:
            """

        @property
        def obj_val(self) -> float | None:
            """obj_val

            Returns:
                float | None:
            """

        @property
        def obj_bound(self) -> float | None:
            """obj_bound

            Returns:
                float | None:
            """

        @property
        def obj_bound_c(self) -> float | None:
            """obj_bound_c

            Returns:
                float | None:
            """

        @property
        def mip_gap(self) -> float | None:
            """mip_gap

            Returns:
                float | None:
            """

        @property
        def int_vio(self) -> float | None:
            """int_vio

            Returns:
                float | None:
            """

        @property
        def bound_vio(self) -> float | None:
            """bound_vio

            Returns:
                float | None:
            """

        @property
        def constr_vio(self) -> float | None:
            """constr_vio

            Returns:
                float | None:
            """

        @property
        def iter_count(self) -> float | None:
            """iter_count

            Returns:
                float | None:
            """

        @property
        def bar_iter_count(self) -> int | None:
            """bar_iter_count

            Returns:
                int | None:
            """

        @property
        def node_count(self) -> float | None:
            """node_count

            Returns:
                float | None:
            """

        @property
        def farkas_proof(self) -> float | None:
            """farkas_proof

            Returns:
                float | None:
            """

        @property
        def sol_count(self) -> int | None:
            """sol_count

            Returns:
                int | None:
            """

        @property
        def pool_obj_bound(self) -> float | None:
            """pool_obj_bound

            Returns:
                float | None:
            """

        @property
        def pool_obj_val(
            self,
        ) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64", "writable": False}] | None:
            """pool_obj_val

            Returns:
                numpy.ndarray:
            """

        @property
        def var_names(self) -> list[str] | None:
            """var_names

            Returns:
                list[str] | None:
            """

        @property
        def x(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64", "writable": False}] | None:
            """x

            Returns:
                numpy.ndarray:
            """

        @property
        def xn(
            self,
        ) -> list[Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64", "writable": False}]] | None:
            """xn

            Returns:
                list[numpy.ndarray[dtype=float64, writable=False]] | None:
            """

        @property
        def time_stamps(self) -> list[tuple[float, datetime.timedelta]]:
            """time_stamps

            Returns:
                list[tuple[float, datetime.timedelta]]:
            """

    @property
    def library_path(self) -> str | None:
        """library_path

        Returns:
            str | None:
        """

    @library_path.setter
    def library_path(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def write_request_data(self) -> str | None:
        """write_request_data

        Returns:
            str | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def write_response_data(self) -> str | None:
        """write_response_data

        Returns:
            str | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def parameters(self) -> GurobiClient.Parameters:
        """parameters

        Returns:
            amplify.GurobiClient.Parameters:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> GurobiClient.Result: ...
    @overload
    def solve(
        self, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> GurobiClient.Result: ...
    @overload
    def solve(
        self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> GurobiClient.Result: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.GurobiClient.Result:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.GurobiClient.Result:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.GurobiClient.Result:

        .. admonition:: Overload 4.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 5.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 6.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
        """

    @overload
    def tune(self, objective: Poly | Matrix) -> None: ...
    @overload
    def tune(self, constraint: Constraint | ConstraintList) -> None: ...
    @overload
    def tune(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList) -> None:
        """tune

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
        """

class HitachiClient(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", url: str = "", proxy: str | None = None) -> None: ...

    class Parameters:
        class Outputs:
            @property
            def energies(self) -> bool | None:
                """energies

                Returns:
                    bool | None:
                """

            @energies.setter
            def energies(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def spins(self) -> bool | None:
                """spins

                Returns:
                    bool | None:
                """

            @spins.setter
            def spins(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def execution_time(self) -> bool | None:
                """execution_time

                Returns:
                    bool | None:
                """

            @execution_time.setter
            def execution_time(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def num_outputs(self) -> int | None:
                """num_outputs

                Returns:
                    int | None:
                """

            @num_outputs.setter
            def num_outputs(self, arg: int | None) -> None:
                """
                Args:
                    arg (int | None)
                """

            @property
            def averaged_spins(self) -> bool | None:
                """averaged_spins

                Returns:
                    bool | None:
                """

            @averaged_spins.setter
            def averaged_spins(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

            @property
            def averaged_energy(self) -> bool | None:
                """averaged_energy

                Returns:
                    bool | None:
                """

            @averaged_energy.setter
            def averaged_energy(self, arg: bool | None) -> None:
                """
                Args:
                    arg (bool | None)
                """

        class TemperatureParameters:
            @property
            def temperature_num_steps(self) -> int | None:
                """temperature_num_steps

                Returns:
                    int | None:
                """

            @temperature_num_steps.setter
            def temperature_num_steps(self, arg: int | None) -> None:
                """
                Args:
                    arg (int | None)
                """

            @property
            def temperature_step_length(self) -> int | None:
                """temperature_step_length

                Returns:
                    int | None:
                """

            @temperature_step_length.setter
            def temperature_step_length(self, arg: int | None) -> None:
                """
                Args:
                    arg (int | None)
                """

            @property
            def temperature_initial(self) -> float | None:
                """temperature_initial

                Returns:
                    float | None:
                """

            @temperature_initial.setter
            def temperature_initial(self, arg: float | None) -> None:
                """
                Args:
                    arg (float | None)
                """

            @property
            def temperature_target(self) -> float | None:
                """temperature_target

                Returns:
                    float | None:
                """

            @temperature_target.setter
            def temperature_target(self, arg: float | None) -> None:
                """
                Args:
                    arg (float | None)
                """

        @property
        def type(self) -> int:
            """type

            Returns:
                int:
            """

        @type.setter
        def type(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def num_executions(self) -> int | None:
            """num_executions

            Returns:
                int | None:
            """

        @num_executions.setter
        def num_executions(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def outputs(self) -> HitachiClient.Parameters.Outputs:
            """outputs

            Returns:
                amplify.HitachiClient.Parameters.Outputs:
            """

        @property
        def parameters(self) -> HitachiClient.Parameters.TemperatureParameters:
            """parameters

            Returns:
                amplify.HitachiClient.Parameters.TemperatureParameters:
            """

    class Result:
        class Result:
            @property
            def execution_time(self) -> datetime.timedelta:
                """execution_time

                Returns:
                    datetime.timedelta:
                """

            @property
            def energies(
                self,
            ) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64", "writable": False}]:
                """energies

                Returns:
                    numpy.ndarray:
                """

            @property
            def spins(self) -> list[list[tuple[int, int, int]]]:
                """spins

                Returns:
                    list[list[tuple[int, int, int]]]:
                """

            @property
            def averaged_energy(self) -> float | None:
                """averaged_energy

                Returns:
                    float | None:
                """

            @property
            def averaged_spins(self) -> list[tuple[int, int, float]] | None:
                """averaged_spins

                Returns:
                    list[tuple[int, int, float]] | None:
                """

        @property
        def status(self) -> int:
            """status

            Returns:
                int:
            """

        @property
        def message(self) -> str | None:
            """message

            Returns:
                str | None:
            """

        @property
        def job_id(self) -> str:
            """job_id

            Returns:
                str:
            """

        @property
        def result(self) -> HitachiClient.Result.Result:
            """result

            Returns:
                amplify.HitachiClient.Result.Result:
            """

    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def parameters(self) -> HitachiClient.Parameters:
        """parameters

        Returns:
            amplify.HitachiClient.Parameters:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> HitachiClient.Result: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.HitachiClient.Result:

        .. admonition:: Overload 2.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])
        """

    @property
    def graph(self) -> Graph:
        """graph

        Returns:
            amplify.Graph:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

class IntegerEncodingMethod(enum.Enum):
    Unary = 0
    Linear = 1
    Binary = 2
    Default = 3

class LeapHybridCQMSamplerClient(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", solver: str = "", url: str = "", proxy: str | None = None) -> None: ...
    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def solver(self) -> str:
        """solver

        Returns:
            str:
        """

    @solver.setter
    def solver(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def compress_qpu_problem_data(self) -> bool | None:
        """compress_qpu_problem_data

        Returns:
            bool | None:
        """

    @compress_qpu_problem_data.setter
    def compress_qpu_problem_data(self, arg: bool | None) -> None:
        """
        Args:
            arg (bool | None)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def parameters(self) -> DWaveClientParameters:
        """parameters

        Returns:
            amplify.DWaveClientParameters:
        """

    @property
    def solver_names(self) -> list[str]:
        """solver_names

        Returns:
            list[str]:
        """

    @property
    def properties(self) -> dict[str, Any]:
        """properties

        Returns:
            dict[str, typing.Any]:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> dimod.sampleset.SampleSet: ...
    @overload
    def solve(
        self, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> dimod.sampleset.SampleSet: ...
    @overload
    def solve(
        self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> dimod.sampleset.SampleSet: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                dimod.sampleset.SampleSet:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                dimod.sampleset.SampleSet:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                dimod.sampleset.SampleSet:

        .. admonition:: Overload 4.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 5.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 6.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
        """

class LeapHybridSamplerClient(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", solver: str = "", url: str = "", proxy: str | None = None) -> None: ...
    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def solver(self) -> str:
        """solver

        Returns:
            str:
        """

    @solver.setter
    def solver(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def compress_qpu_problem_data(self) -> bool | None:
        """compress_qpu_problem_data

        Returns:
            bool | None:
        """

    @compress_qpu_problem_data.setter
    def compress_qpu_problem_data(self, arg: bool | None) -> None:
        """
        Args:
            arg (bool | None)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def parameters(self) -> DWaveClientParameters:
        """parameters

        Returns:
            amplify.DWaveClientParameters:
        """

    @property
    def solver_names(self) -> list[str]:
        """solver_names

        Returns:
            list[str]:
        """

    @property
    def properties(self) -> dict[str, Any]:
        """properties

        Returns:
            dict[str, typing.Any]:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> dimod.sampleset.SampleSet: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                dimod.sampleset.SampleSet:

        .. admonition:: Overload 2.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])
        """

class Matrix:
    def __eq__(self, arg: object, /) -> bool: ...
    def __ne__(self, arg: object, /) -> bool: ...
    @property
    def quadratic(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]:
        """quadratic

        Returns:
            numpy.ndarray:
        """

    @quadratic.setter
    def quadratic(self, arg: Annotated[NDArray, {"dtype": "float64"}], /) -> None:
        """
        Args:
            arg (ndarray)
        """

    @property
    def linear(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]:
        """linear

        Returns:
            numpy.ndarray:
        """

    @linear.setter
    def linear(self, arg: Annotated[NDArray, {"dtype": "float64"}], /) -> None:
        """
        Args:
            arg (ndarray)
        """

    @property
    def constant(self) -> float:
        """constant

        Returns:
            float:
        """

    @constant.setter
    def constant(self, arg: float, /) -> None:
        """
        Args:
            arg (float)
        """

    @property
    def variable_array(self) -> PolyArray[Dim]:
        """variable_array

        Returns:
            amplify.PolyArray[Dim]:
        """

    def to_poly(self) -> Poly:
        """to_poly

        Returns:
            amplify.Poly:
        """

    def __add__(self, arg: Constraint | ConstraintList, /) -> Model: ...
    def __radd__(self, arg: Constraint | ConstraintList, /) -> Model: ...
    @overload
    def evaluate(self, values: Values) -> float: ...
    @overload
    def evaluate(self, values: Values, default: float) -> float:
        """evaluate

        .. admonition:: Overload 1.

            Args:
                * values (amplify.Values)

            Returns:
                float:

        .. admonition:: Overload 2.

            Args:
                * values (amplify.Values)
                * default (float)

            Returns:
                float:
        """

class Model:
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, objective: Poly | Matrix) -> None: ...
    @overload
    def __init__(self, constraint: Constraint | ConstraintList) -> None: ...
    @overload
    def __init__(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList) -> None: ...
    def __eq__(self, arg: object, /) -> bool: ...
    def __ne__(self, arg: object, /) -> bool: ...
    def __iadd__(self, arg: Poly | Constraint | ConstraintList, /) -> Model: ...
    def __add__(self, arg: Poly | Constraint | ConstraintList, /) -> Model: ...
    def __radd__(self, arg: Poly | Constraint | ConstraintList, /) -> Model: ...
    def __isub__(self, arg: Poly, /) -> Model: ...
    def __sub__(self, arg: Poly, /) -> Model: ...
    @property
    def objective(self) -> Poly | Matrix:
        """objective

        Returns:
            amplify.Poly | amplify.Matrix:
        """

    @objective.setter
    def objective(self, arg: Poly | Matrix, /) -> None:
        """
        Args:
            arg (amplify.Poly | amplify.Matrix)
        """

    @property
    def constraints(self) -> ConstraintList:
        """constraints

        Returns:
            amplify.ConstraintList:
        """

    @constraints.setter
    def constraints(self, arg: Constraint | ConstraintList, /) -> None:
        """
        Args:
            arg (amplify.Constraint | amplify.ConstraintList)
        """

    @property
    def variables(self) -> list[Variable]:
        """alias of `~amplify.Model.get_variables`

        Returns:
            list[amplify.Variable]:
        """

    def get_variables(self, with_penalty: bool = False) -> list[Variable]:
        """get_variables

        Args:
            with_penalty (bool): Defaults to ``False``.

        Returns:
            list[amplify.Variable]:
        """

    def to_intermediate_model(
        self,
        acceptable_degrees: AcceptableDegrees,
        integer_encoding_method: Literal["Unary", "Linear", "Binary", "Default"]
        | IntegerEncodingMethod = IntegerEncodingMethod.Default,
        real_encoding_method: Literal["Random4", "Random8", "Random16", "Random32"]
        | RealEncodingMethod = RealEncodingMethod.Random16,
        quadratization_method: Literal["IshikawaKZFD", "Substitute"]
        | QuadratizationMethod = QuadratizationMethod.IshikawaKZFD,
        substitution_multiplier: float = 1.0,
    ) -> tuple[Model, Result.ModelConversion.IntermediateMapping]:
        """to_intermediate_model

        Args:
            acceptable_degrees (amplify.AcceptableDegrees)
            integer_encoding_method (typing.Literal['Unary', 'Linear', 'Binary', 'Default'] | amplify.IntegerEncodingMethod): Defaults to ``IntegerEncodingMethod.Default``.
            real_encoding_method (typing.Literal['Random4', 'Random8', 'Random16', 'Random32'] | amplify.RealEncodingMethod): Defaults to ``RealEncodingMethod.Random16``.
            quadratization_method (typing.Literal['IshikawaKZFD', 'Substitute'] | amplify.QuadratizationMethod): Defaults to ``QuadratizationMethod.IshikawaKZFD``.
            substitution_multiplier (float): Defaults to ``1.0``.

        Returns:
            tuple[amplify.Model, amplify.Result.ModelConversion.IntermediateMapping]:
        """

    def to_unconstrained_poly(self) -> Poly:
        """to_unconstrained_poly

        Returns:
            amplify.Poly:
        """

    def copy(self) -> Model:
        """copy

        Returns:
            amplify.Model:
        """

class NECVA2Client(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", url: str = "", proxy: str | None = None) -> None: ...

    class Parameters:
        @property
        def num_reads(self) -> int | None:
            """num_reads

            Returns:
                int | None:
            """

        @num_reads.setter
        def num_reads(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def num_results(self) -> int | None:
            """num_results

            Returns:
                int | None:
            """

        @num_results.setter
        def num_results(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def num_sweeps(self) -> int | None:
            """num_sweeps

            Returns:
                int | None:
            """

        @num_sweeps.setter
        def num_sweeps(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def beta_range(self) -> tuple[float, float, int] | None:
            """beta_range

            Returns:
                tuple[float, float, int] | None:
            """

        @beta_range.setter
        def beta_range(self, arg: tuple[float, float, int] | None) -> None:
            """
            Args:
                arg (tuple[float, float, int] | None)
            """

        @property
        def beta_list(self) -> list[float] | None:
            """beta_list

            Returns:
                list[float] | None:
            """

        @beta_list.setter
        def beta_list(self, arg: Sequence[float] | None) -> None:
            """
            Args:
                arg (collections.abc.Sequence[float] | None)
            """

        @property
        def dense(self) -> bool | None:
            """dense

            Returns:
                bool | None:
            """

        @dense.setter
        def dense(self, arg: bool | None) -> None:
            """
            Args:
                arg (bool | None)
            """

        @property
        def vector_mode(self) -> NECVA2Client.VectorMode | None:
            """vector_mode

            Returns:
                amplify.NECVA2Client.VectorMode | None:
            """

        @vector_mode.setter
        def vector_mode(self, arg: Literal["Accuracy", "Speed"] | NECVA2Client.VectorMode | None) -> None:
            """
            Args:
                arg (typing.Literal['Accuracy', 'Speed'] | amplify.NECVA2Client.VectorMode | None)
            """

        @property
        def timeout(self) -> datetime.timedelta | None:
            """timeout

            Returns:
                datetime.timedelta | None:
            """

        @timeout.setter
        def timeout(self, arg: int | datetime.timedelta | None) -> None:
            """
            Args:
                arg (int | datetime.timedelta | None)
            """

        @property
        def ve_num(self) -> int | None:
            """ve_num

            Returns:
                int | None:
            """

        @ve_num.setter
        def ve_num(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

    class Result:
        class Timing:
            @property
            def execution_time(self) -> datetime.timedelta:
                """execution_time

                Returns:
                    datetime.timedelta:
                """

            @property
            def solve_qubo_time(self) -> datetime.timedelta:
                """solve_qubo_time

                Returns:
                    datetime.timedelta:
                """

            @property
            def queue_time(self) -> datetime.timedelta:
                """queue_time

                Returns:
                    datetime.timedelta:
                """

        class Result:
            @property
            def constraint(self) -> bool:
                """constraint

                Returns:
                    bool:
                """

            @property
            def energy(self) -> float:
                """energy

                Returns:
                    float:
                """

            @property
            def memory_usage(self) -> float:
                """memory_usage

                Returns:
                    float:
                """

            @property
            def spin(self) -> dict[str, int]:
                """spin

                Returns:
                    dict[str, int]:
                """

            @property
            def time(self) -> datetime.timedelta:
                """time

                Returns:
                    datetime.timedelta:
                """

        @property
        def result(self) -> list[NECVA2Client.Result.Result]:
            """result

            Returns:
                list[amplify.NECVA2Client.Result.Result]:
            """

        @property
        def timing(self) -> NECVA2Client.Result.Timing:
            """timing

            Returns:
                amplify.NECVA2Client.Result.Timing:
            """

    class VectorMode(enum.Enum):
        Accuracy = 0
        Speed = 1

    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def set_onehot(self) -> bool:
        """set_onehot

        Returns:
            bool:
        """

    @set_onehot.setter
    def set_onehot(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_fixed(self) -> bool:
        """set_fixed

        Returns:
            bool:
        """

    @set_fixed.setter
    def set_fixed(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_andzero(self) -> bool:
        """set_andzero

        Returns:
            bool:
        """

    @set_andzero.setter
    def set_andzero(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_orone(self) -> bool:
        """set_orone

        Returns:
            bool:
        """

    @set_orone.setter
    def set_orone(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_supplement(self) -> bool:
        """set_supplement

        Returns:
            bool:
        """

    @set_supplement.setter
    def set_supplement(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_maxone(self) -> bool:
        """set_maxone

        Returns:
            bool:
        """

    @set_maxone.setter
    def set_maxone(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def set_minmaxone(self) -> bool:
        """set_minmaxone

        Returns:
            bool:
        """

    @set_minmaxone.setter
    def set_minmaxone(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def parameters(self) -> NECVA2Client.Parameters:
        """parameters

        Returns:
            amplify.NECVA2Client.Parameters:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> NECVA2Client.Result: ...
    @overload
    def solve(
        self, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> NECVA2Client.Result: ...
    @overload
    def solve(
        self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> NECVA2Client.Result: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.NECVA2Client.Result:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.NECVA2Client.Result:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.NECVA2Client.Result:

        .. admonition:: Overload 4.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 5.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 6.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
        """

class PenaltyFormulation(enum.Enum):
    Default = 0
    IntegerVariable = 1
    RealVariable = 2
    Relaxation = 3
    LinearRelaxation = 4
    QuadraticRelaxation = 5

class Poly:
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, arg: float, /) -> None: ...
    @overload
    def __init__(self, arg: Variable, /) -> None: ...
    @overload
    def __add__(self, arg: Poly | float, /) -> Poly: ...
    @overload
    def __add__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __add__(self, arg: Constraint | ConstraintList, /) -> Model: ...
    @overload
    def __radd__(self, arg: float, /) -> Poly: ...
    @overload
    def __radd__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __radd__(self, arg: Constraint | ConstraintList, /) -> Model: ...
    def __iadd__(self, arg: Poly | float, /) -> Poly: ...
    @overload
    def __sub__(self, arg: Poly | float, /) -> Poly: ...
    @overload
    def __sub__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __rsub__(self, arg: float, /) -> Poly: ...
    @overload
    def __rsub__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    def __isub__(self, arg: Poly | float, /) -> Poly: ...
    @overload
    def __mul__(self, arg: Poly | float, /) -> Poly: ...
    @overload
    def __mul__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __rmul__(self, arg: float, /) -> Poly: ...
    @overload
    def __rmul__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    def __imul__(self, arg: Poly | float, /) -> Poly: ...
    @overload
    def __truediv__(self, arg: float, /) -> Poly: ...
    @overload
    def __truediv__(self, arg: NDArray, /) -> PolyArray[Dim]: ...
    def __itruediv__(self, arg: float, /) -> Poly: ...
    def __eq__(self, arg: object, /) -> bool: ...
    def __ne__(self, arg: object, /) -> bool: ...
    def __pos__(self) -> Poly: ...
    def __neg__(self) -> Poly: ...
    @overload
    def __and__(self, arg: Poly | bool, /) -> Poly: ...
    @overload
    def __and__(self, arg: Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __rand__(self, arg: bool, /) -> Poly: ...
    @overload
    def __rand__(self, arg: Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    def __iand__(self, arg: Poly | bool, /) -> Poly: ...
    @overload
    def __or__(self, arg: Poly | bool, /) -> Poly: ...
    @overload
    def __or__(self, arg: Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __ror__(self, arg: bool, /) -> Poly: ...
    @overload
    def __ror__(self, arg: Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    def __ior__(self, arg: Poly | bool, /) -> Poly: ...
    @overload
    def __xor__(self, arg: Poly | bool, /) -> Poly: ...
    @overload
    def __xor__(self, arg: Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __rxor__(self, arg: bool, /) -> Poly: ...
    @overload
    def __rxor__(self, arg: Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    def __ixor__(self, arg: Poly | bool, /) -> Poly: ...
    def __invert__(self) -> Poly: ...
    def __len__(self) -> int: ...
    def __pow__(self, arg: int, /) -> Poly: ...
    def __ipow__(self, arg: int, /) -> Poly: ...
    def __int__(self) -> int: ...
    def __index__(self) -> int: ...
    def __float__(self) -> float: ...
    def __bool__(self) -> bool: ...

    class Iterator:
        def __iter__(self) -> Poly.Iterator: ...
        def __next__(self) -> tuple[tuple[Poly, ...], float]: ...

    def __iter__(self) -> Poly.Iterator: ...
    @property
    def id(self) -> int:
        """id

        Returns:
            int:
        """

    @property
    def type(self) -> VariableType:
        """type

        Returns:
            amplify.VariableType:
        """

    @property
    def name(self) -> str:
        """name

        Returns:
            str:
        """

    @name.setter
    def name(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def lower_bound(self) -> float | None:
        """lower_bound

        Returns:
            float | None:
        """

    @lower_bound.setter
    def lower_bound(self, arg: float | None) -> None:
        """
        Args:
            arg (float | None)
        """

    @property
    def upper_bound(self) -> float | None:
        """upper_bound

        Returns:
            float | None:
        """

    @upper_bound.setter
    def upper_bound(self, arg: float | None) -> None:
        """
        Args:
            arg (float | None)
        """

    def as_variable(self) -> Variable:
        """as_variable

        Returns:
            amplify.Variable:
        """

    @property
    def variables(self) -> list[Variable]:
        """variables

        Returns:
            list[amplify.Variable]:
        """

    @overload
    def evaluate(self, values: Values) -> float: ...
    @overload
    def evaluate(self, values: Values, default: float) -> float: ...
    @overload
    def evaluate(self, values: Values, default: None) -> Poly:
        """evaluate

        .. admonition:: Overload 1.

            Args:
                * values (amplify.Values)

            Returns:
                float:

        .. admonition:: Overload 2.

            Args:
                * values (amplify.Values)
                * default (float)

            Returns:
                float:

        .. admonition:: Overload 3.

            Args:
                * values (amplify.Values)
                * default (None)

            Returns:
                amplify.Poly:
        """

    @overload
    def decode(self, values: Values) -> float: ...
    @overload
    def decode(self, values: Values, default: float) -> float: ...
    @overload
    def decode(self, values: Values, default: None) -> Poly:
        """decode

        .. admonition:: Overload 1.

            Args:
                * values (amplify.Values)

            Returns:
                float:

        .. admonition:: Overload 2.

            Args:
                * values (amplify.Values)
                * default (float)

            Returns:
                float:

        .. admonition:: Overload 3.

            Args:
                * values (amplify.Values)
                * default (None)

            Returns:
                amplify.Poly:
        """

    def substitute(self, mapping: Mapping[Poly, Poly | float] | Mapping[Variable, Poly | float]) -> Poly:
        """substitute

        Args:
            mapping (collections.abc.Mapping[amplify.Poly, amplify.Poly | float] | collections.abc.Mapping[amplify.Variable, amplify.Poly | float])

        Returns:
            amplify.Poly:
        """

    def is_number(self) -> bool:
        """is_number

        Returns:
            bool:
        """

    def is_variable(self) -> bool:
        """is_variable

        Returns:
            bool:
        """

    def is_linear(self) -> bool:
        """is_linear

        Returns:
            bool:
        """

    def is_quadratic(self) -> bool:
        """is_quadratic

        Returns:
            bool:
        """

    def degree(self) -> int:
        """degree

        Returns:
            int:
        """

    def as_dict(self) -> dict[tuple[int, ...], float]:
        """as_dict

        Returns:
            dict[tuple[int, ...], float]:
        """

    def asdict(self) -> dict[tuple[int, ...], float]:
        """alias of :meth:`~amplify.Poly.as_dict`

        Returns:
            dict[tuple[int, ...], float]:
        """

class PolyArray(Generic[_Dim]):
    def __init__(self, object: NDArray | list | Poly) -> None: ...

    __Dim = TypeVar("__Dim", bound=Dim)

    @overload
    def __add__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __add__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __add__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __add__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __add__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __add__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __add__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __add__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __add__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __add__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __add__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __add__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __add__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __radd__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __radd__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __iadd__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __iadd__(self, arg: list | NDArray, /) -> PolyArray[_Dim]: ...
    @overload
    def __iadd__(self: PolyArray[Dim], arg: PolyArray[__Dim]) -> PolyArray[Dim]: ...
    @overload
    def __iadd__(self: PolyArray[Dim0], arg: PolyArray[Dim0]) -> PolyArray[Dim0]: ...
    @overload
    def __iadd__(self: PolyArray[Dim1], arg: PolyArray[Dim0 | Dim1]) -> PolyArray[Dim1]: ...
    @overload
    def __iadd__(self: PolyArray[Dim2], arg: PolyArray[Dim0 | Dim1 | Dim2]) -> PolyArray[Dim2]: ...
    @overload
    def __iadd__(self: PolyArray[Dim3], arg: PolyArray[Dim0 | Dim1 | Dim2 | Dim3]) -> PolyArray[Dim3]: ...
    @overload
    def __iadd__(self: PolyArray[Dim4], arg: PolyArray[Dim0 | Dim1 | Dim2 | Dim3 | Dim4]) -> PolyArray[Dim4]: ...
    @overload
    def __sub__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __sub__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __sub__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __sub__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __sub__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __sub__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __sub__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __sub__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __sub__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __sub__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __sub__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __sub__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __sub__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __rsub__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __rsub__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __isub__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __isub__(self, arg: list | NDArray, /) -> PolyArray[_Dim]: ...
    @overload
    def __isub__(self: PolyArray[Dim], arg: PolyArray[__Dim]) -> PolyArray[Dim]: ...
    @overload
    def __isub__(self: PolyArray[Dim0], arg: PolyArray[Dim0]) -> PolyArray[Dim0]: ...
    @overload
    def __isub__(self: PolyArray[Dim1], arg: PolyArray[Dim0 | Dim1]) -> PolyArray[Dim1]: ...
    @overload
    def __isub__(self: PolyArray[Dim2], arg: PolyArray[Dim0 | Dim1 | Dim2]) -> PolyArray[Dim2]: ...
    @overload
    def __isub__(self: PolyArray[Dim3], arg: PolyArray[Dim0 | Dim1 | Dim2 | Dim3]) -> PolyArray[Dim3]: ...
    @overload
    def __isub__(self: PolyArray[Dim4], arg: PolyArray[Dim0 | Dim1 | Dim2 | Dim3 | Dim4]) -> PolyArray[Dim4]: ...
    @overload
    def __mul__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __mul__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __mul__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __mul__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __mul__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __mul__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __mul__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __mul__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __mul__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __mul__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __mul__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __mul__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __mul__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __rmul__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __rmul__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    @overload
    def __imul__(self, arg: Poly | float, /) -> PolyArray[_Dim]: ...
    @overload
    def __imul__(self, arg: list | NDArray, /) -> PolyArray[_Dim]: ...
    @overload
    def __imul__(self: PolyArray[Dim], arg: PolyArray[__Dim]) -> PolyArray[Dim]: ...
    @overload
    def __imul__(self: PolyArray[Dim0], arg: PolyArray[Dim0]) -> PolyArray[Dim0]: ...
    @overload
    def __imul__(self: PolyArray[Dim1], arg: PolyArray[Dim0 | Dim1]) -> PolyArray[Dim1]: ...
    @overload
    def __imul__(self: PolyArray[Dim2], arg: PolyArray[Dim0 | Dim1 | Dim2]) -> PolyArray[Dim2]: ...
    @overload
    def __imul__(self: PolyArray[Dim3], arg: PolyArray[Dim0 | Dim1 | Dim2 | Dim3]) -> PolyArray[Dim3]: ...
    @overload
    def __imul__(self: PolyArray[Dim4], arg: PolyArray[Dim0 | Dim1 | Dim2 | Dim3 | Dim4]) -> PolyArray[Dim4]: ...
    @overload
    def __truediv__(self, arg: float, /) -> PolyArray[_Dim]: ...
    @overload
    def __truediv__(self, arg: list | NDArray, /) -> PolyArray[Dim]: ...
    def __itruediv__(self, arg: float | list | NDArray, /) -> PolyArray[_Dim]: ...
    def __matmul__(self, arg: PolyArray | NDArray, /) -> Poly | PolyArray[_Dim]: ...
    def __rmatmul__(self, arg: NDArray | PolyArray[__Dim]) -> Poly | PolyArray[__Dim]: ...
    @overload
    def __eq__(
        self, arg: PolyArray | Poly | float | list | NDArray, /
    ) -> Annotated[numpy.typing.NDArray[numpy.bool_], {"dtype": "bool"}]: ...
    @overload
    def __eq__(self, arg: object, /) -> bool: ...
    @overload
    def __ne__(
        self, arg: PolyArray | Poly | float | list | NDArray, /
    ) -> Annotated[numpy.typing.NDArray[numpy.bool_], {"dtype": "bool"}]: ...
    @overload
    def __ne__(self, arg: object, /) -> bool: ...
    def __pos__(self) -> PolyArray[_Dim]: ...
    def __neg__(self) -> PolyArray[_Dim]: ...
    @overload
    def __and__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __and__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __and__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __and__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __and__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __and__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __and__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __and__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __and__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __and__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __and__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __and__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __and__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __rand__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __rand__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __iand__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __iand__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __iand__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __iand__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __iand__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __iand__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __iand__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __iand__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __iand__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __iand__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __iand__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __iand__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __iand__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __or__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __or__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __or__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __or__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __or__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __or__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __or__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __or__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __or__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __or__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __or__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __or__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __or__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __ror__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __ror__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __ior__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __ior__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __ior__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __ior__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ior__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __ior__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ior__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __ior__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ior__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __ior__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ior__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __ior__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ior__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __xor__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __xor__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __xor__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __xor__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __xor__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __xor__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __xor__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __xor__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __xor__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __xor__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __xor__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __xor__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __xor__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    @overload
    def __rxor__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __rxor__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __ixor__(self, arg: Poly | bool, /) -> PolyArray[_Dim]: ...
    @overload
    def __ixor__(self, arg: list | Annotated[NDArray, {"dtype": "bool"}], /) -> PolyArray[Dim]: ...
    @overload
    def __ixor__(self, arg: PolyArray[Dim]) -> PolyArray[Dim]: ...
    @overload
    def __ixor__(self: PolyArray[Dim0], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ixor__(self, arg: PolyArray[Dim0]) -> PolyArray[_Dim]: ...
    @overload
    def __ixor__(self: PolyArray[Dim1], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ixor__(self, arg: PolyArray[Dim1]) -> PolyArray[_Dim]: ...
    @overload
    def __ixor__(self: PolyArray[Dim2], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ixor__(self, arg: PolyArray[Dim2]) -> PolyArray[_Dim]: ...
    @overload
    def __ixor__(self: PolyArray[Dim3], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ixor__(self, arg: PolyArray[Dim3]) -> PolyArray[_Dim]: ...
    @overload
    def __ixor__(self: PolyArray[Dim4], arg: PolyArray[__Dim]) -> PolyArray[__Dim]: ...
    @overload
    def __ixor__(self, arg: PolyArray[Dim4]) -> PolyArray[_Dim]: ...
    def __pow__(self, arg: int, /) -> PolyArray[_Dim]: ...
    def __ipow__(self, arg: int, /) -> PolyArray[_Dim]: ...
    def __index__(self) -> int: ...
    def __int__(self) -> int: ...
    def __float__(self) -> float: ...
    def __bool__(self) -> bool: ...
    def __invert__(self) -> PolyArray[_Dim]: ...
    @overload
    def __getitem__(self: PolyArray[Dim0], arg: tuple[()]) -> Poly: ...
    @overload
    def __getitem__(self: PolyArray[Dim0], arg: types.EllipsisType) -> PolyArray[Dim0]: ...
    @overload
    def __getitem__(self: PolyArray[Dim1], arg: tuple[int] | int) -> Poly: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim1], arg: types.EllipsisType | tuple[()] | slice | tuple[slice]
    ) -> PolyArray[Dim1]: ...
    @overload
    def __getitem__(self: PolyArray[Dim2], arg: tuple[int, int]) -> Poly: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim2], arg: int | tuple[int] | tuple[slice, int] | tuple[int, slice]
    ) -> PolyArray[Dim1]: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim2], arg: types.EllipsisType | tuple[()] | slice | tuple[slice] | tuple[slice, slice]
    ) -> PolyArray[Dim2]: ...
    @overload
    def __getitem__(self: PolyArray[Dim3], arg: tuple[int, int, int]) -> Poly: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim3],
        arg: tuple[int, int] | tuple[slice, int, int] | tuple[int, slice, int] | tuple[int, int, slice],
    ) -> PolyArray[Dim1]: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim3],
        arg: int
        | tuple[int]
        | tuple[slice, int]
        | tuple[int, slice]
        | tuple[slice, slice, int]
        | tuple[slice, int, slice]
        | tuple[int, slice, slice],
    ) -> PolyArray[Dim2]: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim3],
        arg: types.EllipsisType | tuple[()] | slice | tuple[slice] | tuple[slice, slice] | tuple[slice, slice, slice],
    ) -> PolyArray[Dim3]: ...
    @overload
    def __getitem__(self: PolyArray[Dim4], arg: tuple[int, int, int, int]) -> Poly: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim4],
        arg: tuple[int, int, int]
        | tuple[slice, int, int, int]
        | tuple[int, slice, int, int]
        | tuple[int, int, slice, int]
        | tuple[int, int, int, slice],
    ) -> PolyArray[Dim1]: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim4],
        arg: tuple[int, int]
        | tuple[slice, int, int]
        | tuple[int, slice, int]
        | tuple[int, int, slice]
        | tuple[slice, slice, int, int]
        | tuple[slice, int, slice, int]
        | tuple[slice, int, int, slice]
        | tuple[int, slice, slice, int]
        | tuple[int, slice, int, slice]
        | tuple[int, int, slice, slice],
    ) -> PolyArray[Dim2]: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim4],
        arg: int
        | tuple[int]
        | tuple[slice, int]
        | tuple[int, slice]
        | tuple[slice, slice, int]
        | tuple[int, slice, slice]
        | tuple[slice, int, slice]
        | tuple[slice, slice, slice, int]
        | tuple[slice, slice, int, slice]
        | tuple[slice, int, slice, slice]
        | tuple[int, slice, slice, slice],
    ) -> PolyArray[Dim3]: ...
    @overload
    def __getitem__(
        self: PolyArray[Dim4],
        arg: types.EllipsisType
        | tuple[()]
        | slice
        | tuple[slice]
        | tuple[slice, slice]
        | tuple[slice, slice, slice]
        | tuple[slice, slice, slice, slice],
    ) -> PolyArray[Dim4]: ...
    @overload
    def __getitem__(self: PolyArray[Dim], arg: tuple | slice | types.EllipsisType | int | None) -> PolyArray[Dim]: ...
    def __setitem__(
        self, arg: tuple | slice | types.EllipsisType | int | None, value: Poly | float | list | NDArray | PolyArray
    ) -> None: ...
    def __copy__(self) -> PolyArray[_Dim]: ...
    def __deepcopy__(self, arg: dict, /) -> PolyArray[_Dim]: ...

    class Iterator:
        def __iter__(self) -> PolyArray.Iterator: ...
        def __next__(self) -> Poly | PolyArray[Dim]: ...

    def __iter__(self) -> PolyArray.Iterator: ...
    def __len__(self) -> int: ...
    @property
    def size(self) -> int:
        """size

        Returns:
            int:
        """

    @property
    def ndim(self) -> int:
        """ndim

        Returns:
            int:
        """

    @property
    def flat(self) -> PolyArray[Dim1]:
        """flat

        Returns:
            amplify.PolyArray[Dim1]:
        """

    @property
    def shape(self) -> tuple[int, ...]:
        """shape

        Returns:
            tuple[int, ...]:
        """

    @shape.setter
    def shape(self, arg: int | tuple[int, ...], /) -> None:
        """
        Args:
            arg (int | tuple[int, ...])
        """

    @property
    def T(self) -> PolyArray[_Dim]:
        """T

        Returns:
            amplify.PolyArray[_Dim]:
        """

    def copy(self) -> PolyArray[_Dim]:
        """copy

        Returns:
            amplify.PolyArray[_Dim]:
        """

    def view(self) -> PolyArray[_Dim]:
        """view

        Returns:
            amplify.PolyArray[_Dim]:
        """

    @overload
    def transpose(self, axes: tuple[int, ...] | None = None) -> PolyArray[_Dim]: ...
    @overload
    def transpose(self, *axes: int) -> PolyArray[_Dim]:
        r"""transpose

        .. admonition:: Overload 1.

            Args:
                * axes (tuple[int, ...] | None): Defaults to ``None``.

            Returns:
                amplify.PolyArray[_Dim]:

        .. admonition:: Overload 2.

            Args:
                * \*axes (int)

            Returns:
                amplify.PolyArray[_Dim]:
        """

    def swapaxes(self, axis1: int, axis2: int) -> PolyArray[_Dim]:
        """swapaxes

        Args:
            axis1 (int)
            axis2 (int)

        Returns:
            amplify.PolyArray[_Dim]:
        """

    @overload
    def diagonal(self: PolyArray[Dim], offset: int = 0, axis1: int = 0, axis2: int = 1) -> PolyArray[Dim]: ...
    @overload
    def diagonal(self: PolyArray[Dim2], offset: int = 0, axis1: int = 0, axis2: int = 1) -> PolyArray[Dim1]: ...
    @overload
    def diagonal(self: PolyArray[Dim3], offset: int = 0, axis1: int = 0, axis2: int = 1) -> PolyArray[Dim2]: ...
    @overload
    def diagonal(self: PolyArray[Dim4], offset: int = 0, axis1: int = 0, axis2: int = 1) -> PolyArray[Dim3]:
        """diagonal

        .. admonition:: Overload 1.

            Args:
                * self (amplify.PolyArray[Dim])
                * offset (int): Defaults to ``0``.
                * axis1 (int): Defaults to ``0``.
                * axis2 (int): Defaults to ``1``.

            Returns:
                amplify.PolyArray[Dim]:

        .. admonition:: Overload 2.

            Args:
                * self (amplify.PolyArray[Dim2])
                * offset (int): Defaults to ``0``.
                * axis1 (int): Defaults to ``0``.
                * axis2 (int): Defaults to ``1``.

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 3.

            Args:
                * self (amplify.PolyArray[Dim3])
                * offset (int): Defaults to ``0``.
                * axis1 (int): Defaults to ``0``.
                * axis2 (int): Defaults to ``1``.

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 4.

            Args:
                * self (amplify.PolyArray[Dim4])
                * offset (int): Defaults to ``0``.
                * axis1 (int): Defaults to ``0``.
                * axis2 (int): Defaults to ``1``.

            Returns:
                amplify.PolyArray[Dim3]:
        """

    def fill(self, value: Poly | float) -> None:
        """fill

        Args:
            value (amplify.Poly | float)
        """

    def flatten(self) -> PolyArray[Dim1]:
        """flatten

        Returns:
            amplify.PolyArray[Dim1]:
        """

    def ravel(self) -> PolyArray[Dim1]:
        """ravel

        Returns:
            amplify.PolyArray[Dim1]:
        """

    def roll(self, shift: int, axis: int | None = None) -> PolyArray[_Dim]:
        """roll

        Args:
            shift (int)
            axis (int | None): Defaults to ``None``.

        Returns:
            amplify.PolyArray[_Dim]:
        """

    @overload
    def repeat(self, repeats: int | list[int], axis: Literal[None] = None) -> PolyArray[Dim1]: ...
    @overload
    def repeat(self, repeats: int | list[int], axis: int) -> PolyArray[_Dim]:
        """repeat

        .. admonition:: Overload 1.

            Args:
                * repeats (int | list[int])
                * axis (typing.Literal[None]): Defaults to ``None``.

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 2.

            Args:
                * repeats (int | list[int])
                * axis (int)

            Returns:
                amplify.PolyArray[_Dim]:
        """

    @overload
    def reshape(self, shape: int | tuple[int]) -> PolyArray[Dim1]: ...
    @overload
    def reshape(self, shape: tuple[int, int]) -> PolyArray[Dim2]: ...
    @overload
    def reshape(self, shape: tuple[int, int, int]) -> PolyArray[Dim3]: ...
    @overload
    def reshape(self, shape: tuple[int, int, int, int]) -> PolyArray[Dim4]: ...
    @overload
    def reshape(self, arg0: int, arg1: int) -> PolyArray[Dim2]: ...
    @overload
    def reshape(self, arg0: int, arg1: int, arg2: int) -> PolyArray[Dim3]: ...
    @overload
    def reshape(self, arg0: int, arg1: int, arg2: int, arg3: int) -> PolyArray[Dim4]: ...
    @overload
    def reshape(self, shape: tuple[int, int, int, int, int, *tuple[int, ...]]) -> PolyArray[Dim]: ...
    @overload
    def reshape(self, *shape: int) -> PolyArray[Dim]:
        r"""reshape

        .. admonition:: Overload 1.

            Args:
                * shape (int | tuple[int])

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 2.

            Args:
                * shape (tuple[int, int])

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 3.

            Args:
                * shape (tuple[int, int, int])

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 4.

            Args:
                * shape (tuple[int, int, int, int])

            Returns:
                amplify.PolyArray[Dim4]:

        .. admonition:: Overload 5.

            Args:
                * arg0 (int)
                * arg1 (int)

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 6.

            Args:
                * arg0 (int)
                * arg1 (int)
                * arg2 (int)

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 7.

            Args:
                * arg0 (int)
                * arg1 (int)
                * arg2 (int)
                * arg3 (int)

            Returns:
                amplify.PolyArray[Dim4]:

        .. admonition:: Overload 8.

            Args:
                * shape (tuple[int, int, int, int, int, *tuple[int, ...]])

            Returns:
                amplify.PolyArray[Dim]:

        .. admonition:: Overload 9.

            Args:
                * \*shape (int)

            Returns:
                amplify.PolyArray[Dim]:
        """

    @overload
    def sum(self: PolyArray[Dim], axis: Literal[None] = None) -> Poly: ...
    @overload
    def sum(self: PolyArray[Dim1], axis: int | tuple[int]) -> Poly: ...
    @overload
    def sum(self: PolyArray[Dim2], axis: tuple[int, int]) -> Poly: ...
    @overload
    def sum(self: PolyArray[Dim2], axis: int | tuple[int]) -> PolyArray[Dim1]: ...
    @overload
    def sum(self: PolyArray[Dim3], axis: tuple[int, int, int]) -> Poly: ...
    @overload
    def sum(self: PolyArray[Dim3], axis: tuple[int, int]) -> PolyArray[Dim1]: ...
    @overload
    def sum(self: PolyArray[Dim3], axis: int | tuple[int]) -> PolyArray[Dim2]: ...
    @overload
    def sum(self: PolyArray[Dim4], axis: tuple[int, int, int, int]) -> Poly: ...
    @overload
    def sum(self: PolyArray[Dim4], axis: tuple[int, int, int]) -> PolyArray[Dim1]: ...
    @overload
    def sum(self: PolyArray[Dim4], axis: tuple[int, int]) -> PolyArray[Dim2]: ...
    @overload
    def sum(self: PolyArray[Dim4], axis: int | tuple[int]) -> PolyArray[Dim3]: ...
    @overload
    def sum(self: PolyArray[_Dim], axis: tuple[()]) -> PolyArray[_Dim]: ...
    @overload
    def sum(self: PolyArray[Dim], axis: int | tuple[int, ...]) -> Poly | PolyArray[Dim]:
        """sum

        .. admonition:: Overload 1.

            Args:
                * self (amplify.PolyArray[Dim])
                * axis (typing.Literal[None]): Defaults to ``None``.

            Returns:
                amplify.Poly:

        .. admonition:: Overload 2.

            Args:
                * self (amplify.PolyArray[Dim1])
                * axis (int | tuple[int])

            Returns:
                amplify.Poly:

        .. admonition:: Overload 3.

            Args:
                * self (amplify.PolyArray[Dim2])
                * axis (tuple[int, int])

            Returns:
                amplify.Poly:

        .. admonition:: Overload 4.

            Args:
                * self (amplify.PolyArray[Dim2])
                * axis (int | tuple[int])

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 5.

            Args:
                * self (amplify.PolyArray[Dim3])
                * axis (tuple[int, int, int])

            Returns:
                amplify.Poly:

        .. admonition:: Overload 6.

            Args:
                * self (amplify.PolyArray[Dim3])
                * axis (tuple[int, int])

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 7.

            Args:
                * self (amplify.PolyArray[Dim3])
                * axis (int | tuple[int])

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 8.

            Args:
                * self (amplify.PolyArray[Dim4])
                * axis (tuple[int, int, int, int])

            Returns:
                amplify.Poly:

        .. admonition:: Overload 9.

            Args:
                * self (amplify.PolyArray[Dim4])
                * axis (tuple[int, int, int])

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 10.

            Args:
                * self (amplify.PolyArray[Dim4])
                * axis (tuple[int, int])

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 11.

            Args:
                * self (amplify.PolyArray[Dim4])
                * axis (int | tuple[int])

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 12.

            Args:
                * self (amplify.PolyArray[_Dim])
                * axis (tuple[()])

            Returns:
                amplify.PolyArray[_Dim]:

        .. admonition:: Overload 13.

            Args:
                * self (amplify.PolyArray[Dim])
                * axis (int | tuple[int, ...])

            Returns:
                Poly | PolyArray[Dim]:
        """

    @overload
    def take(self, indices: int, axis: Literal[None] = None) -> Poly: ...
    @overload
    def take(self, indices: list[int], axis: Literal[None] = None) -> PolyArray[Dim1]: ...
    @overload
    def take(self: PolyArray[Dim], indices: int | list[int], axis: int | None = None) -> PolyArray[Dim]: ...
    @overload
    def take(self: PolyArray[Dim2], indices: int, axis: int) -> PolyArray[Dim1]: ...
    @overload
    def take(self: PolyArray[Dim2], indices: list[int], axis: int) -> PolyArray[Dim2]: ...
    @overload
    def take(self: PolyArray[Dim3], indices: int, axis: int) -> PolyArray[Dim2]: ...
    @overload
    def take(self: PolyArray[Dim3], indices: list[int], axis: int) -> PolyArray[Dim3]: ...
    @overload
    def take(self: PolyArray[Dim4], indices: int, axis: int) -> PolyArray[Dim3]: ...
    @overload
    def take(self: PolyArray[Dim4], indices: list[int], axis: int) -> PolyArray[Dim4]:
        """take

        .. admonition:: Overload 1.

            Args:
                * indices (int)
                * axis (typing.Literal[None]): Defaults to ``None``.

            Returns:
                amplify.Poly:

        .. admonition:: Overload 2.

            Args:
                * indices (list[int])
                * axis (typing.Literal[None]): Defaults to ``None``.

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 3.

            Args:
                * self (amplify.PolyArray[Dim])
                * indices (int | list[int])
                * axis (int | None): Defaults to ``None``.

            Returns:
                amplify.PolyArray[Dim]:

        .. admonition:: Overload 4.

            Args:
                * self (amplify.PolyArray[Dim2])
                * indices (int)
                * axis (int)

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 5.

            Args:
                * self (amplify.PolyArray[Dim2])
                * indices (list[int])
                * axis (int)

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 6.

            Args:
                * self (amplify.PolyArray[Dim3])
                * indices (int)
                * axis (int)

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 7.

            Args:
                * self (amplify.PolyArray[Dim3])
                * indices (list[int])
                * axis (int)

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 8.

            Args:
                * self (amplify.PolyArray[Dim4])
                * indices (int)
                * axis (int)

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 9.

            Args:
                * self (amplify.PolyArray[Dim4])
                * indices (list[int])
                * axis (int)

            Returns:
                amplify.PolyArray[Dim4]:
        """

    def nonzero(self) -> tuple[Annotated[numpy.typing.NDArray[numpy.uint64], {"dtype": "uint64"}], ...]:
        """nonzero

        Returns:
            tuple[numpy.ndarray[dtype=uint64], ...]:
        """

    @overload
    def evaluate(self, values: Values) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]: ...
    @overload
    def evaluate(
        self, values: Values, default: float
    ) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]: ...
    @overload
    def evaluate(self, values: Values, default: None) -> PolyArray[_Dim]:
        """evaluate

        .. admonition:: Overload 1.

            Args:
                * values (amplify.Values)

            Returns:
                numpy.ndarray:

        .. admonition:: Overload 2.

            Args:
                * values (amplify.Values)
                * default (float)

            Returns:
                numpy.ndarray:

        .. admonition:: Overload 3.

            Args:
                * values (amplify.Values)
                * default (None)

            Returns:
                amplify.PolyArray[_Dim]:
        """

    @overload
    def decode(self, values: Values) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]: ...
    @overload
    def decode(
        self, values: Values, default: float
    ) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]: ...
    @overload
    def decode(self, values: Values, default: None) -> PolyArray[_Dim]:
        """decode

        .. admonition:: Overload 1.

            Args:
                * values (amplify.Values)

            Returns:
                numpy.ndarray:

        .. admonition:: Overload 2.

            Args:
                * values (amplify.Values)
                * default (float)

            Returns:
                numpy.ndarray:

        .. admonition:: Overload 3.

            Args:
                * values (amplify.Values)
                * default (None)

            Returns:
                amplify.PolyArray[_Dim]:
        """

    def substitute(self, mapping: Mapping[Poly, Poly | float] | Mapping[Variable, Poly | float]) -> PolyArray[_Dim]:
        """substitute

        Args:
            mapping (collections.abc.Mapping[amplify.Poly, amplify.Poly | float] | collections.abc.Mapping[amplify.Variable, amplify.Poly | float])

        Returns:
            amplify.PolyArray[_Dim]:
        """

    @overload
    def to_list(self: PolyArray[Dim0]) -> Poly: ...
    @overload
    def to_list(self: PolyArray[Dim1]) -> list[Poly]: ...
    @overload
    def to_list(self: PolyArray[Dim2]) -> list[list[Poly]]: ...
    @overload
    def to_list(self: PolyArray[Dim3]) -> list[list[list[Poly]]]: ...
    @overload
    def to_list(self: PolyArray[Dim4]) -> list[list[list[list[Poly]]]]: ...
    @overload
    def to_list(self: PolyArray[Dim]) -> list:
        """to_list

        .. admonition:: Overload 1.

            Args:
                * self (amplify.PolyArray[Dim0])

            Returns:
                amplify.Poly:

        .. admonition:: Overload 2.

            Args:
                * self (amplify.PolyArray[Dim1])

            Returns:
                list[amplify.Poly]:

        .. admonition:: Overload 3.

            Args:
                * self (amplify.PolyArray[Dim2])

            Returns:
                list[list[amplify.Poly]]:

        .. admonition:: Overload 4.

            Args:
                * self (amplify.PolyArray[Dim3])

            Returns:
                list[list[list[amplify.Poly]]]:

        .. admonition:: Overload 5.

            Args:
                * self (amplify.PolyArray[Dim4])

            Returns:
                list[list[list[list[amplify.Poly]]]]:

        .. admonition:: Overload 6.

            Args:
                * self (amplify.PolyArray[Dim])

            Returns:
                list:
        """

    @overload
    def tolist(self: PolyArray[Dim0]) -> Poly: ...
    @overload
    def tolist(self: PolyArray[Dim1]) -> list[Poly]: ...
    @overload
    def tolist(self: PolyArray[Dim2]) -> list[list[Poly]]: ...
    @overload
    def tolist(self: PolyArray[Dim3]) -> list[list[list[Poly]]]: ...
    @overload
    def tolist(self: PolyArray[Dim4]) -> list[list[list[list[Poly]]]]: ...
    @overload
    def tolist(self: PolyArray[Dim]) -> list:
        """tolist alias of :meth:`~amplify.PolyArray.to_list`

        .. admonition:: Overload 1.

            Args:
                * self (amplify.PolyArray[Dim0])

            Returns:
                amplify.Poly:

        .. admonition:: Overload 2.

            Args:
                * self (amplify.PolyArray[Dim1])

            Returns:
                list[amplify.Poly]:

        .. admonition:: Overload 3.

            Args:
                * self (amplify.PolyArray[Dim2])

            Returns:
                list[list[amplify.Poly]]:

        .. admonition:: Overload 4.

            Args:
                * self (amplify.PolyArray[Dim3])

            Returns:
                list[list[list[amplify.Poly]]]:

        .. admonition:: Overload 5.

            Args:
                * self (amplify.PolyArray[Dim4])

            Returns:
                list[list[list[list[amplify.Poly]]]]:

        .. admonition:: Overload 6.

            Args:
                * self (amplify.PolyArray[Dim])

            Returns:
                list:
        """

    def to_numpy(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]:
        """to_numpy

        Returns:
            numpy.ndarray:
        """

    def tonumpy(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]:
        """alias of :meth:`~amplify.PolyArray.to_numpy`

        Returns:
            numpy.ndarray:
        """

class QuadratizationMethod(enum.Enum):
    IshikawaKZFD = 0
    Substitute = 1
    ISHIKAWA_KZFD = 0
    """deprecated, use IshikawaKZFD"""

    SUBSTITUTION = 1
    """deprecated, use Substitute"""

class RealEncodingMethod(enum.Enum):
    Random4 = 260
    Random8 = 264
    Random16 = 272
    Random32 = 288

class Result(Generic[_AmplifyClient]):
    class Solution:
        @property
        def values(self) -> Values:
            """values

            Returns:
                amplify.Values:
            """

        @property
        def feasible(self) -> bool:
            """feasible

            Returns:
                bool:
            """

        @property
        def objective(self) -> float:
            """objective

            Returns:
                float:
            """

        @property
        def time(self) -> datetime.timedelta:
            """time

            Returns:
                datetime.timedelta:
            """

    class SolutionList:
        class Iterator:
            def __iter__(self) -> Result.SolutionList.Iterator: ...
            def __next__(self) -> Result.Solution: ...

        def __iter__(self) -> Result.SolutionList.Iterator: ...
        def __len__(self) -> int: ...
        def __bool__(self) -> bool: ...
        def __getitem__(self, key: int) -> Result.Solution: ...

    class ValuesList:
        class Iterator:
            def __iter__(self) -> Result.ValuesList.Iterator: ...
            def __next__(self) -> Values: ...

        def __iter__(self) -> Result.ValuesList.Iterator: ...
        def __len__(self) -> int: ...
        def __bool__(self) -> bool: ...
        def __getitem__(self, key: int) -> Values: ...

    class SplitView:
        def __iter__(self) -> Result.SplitView: ...
        def __next__(self) -> Result: ...
        def __getitem__(self, key: int) -> Result: ...
        def __len__(self) -> int: ...
        def __bool__(self) -> bool: ...

    class ModelConversion:
        class IntermediateMapping:
            class KeysView:
                @overload
                def __contains__(self, arg: Poly, /) -> bool: ...
                @overload
                def __contains__(self, arg: object, /) -> bool: ...
                def __len__(self) -> int: ...
                def __bool__(self) -> bool: ...
                def __iter__(self) -> Result.ModelConversion.IntermediateMapping.KeysView: ...
                def __next__(self) -> Poly: ...

            class ValuesView:
                def __len__(self) -> int: ...
                def __bool__(self) -> bool: ...
                def __iter__(self) -> Result.ModelConversion.IntermediateMapping.ValuesView: ...
                def __next__(self) -> Poly: ...

            class ItemsView:
                def __len__(self) -> int: ...
                def __bool__(self) -> bool: ...
                def __iter__(self) -> Result.ModelConversion.IntermediateMapping.ItemsView: ...
                def __next__(self) -> tuple[Poly, Poly]: ...

            def __len__(self) -> int: ...
            def __bool__(self) -> bool: ...
            @overload
            def __contains__(self, arg: Poly, /) -> bool: ...
            @overload
            def __contains__(self, arg: object, /) -> bool: ...
            def __iter__(self) -> Result.ModelConversion.IntermediateMapping.KeysView: ...
            def __getitem__(self, arg: Poly, /) -> Poly: ...
            def keys(self) -> Result.ModelConversion.IntermediateMapping.KeysView:
                """keys

                Returns:
                    amplify.Result.ModelConversion.IntermediateMapping.KeysView:
                """

            def values(self) -> Result.ModelConversion.IntermediateMapping.ValuesView:
                """values

                Returns:
                    amplify.Result.ModelConversion.IntermediateMapping.ValuesView:
                """

            def items(self) -> Result.ModelConversion.IntermediateMapping.ItemsView:
                """items

                Returns:
                    amplify.Result.ModelConversion.IntermediateMapping.ItemsView:
                """

        @property
        def model(self) -> Model:
            """model

            Returns:
                amplify.Model:
            """

        @property
        def mapping(self) -> Result.ModelConversion.IntermediateMapping:
            """mapping

            Returns:
                amplify.Result.ModelConversion.IntermediateMapping:
            """

        @property
        def values_list(self) -> Result.ValuesList:
            """values_list

            Returns:
                amplify.Result.ValuesList:
            """

        @property
        def num_variables(self) -> int:
            """num_variables

            Returns:
                int:
            """

    class GraphConversion:
        class ChainBreakFractions:
            class Iterator:
                def __iter__(self) -> Result.GraphConversion.ChainBreakFractions.Iterator: ...
                def __next__(self) -> float: ...

            def __iter__(self) -> Result.GraphConversion.ChainBreakFractions.Iterator: ...
            def __len__(self) -> int: ...
            def __bool__(self) -> bool: ...
            def __getitem__(self, key: int) -> float: ...

        @property
        def src_graph(self) -> list[tuple[int, int]]:
            """src_graph

            Returns:
                list[tuple[int, int]]:
            """

        @property
        def dst_graph(self) -> Graph:
            """dst_graph

            Returns:
                amplify.Graph:
            """

        @property
        def chains(self) -> list[Annotated[numpy.typing.NDArray[numpy.uint32], {"dtype": "uint32", "writable": False}]]:
            """chains

            Returns:
                list[numpy.ndarray[dtype=uint32, writable=False]]:
            """

        @property
        def poly(self) -> Poly:
            """poly

            Returns:
                amplify.Poly:
            """

        @property
        def values_list(self) -> Result.ValuesList:
            """values_list

            Returns:
                amplify.Result.ValuesList:
            """

        @property
        def chain_break_fractions(self) -> Result.GraphConversion.ChainBreakFractions:
            """chain_break_fractions

            Returns:
                amplify.Result.GraphConversion.ChainBreakFractions:
            """

        @property
        def num_variables(self) -> int:
            """num_variables

            Returns:
                int:
            """

    class _ClientResultDescriptor:
        @overload
        def __get__(self, instance: Result[FixstarsClient], owner: Any) -> FixstarsClient.Result: ...
        @overload
        def __get__(self, instance: Result[DWaveSamplerClient], owner: Any) -> dimod.sampleset.SampleSet: ...
        @overload
        def __get__(self, instance: Result[LeapHybridSamplerClient], owner: Any) -> dimod.sampleset.SampleSet: ...
        @overload
        def __get__(self, instance: Result[LeapHybridCQMSamplerClient], owner: Any) -> dimod.sampleset.SampleSet: ...
        @overload
        def __get__(self, instance: Result[FujitsuDA4Client], owner: Any) -> FujitsuDA4Client.Result: ...
        @overload
        def __get__(self, instance: Result[ToshibaSQBM2Client], owner: Any) -> ToshibaSQBM2Client.Result: ...
        @overload
        def __get__(self, instance: Result[GurobiClient], owner: Any) -> GurobiClient.Result: ...
        @overload
        def __get__(self, instance: Result[NECVA2Client], owner: Any) -> NECVA2Client.Result: ...
        @overload
        def __get__(self, instance: Result[DASClient], owner: Any) -> DASClient.Result: ...
        @overload
        def __get__(self, instance: Result[HitachiClient], owner: Any) -> HitachiClient.Result: ...
        @overload
        def __get__(self, instance: Any, owner: Any) -> None: ...

    class _EmbeddingDescriptor:
        @overload
        def __get__(
            self, instance: Result[DWaveSamplerClient] | Result[HitachiClient], owner: Any
        ) -> Result.GraphConversion: ...
        @overload
        def __get__(self, instance: Any, owner: Any) -> None: ...

    def __iter__(self) -> Result.SolutionList.Iterator: ...
    def __len__(self) -> int: ...
    def __getitem__(self, key: int) -> Result.Solution: ...
    def sort(self) -> None:
        """sort"""

    @property
    def best(self) -> Result.Solution:
        """best

        Returns:
            amplify.Result.Solution:
        """

    @property
    def solutions(self) -> Result.SolutionList:
        """solutions

        Returns:
            amplify.Result.SolutionList:
        """

    @property
    def filter_solution(self) -> bool:
        """filter_solution

        Returns:
            bool:
        """

    @filter_solution.setter
    def filter_solution(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def num_solves(self) -> int:
        """num_solves

        Returns:
            int:
        """

    @property
    def split(self) -> Result.SplitView:
        """split

        Returns:
            amplify.Result.SplitView:
        """

    @property
    def intermediate(self) -> Result.ModelConversion:
        """intermediate

        Returns:
            amplify.Result.ModelConversion:
        """

    embedding: Result._EmbeddingDescriptor
    """embedding

    Returns:
        amplify.Result.GraphConversion | None:
    """

    client_result: Result._ClientResultDescriptor
    """client_result

    Returns:
        amplify.FixstarsClient.Result | dimod.sampleset.SampleSet | amplify.FujitsuDA4Client.Result | amplify.ToshibaSQBM2Client.Result | amplify.GurobiClient.Result | amplify.NECVA2Client.Result | amplify.DASClient.Result | amplify.HitachiClient.Result | None:
    """

    @property
    def total_time(self) -> datetime.timedelta:
        """total_time

        Returns:
            datetime.timedelta:
        """

    @property
    def execution_time(self) -> datetime.timedelta:
        """execution_time

        Returns:
            datetime.timedelta:
        """

    @property
    def response_time(self) -> datetime.timedelta:
        """response_time

        Returns:
            datetime.timedelta:
        """

class ToshibaSQBM2Client(AmplifyClient):
    @overload
    def __init__(self) -> None: ...
    @overload
    def __init__(self, token: str = "", url: str = "", proxy: str | None = None) -> None: ...

    class Parameters:
        @property
        def steps(self) -> int | None:
            """steps

            Returns:
                int | None:
            """

        @steps.setter
        def steps(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def loops(self) -> int | None:
            """loops

            Returns:
                int | None:
            """

        @loops.setter
        def loops(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def timeout(self) -> datetime.timedelta | None:
            """timeout

            Returns:
                datetime.timedelta | None:
            """

        @timeout.setter
        def timeout(self, arg: float | datetime.timedelta | None) -> None:
            """
            Args:
                arg (float | datetime.timedelta | None)
            """

        @property
        def maxwait(self) -> datetime.timedelta | None:
            """maxwait

            Returns:
                datetime.timedelta | None:
            """

        @maxwait.setter
        def maxwait(self, arg: float | datetime.timedelta | None) -> None:
            """
            Args:
                arg (float | datetime.timedelta | None)
            """

        @property
        def target(self) -> float | None:
            """target

            Returns:
                float | None:
            """

        @target.setter
        def target(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def maxout(self) -> int | None:
            """maxout

            Returns:
                int | None:
            """

        @maxout.setter
        def maxout(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def dt(self) -> float | None:
            """dt

            Returns:
                float | None:
            """

        @dt.setter
        def dt(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def C(self) -> float | None:
            """C

            Returns:
                float | None:
            """

        @C.setter
        def C(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def algo(self) -> int | None:
            """algo

            Returns:
                int | None:
            """

        @algo.setter
        def algo(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def algos(self) -> str | None:
            """algos

            Returns:
                str | None:
            """

        @algos.setter
        def algos(self, arg: str | None) -> None:
            """
            Args:
                arg (str | None)
            """

        @property
        def blocks(self) -> int | None:
            """blocks

            Returns:
                int | None:
            """

        @blocks.setter
        def blocks(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def multishot(self) -> int | None:
            """multishot

            Returns:
                int | None:
            """

        @multishot.setter
        def multishot(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def PD3Orate(self) -> int | None:
            """PD3Orate

            Returns:
                int | None:
            """

        @PD3Orate.setter
        def PD3Orate(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def phi(self) -> float | None:
            """phi

            Returns:
                float | None:
            """

        @phi.setter
        def phi(self, arg: float | None) -> None:
            """
            Args:
                arg (float | None)
            """

        @property
        def detail_level(self) -> int | None:
            """detail_level

            Returns:
                int | None:
            """

        @detail_level.setter
        def detail_level(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def detail_log(self) -> int | None:
            """detail_log

            Returns:
                int | None:
            """

        @detail_log.setter
        def detail_log(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def HMCweight(self) -> int | None:
            """HMCweight

            Returns:
                int | None:
            """

        @HMCweight.setter
        def HMCweight(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

        @property
        def HMCsteps(self) -> int | None:
            """HMCsteps

            Returns:
                int | None:
            """

        @HMCsteps.setter
        def HMCsteps(self, arg: int | None) -> None:
            """
            Args:
                arg (int | None)
            """

    class Solution:
        @property
        def value(self) -> float:
            """value

            Returns:
                float:
            """

        @property
        def result(self) -> Annotated[numpy.typing.NDArray[numpy.float64], {"dtype": "float64"}]:
            """result

            Returns:
                numpy.ndarray:
            """

        @property
        def param(self) -> ToshibaSQBM2Client.Parameters:
            """param

            Returns:
                amplify.ToshibaSQBM2Client.Parameters:
            """

        @property
        def count(self) -> int:
            """count

            Returns:
                int:
            """

    class Result(ToshibaSQBM2Client.Solution):
        @property
        def id(self) -> str:
            """id

            Returns:
                str:
            """

        @property
        def time(self) -> datetime.timedelta:
            """time

            Returns:
                datetime.timedelta:
            """

        @property
        def wait(self) -> datetime.timedelta:
            """wait

            Returns:
                datetime.timedelta:
            """

        @property
        def message(self) -> str:
            """message

            Returns:
                str:
            """

        @property
        def runs(self) -> int:
            """runs

            Returns:
                int:
            """

        @property
        def others(self) -> list[ToshibaSQBM2Client.Solution]:
            """others

            Returns:
                list[amplify.ToshibaSQBM2Client.Solution]:
            """

    class Solver(enum.Enum):
        Qubo = 0
        Qplib = 1
        Pubo = 2

    @property
    def url(self) -> str:
        """url

        Returns:
            str:
        """

    @url.setter
    def url(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def token(self) -> str:
        """token

        Returns:
            str:
        """

    @token.setter
    def token(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def proxy(self) -> str | None:
        """proxy

        Returns:
            str | None:
        """

    @proxy.setter
    def proxy(self, arg: str | None) -> None:
        """
        Args:
            arg (str | None)
        """

    @property
    def compression(self) -> bool:
        """compression

        Returns:
            bool:
        """

    @compression.setter
    def compression(self, arg: bool, /) -> None:
        """
        Args:
            arg (bool)
        """

    @property
    def solver(self) -> ToshibaSQBM2Client.Solver:
        """solver

        Returns:
            amplify.ToshibaSQBM2Client.Solver:
        """

    @solver.setter
    def solver(self, arg: Literal["Qubo", "Qplib", "Pubo"] | ToshibaSQBM2Client.Solver, /) -> None:
        """
        Args:
            arg (typing.Literal['Qubo', 'Qplib', 'Pubo'] | amplify.ToshibaSQBM2Client.Solver)
        """

    @property
    def write_request_data(self) -> pathlib.Path | None:
        """write_request_data

        Returns:
            pathlib.Path | None:
        """

    @write_request_data.setter
    def write_request_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def write_response_data(self) -> pathlib.Path | None:
        """write_response_data

        Returns:
            pathlib.Path | None:
        """

    @write_response_data.setter
    def write_response_data(self, arg: str | os.PathLike | None) -> None:
        """
        Args:
            arg (str | os.PathLike | None)
        """

    @property
    def parameters(self) -> ToshibaSQBM2Client.Parameters:
        """parameters

        Returns:
            amplify.ToshibaSQBM2Client.Parameters:
        """

    @property
    def version(self) -> str:
        """version

        Returns:
            str:
        """

    @property
    def acceptable_degrees(self) -> AcceptableDegrees:
        """acceptable_degrees

        Returns:
            amplify.AcceptableDegrees:
        """

    def health_check(self) -> bool:
        """health_check

        Returns:
            bool:
        """

    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[False] = False) -> ToshibaSQBM2Client.Result: ...
    @overload
    def solve(
        self, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> ToshibaSQBM2Client.Result: ...
    @overload
    def solve(
        self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[False] = False
    ) -> ToshibaSQBM2Client.Result: ...
    @overload
    def solve(self, objective: Poly | Matrix, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None: ...
    @overload
    def solve(self, objective: Poly | Matrix, constraint: Constraint | ConstraintList, dry_run: Literal[True]) -> None:
        """solve

        .. admonition:: Overload 1.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.ToshibaSQBM2Client.Result:

        .. admonition:: Overload 2.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.ToshibaSQBM2Client.Result:

        .. admonition:: Overload 3.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[False]): Defaults to ``False``.

            Returns:
                amplify.ToshibaSQBM2Client.Result:

        .. admonition:: Overload 4.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 5.

            Args:
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])

        .. admonition:: Overload 6.

            Args:
                * objective (amplify.Poly | amplify.Matrix)
                * constraint (amplify.Constraint | amplify.ConstraintList)
                * dry_run (typing.Literal[True])
        """

class Values:
    class KeysView:
        @overload
        def __contains__(self, arg: Poly, /) -> bool: ...
        @overload
        def __contains__(self, arg: object, /) -> bool: ...
        def __len__(self) -> int: ...
        def __bool__(self) -> bool: ...
        def __iter__(self) -> Values.KeysView: ...
        def __next__(self) -> Poly: ...

    class ValuesView:
        def __len__(self) -> int: ...
        def __bool__(self) -> bool: ...
        def __iter__(self) -> Values.ValuesView: ...
        def __next__(self) -> float: ...

    class ItemsView:
        def __len__(self) -> int: ...
        def __bool__(self) -> bool: ...
        def __iter__(self) -> Values.ItemsView: ...
        def __next__(self) -> tuple[Poly, float]: ...

    def __len__(self) -> int: ...
    def __bool__(self) -> bool: ...
    @overload
    def __contains__(self, arg: Poly, /) -> bool: ...
    @overload
    def __contains__(self, arg: object, /) -> bool: ...
    def __iter__(self) -> Values.KeysView: ...
    def __getitem__(self, arg: Poly, /) -> float: ...
    def keys(self) -> Values.KeysView:
        """keys

        Returns:
            amplify.Values.KeysView:
        """

    def values(self) -> Values.ValuesView:
        """values

        Returns:
            amplify.Values.ValuesView:
        """

    def items(self) -> Values.ItemsView:
        """items

        Returns:
            amplify.Values.ItemsView:
        """

class Variable:
    @property
    def id(self) -> int:
        """id

        Returns:
            int:
        """

    @property
    def type(self) -> VariableType:
        """type

        Returns:
            amplify.VariableType:
        """

    @property
    def name(self) -> str:
        """name

        Returns:
            str:
        """

    @name.setter
    def name(self, arg: str, /) -> None:
        """
        Args:
            arg (str)
        """

    @property
    def lower_bound(self) -> float | None:
        """lower_bound

        Returns:
            float | None:
        """

    @lower_bound.setter
    def lower_bound(self, arg: float | None) -> None:
        """
        Args:
            arg (float | None)
        """

    @property
    def upper_bound(self) -> float | None:
        """upper_bound

        Returns:
            float | None:
        """

    @upper_bound.setter
    def upper_bound(self, arg: float | None) -> None:
        """
        Args:
            arg (float | None)
        """

    def __eq__(self, arg: object, /) -> bool: ...
    def __ne__(self, arg: object, /) -> bool: ...

class VariableGenerator:
    def __init__(self) -> None: ...
    @property
    def variables(self) -> list[Variable]:
        """variables

        Returns:
            list[amplify.Variable]:
        """

    def scalar(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> Poly:
        r"""scalar

        Args:
            type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
            bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
            name (str): Defaults to ``\'\'``.

        Returns:
            amplify.Poly:
        """

    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: tuple[()],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> PolyArray[Dim0]: ...
    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: tuple[int],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> PolyArray[Dim1]: ...
    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: tuple[int, int],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> PolyArray[Dim2]: ...
    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: tuple[int, int, int],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> PolyArray[Dim3]: ...
    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: tuple[int, int, int, int],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> PolyArray[Dim4]: ...
    @overload
    def array(
        self, type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType, arg0: int
    ) -> PolyArray[Dim1]: ...
    @overload
    def array(
        self, type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType, arg0: int, arg1: int
    ) -> PolyArray[Dim2]: ...
    @overload
    def array(
        self, type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType, arg0: int, arg1: int, arg2: int
    ) -> PolyArray[Dim3]: ...
    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        arg0: int,
        arg1: int,
        arg2: int,
        arg3: int,
    ) -> PolyArray[Dim4]: ...
    @overload
    def array(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: tuple[int, int, int, int, int, *tuple[int, ...]],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> PolyArray[Dim]: ...
    @overload
    def array(self, type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType, *shape: int) -> PolyArray[Dim]:
        r"""array

        .. admonition:: Overload 1.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (tuple[()])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.PolyArray[Dim0]:

        .. admonition:: Overload 2.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (tuple[int])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 3.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (tuple[int, int])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 4.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (tuple[int, int, int])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 5.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (tuple[int, int, int, int])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.PolyArray[Dim4]:

        .. admonition:: Overload 6.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * arg0 (int)

            Returns:
                amplify.PolyArray[Dim1]:

        .. admonition:: Overload 7.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * arg0 (int)
                * arg1 (int)

            Returns:
                amplify.PolyArray[Dim2]:

        .. admonition:: Overload 8.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * arg0 (int)
                * arg1 (int)
                * arg2 (int)

            Returns:
                amplify.PolyArray[Dim3]:

        .. admonition:: Overload 9.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * arg0 (int)
                * arg1 (int)
                * arg2 (int)
                * arg3 (int)

            Returns:
                amplify.PolyArray[Dim4]:

        .. admonition:: Overload 10.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (tuple[int, int, int, int, int, *tuple[int, ...]])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.PolyArray[Dim]:

        .. admonition:: Overload 11.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * \*shape (int)

            Returns:
                amplify.PolyArray[Dim]:
        """

    @overload
    def matrix(
        self,
        type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType,
        shape: int | tuple[int, ...],
        bounds: tuple[float | None, ...] = (None, None),
        name: str = "",
    ) -> Matrix: ...
    @overload
    def matrix(self, type: Literal["Binary", "Ising", "Integer", "Real"] | VariableType, *shape: int) -> Matrix:
        r"""matrix

        .. admonition:: Overload 1.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * shape (int | tuple[int, ...])
                * bounds (tuple[float | None, ...]): Defaults to ``(None, None)``.
                * name (str): Defaults to ``\'\'``.

            Returns:
                amplify.Matrix:

        .. admonition:: Overload 2.

            Args:
                * type (typing.Literal['Binary', 'Ising', 'Integer', 'Real'] | amplify.VariableType)
                * \*shape (int)

            Returns:
                amplify.Matrix:
        """

class VariableType(enum.Enum):
    Binary = 0
    Ising = 1
    Integer = 2
    Real = 3

_AmplifyClient = TypeVar("_AmplifyClient", bound=AmplifyClient)

_Dim = TypeVar("_Dim", bound=Dim)

@overload
def clamp(
    poly: Poly,
    bounds: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
) -> Constraint: ...
@overload
def clamp(
    array: PolyArray[_Dim],
    bounds: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: Literal[None] = None,
) -> Constraint: ...
@overload
def clamp(
    array: PolyArray[Dim],
    bounds: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int, ...],
) -> Constraint | ConstraintList: ...
@overload
def clamp(
    array: PolyArray[Dim1],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int],
) -> Constraint: ...
@overload
def clamp(
    array: PolyArray[Dim2],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int],
) -> Constraint: ...
@overload
def clamp(
    array: PolyArray[Dim2],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int],
) -> ConstraintList: ...
@overload
def clamp(
    array: PolyArray[Dim3],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int, int],
) -> Constraint: ...
@overload
def clamp(
    array: PolyArray[Dim3],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int] | tuple[int, int],
) -> ConstraintList: ...
@overload
def clamp(
    array: PolyArray[Dim4],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int, int, int],
) -> Constraint: ...
@overload
def clamp(
    array: PolyArray[Dim4],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int] | tuple[int, int] | tuple[int, int, int],
) -> ConstraintList: ...
@overload
def clamp(
    array: PolyArray[_Dim],
    right: tuple[float | None, float | None],
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[()],
) -> ConstraintList:
    r"""clamp

    .. admonition:: Overload 1.

        Args:
            * poly (amplify.Poly)
            * bounds (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[_Dim])
            * bounds (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (typing.Literal[None]): Defaults to ``None``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 3.

        Args:
            * array (amplify.PolyArray[Dim])
            * bounds (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int, ...])

        Returns:
            amplify.Constraint | amplify.ConstraintList:

    .. admonition:: Overload 4.

        Args:
            * array (amplify.PolyArray[Dim1])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 5.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 6.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 7.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 8.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int] | tuple[int, int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 9.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 10.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int] | tuple[int, int] | tuple[int, int, int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 11.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (tuple[float | None, float | None])
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[()])

        Returns:
            amplify.ConstraintList:
    """

@overload
def domain_wall(array: PolyArray[Dim1], label: str = "", *, ascending: bool = True) -> ConstraintList: ...
@overload
def domain_wall(array: PolyArray[_Dim], label: str = "", *, ascending: bool = True) -> ConstraintList:
    r"""domain_wall

    .. admonition:: Overload 1.

        Args:
            * array (amplify.PolyArray[Dim1])
            * label (str): Defaults to ``\'\'``.
            * ascending (bool): Defaults to ``True``.

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[_Dim])
            * label (str): Defaults to ``\'\'``.
            * ascending (bool): Defaults to ``True``.

        Returns:
            amplify.ConstraintList:
    """

@overload
def dot(arg0: PolyArray, arg1: PolyArray | NDArray, /) -> Poly | PolyArray[Dim]: ...
@overload
def dot(arg0: NDArray, arg1: PolyArray, /) -> Poly | PolyArray[Dim]:
    """dot

    .. admonition:: Overload 1.

        Args:
            * arg0 (amplify.PolyArray)
            * arg1 (amplify.PolyArray | ndarray)

        Returns:
            amplify.Poly | amplify.PolyArray[Dim]:

    .. admonition:: Overload 2.

        Args:
            * arg0 (ndarray)
            * arg1 (amplify.PolyArray)

        Returns:
            amplify.Poly | amplify.PolyArray[Dim]:
    """

def einsum(subscripts: str, *operands: PolyArray | numpy.ndarray) -> Poly | PolyArray[Dim]:
    """einsum

    Args:
        subscripts (str)
        *operands (amplify.PolyArray | numpy.ndarray)

    Returns:
        amplify.Poly | amplify.PolyArray[Dim]:
    """

def embed(
    poly: Poly,
    client_graph: Graph,
    embedding_method: Literal["Default", "Minor", "Clique", "Parallel"] | EmbeddingMethod = EmbeddingMethod.Default,
    embedding_timeout: datetime.timedelta = ...,
    chain_strength: float = 1.0,
) -> tuple[Poly, list[Annotated[numpy.typing.NDArray[numpy.uint32], {"dtype": "uint32"}]], list[tuple[int, int]]]:
    """embed

    Args:
        poly (amplify.Poly)
        client_graph (amplify.Graph)
        embedding_method (typing.Literal['Default', 'Minor', 'Clique', 'Parallel'] | amplify.EmbeddingMethod): Defaults to ``EmbeddingMethod.Default``.
        embedding_timeout (datetime.timedelta): Defaults to ``datetime.timedelta(seconds=10)``.
        chain_strength (float): Defaults to ``1.0``.

    Returns:
        tuple[amplify.Poly, list[numpy.ndarray[dtype=uint32]], list[tuple[int, int]]]:
    """

@overload
def equal_to(poly: Poly, right: float, label: str = "") -> Constraint: ...
@overload
def equal_to(array: PolyArray[_Dim], right: float, label: str = "", *, axis: Literal[None] = None) -> Constraint: ...
@overload
def equal_to(
    array: PolyArray[Dim], right: float, label: str = "", *, axis: int | tuple[int, ...]
) -> Constraint | ConstraintList: ...
@overload
def equal_to(array: PolyArray[_Dim], right: float, label: str = "", *, axis: tuple[()]) -> ConstraintList: ...
@overload
def equal_to(array: PolyArray[Dim1], right: float, label: str = "", *, axis: int | tuple[int]) -> Constraint: ...
@overload
def equal_to(array: PolyArray[Dim2], right: float, label: str = "", *, axis: tuple[int, int]) -> Constraint: ...
@overload
def equal_to(array: PolyArray[Dim2], right: float, label: str = "", *, axis: int | tuple[int]) -> ConstraintList: ...
@overload
def equal_to(array: PolyArray[Dim3], right: float, label: str = "", *, axis: tuple[int, int, int]) -> Constraint: ...
@overload
def equal_to(
    array: PolyArray[Dim3], right: float, label: str = "", *, axis: int | tuple[int] | tuple[int, int]
) -> ConstraintList: ...
@overload
def equal_to(
    array: PolyArray[Dim4], right: float, label: str = "", *, axis: tuple[int, int, int, int]
) -> Constraint: ...
@overload
def equal_to(
    array: PolyArray[Dim4],
    right: float,
    label: str = "",
    *,
    axis: int | tuple[int] | tuple[int, int] | tuple[int, int, int],
) -> ConstraintList:
    r"""equal_to

    .. admonition:: Overload 1.

        Args:
            * poly (amplify.Poly)
            * right (float)
            * label (str): Defaults to ``\'\'``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (typing.Literal[None]): Defaults to ``None``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 3.

        Args:
            * array (amplify.PolyArray[Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int, ...])

        Returns:
            amplify.Constraint | amplify.ConstraintList:

    .. admonition:: Overload 4.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[()])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 5.

        Args:
            * array (amplify.PolyArray[Dim1])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 6.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 7.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 8.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 9.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int] | tuple[int, int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 10.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[int, int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 11.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int] | tuple[int, int] | tuple[int, int, int])

        Returns:
            amplify.ConstraintList:
    """

@overload
def greater_equal(
    poly: Poly,
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
) -> Constraint: ...
@overload
def greater_equal(
    array: PolyArray[_Dim],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: Literal[None] = None,
) -> Constraint: ...
@overload
def greater_equal(
    array: PolyArray[Dim],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int, ...],
) -> Constraint | ConstraintList: ...
@overload
def greater_equal(
    array: PolyArray[_Dim],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[()],
) -> ConstraintList: ...
@overload
def greater_equal(
    array: PolyArray[Dim1],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int],
) -> Constraint: ...
@overload
def greater_equal(
    array: PolyArray[Dim2],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int],
) -> Constraint: ...
@overload
def greater_equal(
    array: PolyArray[Dim2],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int],
) -> ConstraintList: ...
@overload
def greater_equal(
    array: PolyArray[Dim3],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int, int],
) -> Constraint: ...
@overload
def greater_equal(
    array: PolyArray[Dim3],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int] | tuple[int, int],
) -> ConstraintList: ...
@overload
def greater_equal(
    array: PolyArray[Dim4],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int, int, int],
) -> Constraint: ...
@overload
def greater_equal(
    array: PolyArray[Dim4],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int] | tuple[int, int] | tuple[int, int, int],
) -> ConstraintList:
    r"""greater_equal

    .. admonition:: Overload 1.

        Args:
            * poly (amplify.Poly)
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (typing.Literal[None]): Defaults to ``None``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 3.

        Args:
            * array (amplify.PolyArray[Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int, ...])

        Returns:
            amplify.Constraint | amplify.ConstraintList:

    .. admonition:: Overload 4.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[()])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 5.

        Args:
            * array (amplify.PolyArray[Dim1])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 6.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 7.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 8.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 9.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int] | tuple[int, int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 10.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 11.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int] | tuple[int, int] | tuple[int, int, int])

        Returns:
            amplify.ConstraintList:
    """

@overload
def less_equal(
    poly: Poly,
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
) -> Constraint: ...
@overload
def less_equal(
    array: PolyArray[_Dim],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: Literal[None] = None,
) -> Constraint: ...
@overload
def less_equal(
    array: PolyArray[Dim],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int, ...],
) -> Constraint | ConstraintList: ...
@overload
def less_equal(
    array: PolyArray[_Dim],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[()],
) -> ConstraintList: ...
@overload
def less_equal(
    array: PolyArray[Dim1],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int],
) -> Constraint: ...
@overload
def less_equal(
    array: PolyArray[Dim2],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int],
) -> Constraint: ...
@overload
def less_equal(
    array: PolyArray[Dim2],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int],
) -> ConstraintList: ...
@overload
def less_equal(
    array: PolyArray[Dim3],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int, int],
) -> Constraint: ...
@overload
def less_equal(
    array: PolyArray[Dim3],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int] | tuple[int, int],
) -> ConstraintList: ...
@overload
def less_equal(
    array: PolyArray[Dim4],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: tuple[int, int, int, int],
) -> Constraint: ...
@overload
def less_equal(
    array: PolyArray[Dim4],
    right: float,
    label: str = "",
    *,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
    axis: int | tuple[int] | tuple[int, int] | tuple[int, int, int],
) -> ConstraintList:
    r"""less_equal

    .. admonition:: Overload 1.

        Args:
            * poly (amplify.Poly)
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (typing.Literal[None]): Defaults to ``None``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 3.

        Args:
            * array (amplify.PolyArray[Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int, ...])

        Returns:
            amplify.Constraint | amplify.ConstraintList:

    .. admonition:: Overload 4.

        Args:
            * array (amplify.PolyArray[_Dim])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[()])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 5.

        Args:
            * array (amplify.PolyArray[Dim1])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 6.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 7.

        Args:
            * array (amplify.PolyArray[Dim2])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 8.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 9.

        Args:
            * array (amplify.PolyArray[Dim3])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int] | tuple[int, int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 10.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (tuple[int, int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 11.

        Args:
            * array (amplify.PolyArray[Dim4])
            * right (float)
            * label (str): Defaults to ``\'\'``.
            * penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.
            * axis (int | tuple[int] | tuple[int, int] | tuple[int, int, int])

        Returns:
            amplify.ConstraintList:
    """

def load_lp(
    filename: str | os.PathLike,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
) -> tuple[Model, VariableGenerator]:
    """load_lp

    Args:
        filename (str | os.PathLike)
        penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.

    Returns:
        tuple[amplify.Model, amplify.VariableGenerator]:
    """

def load_qplib(
    filename: str | os.PathLike,
    penalty_formulation: Literal[
        "Default", "IntegerVariable", "RealVariable", "Relaxation", "LinearRelaxation", "QuadraticRelaxation"
    ]
    | PenaltyFormulation = PenaltyFormulation.Default,
) -> tuple[Model, VariableGenerator]:
    """load_qplib

    Args:
        filename (str | os.PathLike)
        penalty_formulation (typing.Literal['Default', 'IntegerVariable', 'RealVariable', 'Relaxation', 'LinearRelaxation', 'QuadraticRelaxation'] | amplify.PenaltyFormulation): Defaults to ``PenaltyFormulation.Default``.

    Returns:
        tuple[amplify.Model, amplify.VariableGenerator]:
    """

@overload
def matmul(arg0: PolyArray, arg1: PolyArray | NDArray, /) -> Poly | PolyArray[Dim]: ...
@overload
def matmul(arg0: NDArray, arg1: PolyArray, /) -> Poly | PolyArray[Dim]:
    """matmul

    .. admonition:: Overload 1.

        Args:
            * arg0 (amplify.PolyArray)
            * arg1 (amplify.PolyArray | ndarray)

        Returns:
            amplify.Poly | amplify.PolyArray[Dim]:

    .. admonition:: Overload 2.

        Args:
            * arg0 (ndarray)
            * arg1 (amplify.PolyArray)

        Returns:
            amplify.Poly | amplify.PolyArray[Dim]:
    """

newaxis: None = None

@overload
def one_hot(poly: Poly, label: str = "") -> Constraint: ...
@overload
def one_hot(array: PolyArray[_Dim], label: str = "", *, axis: Literal[None] = None) -> Constraint: ...
@overload
def one_hot(array: PolyArray[Dim], label: str = "", *, axis: int | tuple[int, ...]) -> Constraint | ConstraintList: ...
@overload
def one_hot(array: PolyArray[_Dim], label: str = "", *, axis: tuple[()]) -> ConstraintList: ...
@overload
def one_hot(array: PolyArray[Dim1], label: str = "", *, axis: int | tuple[int]) -> Constraint: ...
@overload
def one_hot(array: PolyArray[Dim2], label: str = "", *, axis: tuple[int, int]) -> Constraint: ...
@overload
def one_hot(array: PolyArray[Dim2], label: str = "", *, axis: int | tuple[int]) -> ConstraintList: ...
@overload
def one_hot(array: PolyArray[Dim3], label: str = "", *, axis: tuple[int, int, int]) -> Constraint: ...
@overload
def one_hot(array: PolyArray[Dim3], label: str = "", *, axis: int | tuple[int] | tuple[int, int]) -> ConstraintList: ...
@overload
def one_hot(array: PolyArray[Dim4], label: str = "", *, axis: tuple[int, int, int, int]) -> Constraint: ...
@overload
def one_hot(
    array: PolyArray[Dim4], label: str = "", *, axis: int | tuple[int] | tuple[int, int] | tuple[int, int, int]
) -> ConstraintList:
    r"""one_hot

    .. admonition:: Overload 1.

        Args:
            * poly (amplify.Poly)
            * label (str): Defaults to ``\'\'``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[_Dim])
            * label (str): Defaults to ``\'\'``.
            * axis (typing.Literal[None]): Defaults to ``None``.

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 3.

        Args:
            * array (amplify.PolyArray[Dim])
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int, ...])

        Returns:
            amplify.Constraint | amplify.ConstraintList:

    .. admonition:: Overload 4.

        Args:
            * array (amplify.PolyArray[_Dim])
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[()])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 5.

        Args:
            * array (amplify.PolyArray[Dim1])
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 6.

        Args:
            * array (amplify.PolyArray[Dim2])
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 7.

        Args:
            * array (amplify.PolyArray[Dim2])
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 8.

        Args:
            * array (amplify.PolyArray[Dim3])
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 9.

        Args:
            * array (amplify.PolyArray[Dim3])
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int] | tuple[int, int])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 10.

        Args:
            * array (amplify.PolyArray[Dim4])
            * label (str): Defaults to ``\'\'``.
            * axis (tuple[int, int, int, int])

        Returns:
            amplify.Constraint:

    .. admonition:: Overload 11.

        Args:
            * array (amplify.PolyArray[Dim4])
            * label (str): Defaults to ``\'\'``.
            * axis (int | tuple[int] | tuple[int, int] | tuple[int, int, int])

        Returns:
            amplify.ConstraintList:
    """

def parallel_solve(
    model: Model
    | Poly
    | Matrix
    | Constraint
    | ConstraintList
    | Sequence[Model | Poly | Matrix | Constraint | ConstraintList],
    client: AmplifyClient | Sequence[AmplifyClient],
    *,
    dry_run: bool | Sequence[bool] = False,
    integer_encoding_method: Literal["Unary", "Linear", "Binary", "Default"]
    | IntegerEncodingMethod
    | Sequence[Literal["Unary", "Linear", "Binary", "Default"] | IntegerEncodingMethod] = IntegerEncodingMethod.Default,
    quadratization_method: Literal["IshikawaKZFD", "Substitute"]
    | QuadratizationMethod
    | Sequence[Literal["IshikawaKZFD", "Substitute"] | QuadratizationMethod] = QuadratizationMethod.IshikawaKZFD,
    substitution_multiplier: float | Sequence[float] = 1.0,
    embedding_method: Literal["Default", "Minor", "Clique", "Parallel"]
    | EmbeddingMethod
    | Sequence[Literal["Default", "Minor", "Clique", "Parallel"] | EmbeddingMethod] = EmbeddingMethod.Default,
    embedding_timeout: float | datetime.timedelta | Sequence[float | datetime.timedelta] = 10.0,
    chain_strength: float | Sequence[float] = 1.0,
    num_solves: int | Sequence[int] = 1,
    filter_solution: bool | Sequence[bool] = True,
    sort_solution: bool | Sequence[bool] = True,
    concurrency: int = 0,
) -> list[Result | None]:
    """parallel_solve

    Args:
        model (amplify.Model | amplify.Poly | amplify.Matrix | amplify.Constraint | amplify.ConstraintList | collections.abc.Sequence[amplify.Model | amplify.Poly | amplify.Matrix | amplify.Constraint | amplify.ConstraintList])
        client (amplify.AmplifyClient | collections.abc.Sequence[amplify.AmplifyClient])
        dry_run (bool | collections.abc.Sequence[bool]): Defaults to ``False``.
        integer_encoding_method (typing.Literal['Unary', 'Linear', 'Binary', 'Default'] | amplify.IntegerEncodingMethod | collections.abc.Sequence[typing.Literal['Unary', 'Linear', 'Binary', 'Default'] | amplify.IntegerEncodingMethod]): Defaults to ``IntegerEncodingMethod.Default``.
        quadratization_method (typing.Literal['IshikawaKZFD', 'Substitute'] | amplify.QuadratizationMethod | collections.abc.Sequence[typing.Literal['IshikawaKZFD', 'Substitute'] | amplify.QuadratizationMethod]): Defaults to ``QuadratizationMethod.IshikawaKZFD``.
        substitution_multiplier (float | collections.abc.Sequence[float]): Defaults to ``1.0``.
        embedding_method (typing.Literal['Default', 'Minor', 'Clique', 'Parallel'] | amplify.EmbeddingMethod | collections.abc.Sequence[typing.Literal['Default', 'Minor', 'Clique', 'Parallel'] | amplify.EmbeddingMethod]): Defaults to ``EmbeddingMethod.Default``.
        embedding_timeout (float | datetime.timedelta | collections.abc.Sequence[float | datetime.timedelta]): Defaults to ``10.0``.
        chain_strength (float | collections.abc.Sequence[float]): Defaults to ``1.0``.
        num_solves (int | collections.abc.Sequence[int]): Defaults to ``1``.
        filter_solution (bool | collections.abc.Sequence[bool]): Defaults to ``True``.
        sort_solution (bool | collections.abc.Sequence[bool]): Defaults to ``True``.
        concurrency (int): Defaults to ``0``.

    Returns:
        list[amplify.Result | None]:
    """

def save_lp(model: Model, filename: str | os.PathLike) -> None:
    """save_lp

    Args:
        model (amplify.Model)
        filename (str | os.PathLike)
    """

def save_qplib(model: Model, filename: str | os.PathLike) -> None:
    """save_qplib

    Args:
        model (amplify.Model)
        filename (str | os.PathLike)
    """

def set_num_threads(arg: int, /) -> None:
    """set_num_threads

    Args:
        arg (int)
    """

def set_seed(arg: int, /) -> None:
    """set_seed

    Args:
        arg (int)
    """

@overload
def solve(
    model: Model | Poly | Matrix | Constraint | ConstraintList,
    client: _AmplifyClient,
    *,
    dry_run: Literal[False] = False,
    integer_encoding_method: Literal["Unary", "Linear", "Binary", "Default"]
    | IntegerEncodingMethod = IntegerEncodingMethod.Default,
    real_encoding_method: Literal["Random4", "Random8", "Random16", "Random32"]
    | RealEncodingMethod = RealEncodingMethod.Random16,
    quadratization_method: Literal["IshikawaKZFD", "Substitute"]
    | QuadratizationMethod = QuadratizationMethod.IshikawaKZFD,
    substitution_multiplier: float = 1.0,
    embedding_method: Literal["Default", "Minor", "Clique", "Parallel"] | EmbeddingMethod = EmbeddingMethod.Default,
    embedding_timeout: float | datetime.timedelta = 10.0,
    chain_strength: float = 1.0,
    num_solves: int = 1,
    filter_solution: bool = True,
    sort_solution: bool = True,
) -> Result[_AmplifyClient]: ...
@overload
def solve(
    model: Model | Poly | Matrix | Constraint | ConstraintList,
    client: AmplifyClient,
    *,
    dry_run: Literal[True],
    integer_encoding_method: Literal["Unary", "Linear", "Binary", "Default"]
    | IntegerEncodingMethod = IntegerEncodingMethod.Default,
    real_encoding_method: Literal["Random4", "Random8", "Random16", "Random32"]
    | RealEncodingMethod = RealEncodingMethod.Random16,
    quadratization_method: Literal["IshikawaKZFD", "Substitute"]
    | QuadratizationMethod = QuadratizationMethod.IshikawaKZFD,
    substitution_multiplier: float = 1.0,
    embedding_method: Literal["Default", "Minor", "Clique", "Parallel"] | EmbeddingMethod = EmbeddingMethod.Default,
    embedding_timeout: float | datetime.timedelta = 10.0,
    chain_strength: float = 1.0,
    num_solves: int = 1,
    filter_solution: bool = True,
    sort_solution: bool = True,
) -> Result:
    """solve

    .. admonition:: Overload 1.

        Args:
            * model (amplify.Model | amplify.Poly | amplify.Matrix | amplify.Constraint | amplify.ConstraintList)
            * client (amplify._AmplifyClient)
            * dry_run (typing.Literal[False]): Defaults to ``False``.
            * integer_encoding_method (typing.Literal['Unary', 'Linear', 'Binary', 'Default'] | amplify.IntegerEncodingMethod): Defaults to ``IntegerEncodingMethod.Default``.
            * real_encoding_method (typing.Literal['Random4', 'Random8', 'Random16', 'Random32'] | amplify.RealEncodingMethod): Defaults to ``RealEncodingMethod.Random16``.
            * quadratization_method (typing.Literal['IshikawaKZFD', 'Substitute'] | amplify.QuadratizationMethod): Defaults to ``QuadratizationMethod.IshikawaKZFD``.
            * substitution_multiplier (float): Defaults to ``1.0``.
            * embedding_method (typing.Literal['Default', 'Minor', 'Clique', 'Parallel'] | amplify.EmbeddingMethod): Defaults to ``EmbeddingMethod.Default``.
            * embedding_timeout (float | datetime.timedelta): Defaults to ``10.0``.
            * chain_strength (float): Defaults to ``1.0``.
            * num_solves (int): Defaults to ``1``.
            * filter_solution (bool): Defaults to ``True``.
            * sort_solution (bool): Defaults to ``True``.

        Returns:
            amplify.Result[amplify._AmplifyClient]:

    .. admonition:: Overload 2.

        Args:
            * model (amplify.Model | amplify.Poly | amplify.Matrix | amplify.Constraint | amplify.ConstraintList)
            * client (amplify.AmplifyClient)
            * dry_run (typing.Literal[True])
            * integer_encoding_method (typing.Literal['Unary', 'Linear', 'Binary', 'Default'] | amplify.IntegerEncodingMethod): Defaults to ``IntegerEncodingMethod.Default``.
            * real_encoding_method (typing.Literal['Random4', 'Random8', 'Random16', 'Random32'] | amplify.RealEncodingMethod): Defaults to ``RealEncodingMethod.Random16``.
            * quadratization_method (typing.Literal['IshikawaKZFD', 'Substitute'] | amplify.QuadratizationMethod): Defaults to ``QuadratizationMethod.IshikawaKZFD``.
            * substitution_multiplier (float): Defaults to ``1.0``.
            * embedding_method (typing.Literal['Default', 'Minor', 'Clique', 'Parallel'] | amplify.EmbeddingMethod): Defaults to ``EmbeddingMethod.Default``.
            * embedding_timeout (float | datetime.timedelta): Defaults to ``10.0``.
            * chain_strength (float): Defaults to ``1.0``.
            * num_solves (int): Defaults to ``1``.
            * filter_solution (bool): Defaults to ``True``.
            * sort_solution (bool): Defaults to ``True``.

        Returns:
            amplify.Result:
    """

@overload
def sum(array: PolyArray[_Dim], axis: Literal[None] = None) -> Poly: ...
@overload
def sum(array: PolyArray[Dim1], axis: int | tuple[int]) -> Poly: ...
@overload
def sum(array: PolyArray[Dim2], axis: tuple[int, int]) -> Poly: ...
@overload
def sum(array: PolyArray[Dim2], axis: int | tuple[int]) -> PolyArray[Dim1]: ...
@overload
def sum(array: PolyArray[Dim3], axis: tuple[int, int, int]) -> Poly: ...
@overload
def sum(array: PolyArray[Dim3], axis: tuple[int, int]) -> PolyArray[Dim1]: ...
@overload
def sum(array: PolyArray[Dim3], axis: int | tuple[int]) -> PolyArray[Dim2]: ...
@overload
def sum(array: PolyArray[Dim4], axis: tuple[int, int, int, int]) -> Poly: ...
@overload
def sum(array: PolyArray[Dim4], axis: tuple[int, int, int]) -> PolyArray[Dim1]: ...
@overload
def sum(array: PolyArray[Dim4], axis: tuple[int, int]) -> PolyArray[Dim2]: ...
@overload
def sum(array: PolyArray[Dim4], axis: int | tuple[int]) -> PolyArray[Dim3]: ...
@overload
def sum(array: PolyArray[_Dim], axis: tuple[()]) -> PolyArray[_Dim]: ...
@overload
def sum(array: PolyArray[Dim], axis: int | tuple[int, ...]) -> Poly | PolyArray[Dim]: ...
@overload
def sum(arg: Sequence[Poly | float] | Iterator[Poly | float]) -> Poly: ...
@overload
def sum(arg: Sequence[Constraint | ConstraintList] | Iterator[Constraint | ConstraintList]) -> ConstraintList: ...
@overload
def sum(stop: int, func: Callable[[int], Poly]) -> Poly: ...
@overload
def sum(start: int, stop: int, func: Callable[[int], Poly]) -> Poly: ...
@overload
def sum(start: int, stop: int, step: int, func: Callable[[int], Poly]) -> Poly: ...
@overload
def sum(sequence: Sequence[object], func: Callable[[object], Poly]) -> Poly: ...
@overload
def sum(iterable: Iterable[object], func: Callable[[object], Poly]) -> Poly: ...
@overload
def sum(iterator: Iterator[object], func: Callable[[object], Poly]) -> Poly: ...
@overload
def sum(iterable: Iterable, /, start: object = 0) -> Any:
    """sum

    .. admonition:: Overload 1.

        Args:
            * array (amplify.PolyArray[_Dim])
            * axis (typing.Literal[None]): Defaults to ``None``.

        Returns:
            amplify.Poly:

    .. admonition:: Overload 2.

        Args:
            * array (amplify.PolyArray[Dim1])
            * axis (int | tuple[int])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 3.

        Args:
            * array (amplify.PolyArray[Dim2])
            * axis (tuple[int, int])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 4.

        Args:
            * array (amplify.PolyArray[Dim2])
            * axis (int | tuple[int])

        Returns:
            amplify.PolyArray[Dim1]:

    .. admonition:: Overload 5.

        Args:
            * array (amplify.PolyArray[Dim3])
            * axis (tuple[int, int, int])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 6.

        Args:
            * array (amplify.PolyArray[Dim3])
            * axis (tuple[int, int])

        Returns:
            amplify.PolyArray[Dim1]:

    .. admonition:: Overload 7.

        Args:
            * array (amplify.PolyArray[Dim3])
            * axis (int | tuple[int])

        Returns:
            amplify.PolyArray[Dim2]:

    .. admonition:: Overload 8.

        Args:
            * array (amplify.PolyArray[Dim4])
            * axis (tuple[int, int, int, int])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 9.

        Args:
            * array (amplify.PolyArray[Dim4])
            * axis (tuple[int, int, int])

        Returns:
            amplify.PolyArray[Dim1]:

    .. admonition:: Overload 10.

        Args:
            * array (amplify.PolyArray[Dim4])
            * axis (tuple[int, int])

        Returns:
            amplify.PolyArray[Dim2]:

    .. admonition:: Overload 11.

        Args:
            * array (amplify.PolyArray[Dim4])
            * axis (int | tuple[int])

        Returns:
            amplify.PolyArray[Dim3]:

    .. admonition:: Overload 12.

        Args:
            * array (amplify.PolyArray[_Dim])
            * axis (tuple[()])

        Returns:
            amplify.PolyArray[_Dim]:

    .. admonition:: Overload 13.

        Args:
            * array (amplify.PolyArray[Dim])
            * axis (int | tuple[int, ...])

        Returns:
            amplify.Poly | amplify.PolyArray[Dim]:

    .. admonition:: Overload 14.

        Args:
            * arg (collections.abc.Sequence[amplify.Poly | float] | collections.abc.Iterator[amplify.Poly | float])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 15.

        Args:
            * arg (collections.abc.Sequence[amplify.Constraint | amplify.ConstraintList] | collections.abc.Iterator[amplify.Constraint | amplify.ConstraintList])

        Returns:
            amplify.ConstraintList:

    .. admonition:: Overload 16.

        Args:
            * stop (int)
            * func (collections.abc.Callable[[int], amplify.Poly])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 17.

        Args:
            * start (int)
            * stop (int)
            * func (collections.abc.Callable[[int], amplify.Poly])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 18.

        Args:
            * start (int)
            * stop (int)
            * step (int)
            * func (collections.abc.Callable[[int], amplify.Poly])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 19.

        Args:
            * sequence (collections.abc.Sequence[object])
            * func (collections.abc.Callable[[object], amplify.Poly])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 20.

        Args:
            * iterable (collections.abc.Iterable[object])
            * func (collections.abc.Callable[[object], amplify.Poly])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 21.

        Args:
            * iterator (collections.abc.Iterator[object])
            * func (collections.abc.Callable[[object], amplify.Poly])

        Returns:
            amplify.Poly:

    .. admonition:: Overload 22.

        Args:
            * iterable (Iterable)
            * start (object): Defaults to ``0``.

        Returns:
            typing.Any:
    """

def to_edges(poly: Poly) -> list[tuple[int, int]]:
    """to_edges

    Args:
        poly (amplify.Poly)

    Returns:
        list[tuple[int, int]]:
    """

__version__: str
