"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
// import required to be here, otherwise causes a cycle when running the generated JavaScript
// tslint:disable-next-line:ordered-imports
const cfn_element_1 = require("./cfn-element");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const cfn_reference_1 = require("./private/cfn-reference");
const removal_policy_1 = require("./removal-policy");
const tag_manager_1 = require("./tag-manager");
const util_1 = require("./util");
/**
 * Represents a CloudFormation resource.
 */
class CfnResource extends cfn_element_1.CfnRefElement {
    /**
     * Creates a resource construct.
     * @param cfnResourceType The CloudFormation type of this resource (e.g. AWS::DynamoDB::Table)
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("CFN") resources (such as `s3.CfnBucket`). These resources will have a
        // property for each CloudFormation property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `cfnXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         */
        this.cfnOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        if (!props.type) {
            throw new Error('The `type` property is required');
        }
        this.cfnResourceType = props.type;
        this._cfnProperties = props.properties || {};
        // if aws:cdk:enable-path-metadata is set, embed the current construct's
        // path in the CloudFormation template, so it will be possible to trace
        // back to the actual construct path.
        if (this.node.tryGetContext(cxapi.PATH_METADATA_ENABLE_CONTEXT)) {
            this.cfnOptions.metadata = {
                [cxapi.PATH_METADATA_KEY]: this.node.path
            };
        }
    }
    /**
     * Check whether the given construct is a CfnResource
     */
    static isCfnResource(construct) {
        return construct.cfnResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     */
    applyRemovalPolicy(policy, options = {}) {
        policy = policy || options.default || removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
                break;
            case removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.cfnOptions.deletionPolicy = deletionPolicy;
        if (options.applyToUpdateReplacePolicy !== false) {
            this.cfnOptions.updateReplacePolicy = deletionPolicy;
        }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     * @param attributeName The name of the attribute.
     */
    getAtt(attributeName) {
        return cfn_reference_1.CfnReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized CloudFormation resource. To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * @param path  The path of the property, you can use dot notation to
     *        override values in complex types. Any intermdediate keys
     *        will be created as needed.
     * @param value The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = path.split('.');
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null && typeof (curr[key]) === 'object' && !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property
     * @param value The value
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     * @param propertyPath The path to the property.
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be provisioned
     * unless the other resource has been successfully provisioned.
     */
    addDependsOn(resource) {
        this.dependsOn.add(resource);
    }
    /**
     * @returns a string representation of this resource
     */
    toString() {
        return `${super.toString()} [${this.cfnResourceType}]`;
    }
    /**
     * Emits CloudFormation for this resource.
     * @internal
     */
    _toCloudFormation() {
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.logicalId]: new util_1.PostResolveToken({
                        Type: this.cfnResourceType,
                        Properties: util_1.ignoreEmpty(this.cfnProperties),
                        DependsOn: util_1.ignoreEmpty(renderDependsOn(this.dependsOn)),
                        CreationPolicy: util_1.capitalizePropertyNames(this, renderCreationPolicy(this.cfnOptions.creationPolicy)),
                        UpdatePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updatePolicy),
                        UpdateReplacePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.deletionPolicy),
                        Metadata: util_1.ignoreEmpty(this.cfnOptions.metadata),
                        Condition: this.cfnOptions.condition && this.cfnOptions.condition.logicalId
                    }, props => {
                        const renderedProps = this.renderProperties(props.Properties || {});
                        props.Properties = renderedProps && (Object.values(renderedProps).find(v => !!v) ? renderedProps : undefined);
                        return deepMerge(props, this.rawOverrides);
                    })
                }
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ['--- resource created at ---', ...trace].join('\n  at ');
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        // returns the set of logical ID (tokens) this resource depends on
        // sorted by construct paths to ensure test determinism
        function renderDependsOn(dependsOn) {
            return Array
                .from(dependsOn)
                .sort((x, y) => x.node.path.localeCompare(y.node.path))
                .map(r => r.logicalId);
        }
        function renderCreationPolicy(policy) {
            if (!policy) {
                return undefined;
            }
            const result = Object.assign({}, policy);
            if (policy.resourceSignal && policy.resourceSignal.timeout) {
                result.resourceSignal = policy.resourceSignal;
            }
            return result;
        }
    }
    get cfnProperties() {
        const tags = tag_manager_1.TagManager.isTaggable(this) ? this.tags.renderTags() : {};
        return deepMerge(this._cfnProperties || {}, { tags });
    }
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     */
    get updatedProperites() {
        return this._cfnProperties;
    }
    validateProperties(_properties) {
        // Nothing
    }
}
exports.CfnResource = CfnResource;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof (source) !== 'object' || typeof (target) !== 'object') {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof (value) === 'object' && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
//# sourceMappingURL=data:application/json;base64,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