import { Dialog, InputDialog, showDialog, showErrorMessage } from '@jupyterlab/apputils';
/**
 * The command IDs used by the git plugin.
 */
export var CommandIDs;
(function (CommandIDs) {
    CommandIDs.gitUI = 'git:ui';
    CommandIDs.gitTerminalCommand = 'git:terminal-command';
    CommandIDs.gitInit = 'git:init';
    CommandIDs.gitOpenUrl = 'git:open-url';
    CommandIDs.gitToggleSimpleStaging = 'git:toggle-simple-staging';
    CommandIDs.gitAddRemote = 'git:add-remote';
})(CommandIDs || (CommandIDs = {}));
/**
 * Add the commands for the git extension.
 */
export function addCommands(app, model, fileBrowser, settings) {
    const { commands, shell } = app;
    /**
     * Add open terminal in the Git repository
     */
    commands.addCommand(CommandIDs.gitTerminalCommand, {
        label: 'Open Terminal in Git Repository',
        caption: 'Open a New Terminal in the Git Repository',
        execute: async (args) => {
            const main = (await commands.execute('terminal:create-new', args));
            try {
                if (model.pathRepository !== null) {
                    const terminal = main.content;
                    terminal.session.send({
                        type: 'stdin',
                        content: [`cd "${model.pathRepository.split('"').join('\\"')}"\n`]
                    });
                }
                return main;
            }
            catch (e) {
                console.error(e);
                main.dispose();
            }
        }
    });
    /** Add open/go to git interface command */
    commands.addCommand(CommandIDs.gitUI, {
        label: 'Git Interface',
        caption: 'Go to Git user interface',
        execute: () => {
            try {
                shell.activateById('jp-git-sessions');
            }
            catch (err) { }
        }
    });
    /** Add git init command */
    commands.addCommand(CommandIDs.gitInit, {
        label: 'Init',
        caption: ' Create an empty Git repository or reinitialize an existing one',
        execute: async () => {
            const currentPath = fileBrowser.model.path;
            const result = await showDialog({
                title: 'Initialize a Repository',
                body: 'Do you really want to make this directory a Git Repo?',
                buttons: [Dialog.cancelButton(), Dialog.warnButton({ label: 'Yes' })]
            });
            if (result.button.accept) {
                await model.init(currentPath);
                model.pathRepository = currentPath;
            }
        }
    });
    /** Open URL externally */
    commands.addCommand(CommandIDs.gitOpenUrl, {
        label: args => args['text'],
        execute: args => {
            const url = args['url'];
            window.open(url);
        }
    });
    /** add toggle for simple staging */
    commands.addCommand(CommandIDs.gitToggleSimpleStaging, {
        label: 'Simple staging',
        isToggled: () => !!settings.composite['simpleStaging'],
        execute: args => {
            settings.set('simpleStaging', !settings.composite['simpleStaging']);
        }
    });
    /** Command to add a remote Git repository */
    commands.addCommand(CommandIDs.gitAddRemote, {
        label: 'Add remote repository',
        caption: 'Add a Git remote repository',
        isEnabled: () => model.pathRepository !== null,
        execute: async (args) => {
            if (model.pathRepository === null) {
                console.warn('Not in a Git repository. Unable to add a remote.');
                return;
            }
            let url = args['url'];
            let name = args['name'];
            if (!url) {
                const result = await InputDialog.getText({
                    title: 'Add a remote repository',
                    placeholder: 'Remote Git repository URL'
                });
                if (result.button.accept) {
                    url = result.value;
                }
            }
            if (url) {
                try {
                    await model.addRemote(url, name);
                }
                catch (error) {
                    console.error(error);
                    showErrorMessage('Error when adding remote repository', error);
                }
            }
        }
    });
}
//# sourceMappingURL=gitMenuCommands.js.map