from unittest import TestCase
from unittest.mock import Mock

import pytest
from pysnmp.smi.error import MibLoadError, MibNotFoundError

from cloudshell.snmp.core.snmp_mib_builder import QualiMibBuilder
from cloudshell.snmp.core.tools.snmp_json_mib import JsonMib
from cloudshell.snmp.core.tools.snmp_json_mib_parser import JsonMibParser

# test_snmp_json_mib_parser.py - Generated by CodiumAI


"""
Code Analysis:
-The JsonMibParser class is responsible for parsing JSON MIB files and providing methods to access them.
- It has an __init__ method that takes a mib_builder object as a parameter and initializes an empty dictionary called json_mibs.
- The load_json_mib method takes a mib_name as a parameter and attempts to load the JSON MIB file with that name from the mib_builder object. If successful, it creates a JsonMib object with the loaded data and adds it to the json_mibs dictionary.
- The guess_mib_by_oid method takes an OID as a parameter and attempts to find the MIB file that contains that OID. It first checks the json_mibs dictionary for the OID, and if not found, it searches through the mibSymbols dictionary of the mib_builder object.
- If the OID is found in either dictionary, the method returns the name of the MIB file. Otherwise, it returns None.
- The JsonMibParser class relies on the JsonMib class to actually parse the JSON data and provide access to the MIB symbols and types.
- Overall, the JsonMibParser class provides a convenient way to load and access JSON MIB files in a Python program.
"""

"""
Test Plan:
- test_load_json_mib_success(): tests that the load_json_mib method successfully loads a JSON MIB file. Tags: [happy path]
- test_guess_mib_by_oid_success(): tests that the guess_mib_by_oid method successfully guesses the MIB file name containing a given OID. Tags: [happy path]
- test_load_json_mib_nonexistent_file(): tests that the load_json_mib method raises a MibNotFoundError exception when attempting to load a non-existent JSON MIB file. Tags: [edge case]
- test_load_json_mib_invalid_json(): tests that the load_json_mib method raises a MibLoadError exception when attempting to load a JSON MIB file with invalid JSON data. Tags: [edge case]
- test_load_json_mib_mib_not_found(): tests that the load_json_mib method raises a MibNotFoundError exception when attempting to load a MIB file that does not exist in any of the loaded MIB sources. Tags: [general behavior]
- test_guess_mib_by_oid_not_found(): tests that the guess_mib_by_oid method returns None when attempting to guess the MIB file name for an OID that does not exist in any loaded MIB file. Tags: [edge case]
- test_guess_mib_by_oid_empty_json_mibs(): tests that the guess_mib_by_oid method returns None when there are no loaded JSON MIB files. Tags: [general behavior]
- test_guess_mib_by_oid_symbol_found(): tests that the guess_mib_by_oid method successfully guesses the MIB file name containing a given OID that is a symbol in a loaded JSON MIB file. Tags: [happy path]
- test_guess_mib_by_oid_type_found(): tests that the guess_mib_by_oid method successfully guesses the MIB file name containing a given OID that is a type in a loaded JSON MIB file. Tags: [happy path]
- test_guess_mib_by_oid_symbol_found_in_mib_builder(): tests that the guess_mib_by_oid method successfully guesses the MIB file name containing a given OID that is a symbol in the mib_builder object. Tags: [happy path]
- test_guess_mib_by_oid_type_found_in_mib_builder(): tests that the guess_mib_by_oid method successfully guesses the MIB file name containing a given OID that is a type in the mib_builder object. Tags: [happy path]
"""


class TestJsonMibParser(TestCase):
    def test_load_json_mib_success(self):
        # Arrange
        mib_builder = Mock()
        mib_source = Mock()
        mib_source.read_json.return_value = ('{"imports": {}}', "path/to/test_mib.json")
        mib_builder.getMibSources.return_value = [mib_source]
        parser = JsonMibParser(mib_builder)

        # Act
        parser.load_json_mib("test_mib")

        # Assert
        assert "test_mib" in parser.json_mibs.keys()

    def test_guess_mib_by_oid_success(self):
        # Arrange
        mib_builder = QualiMibBuilder()
        json_mib = JsonMib(
            mib_builder,
            "test_mib",
            {
                "imports": {},
                "system": {
                    "name": "system",
                    "oid": "1.3.6.1.2.1.1",
                    "class": "objectidentity",
                },
            },
            mib_builder.json_mib_parser,
        )
        mib_builder.json_mib_parser.json_mibs["test_mib"] = json_mib

        # Act
        result = mib_builder.json_mib_parser.guess_mib_by_oid("1.3.6.1.2.1.1")

        # Assert
        assert result == "test_mib"

    def test_load_json_mib_nonexistent_file(self):
        # Arrange
        mib_builder = Mock()
        mib_builder.getMibSources.return_value = []
        parser = JsonMibParser(mib_builder)

        # Act & Assert
        with pytest.raises(MibNotFoundError):
            parser.load_json_mib("nonexistent_mib")

    def test_load_json_mib_invalid_json(self):
        # Arrange
        mib_builder = Mock()
        mib_source = Mock()
        mib_source.read_json.return_value = (
            '{"name": "test_mib",}',
            "path/to/test_mib.json",
        )
        mib_builder.getMibSources.return_value = [mib_source]
        parser = JsonMibParser(mib_builder)

        # Act & Assert
        with pytest.raises(MibLoadError):
            parser.load_json_mib("test_mib")

    def test_load_json_mib_mib_not_found(self):
        # Arrange
        mib_builder = Mock()
        mib_source = Mock()
        mib_source.read_json.side_effect = AttributeError
        mib_builder.getMibSources.return_value = [mib_source]
        parser = JsonMibParser(mib_builder)

        # Act & Assert
        with pytest.raises(MibNotFoundError):
            parser.load_json_mib("nonexistent_mib")

    def test_guess_mib_by_oid_not_found(self):
        # Arrange
        mib_builder = QualiMibBuilder()
        json_mib = JsonMib(
            mib_builder,
            "test_mib",
            {
                "imports": {},
                "system": {
                    "name": "system",
                    "oid": "1.3.6.1.2.1.1",
                    "class": "objectidentity",
                },
            },
            mib_builder.json_mib_parser,
        )
        mib_builder.json_mib_parser.json_mibs["test_mib"] = json_mib

        # Act
        result = mib_builder.json_mib_parser.guess_mib_by_oid("1.3.6.1.4.1.5678")

        # Assert
        assert result is None
