# Proprietary Changes made for Trainy under the Trainy Software License
# Original source: skypilot: https://github.com/skypilot-org/skypilot
# which is Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#     http://www.apache.org/licenses/LICENSE-2.0
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Annotations for public APIs."""

import functools
from typing import Callable, Literal

# Whether the current process is a SkyPilot API server process.
is_on_api_server = True
FUNCTIONS_NEED_RELOAD_CACHE = []


def client_api(func):
    """Mark a function as a client-side API.

    Code invoked by server-side functions will find annotations.is_on_api_server
    to be True, so they can have some server-side handling.
    """

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        global is_on_api_server
        is_on_api_server = False
        return func(*args, **kwargs)

    return wrapper


def lru_cache(
    scope: Literal['global', 'request'], *lru_cache_args, **lru_cache_kwargs
) -> Callable:
    """LRU cache decorator for functions.

    This decorator allows us to track which functions need to be reloaded for a
    new request using the scope argument.

    Args:
        scope: Whether the cache is global or request-specific, i.e. needs to be
            reloaded for a new request.
        lru_cache_args: Arguments for functools.lru_cache.
        lru_cache_kwargs: Keyword arguments for functools.lru_cache.
    """

    def decorator(func: Callable) -> Callable:
        if scope == 'global':
            return functools.lru_cache(*lru_cache_args, **lru_cache_kwargs)(func)
        else:
            cached_func = functools.lru_cache(*lru_cache_args, **lru_cache_kwargs)(func)
            FUNCTIONS_NEED_RELOAD_CACHE.append(cached_func)
            return cached_func

    return decorator
