import numpy as np
import msgpack
import os.path
import time

from haxballgym.game.common_values import INPUT_DOWN, INPUT_SHOOT, INPUT_LEFT, INPUT_RIGHT, INPUT_UP

# This is temporary, until we have a proper game recorder system
# In the meantime, we will use the same recording system than my JS version

def input_translate_js(actions: np.ndarray) -> int:
    result = 0
    
    if actions[0] == -1:
        result += INPUT_LEFT
    elif actions[0] == 1:
        result += INPUT_RIGHT
    
    if actions[1] == -1:
        result += INPUT_DOWN
    elif actions[1] == 1:
        result += INPUT_UP
        
    if actions[2]:
        result += INPUT_SHOOT
    
    return result
    

class GameRecorder():
    
    def __init__(self, game, folder_rec: str = ""):
        
        self.game = game
        self.folder_rec = folder_rec
        
        self.filename = ""
        self.recording = []
        self.player_info = []
        self.player_action = []
        self.options = []
        
    
    def generate_replay_name(self):
        """
        The replay name is generated by the following format:
        'HBR_<timestamp>_<score_red>-<score_blue>_<team_kickoff>.hbr'
        Time stamp is the current time in seconds since 01/01/1970.
        """
        file_name = f"HBR_{str(int(time.time()))}_{self.game.score.red}-{self.game.score.blue}_{self.options[0]}.hbr"
        return file_name
    
    
    def start(self):
        self.player_info = [[player.name, f"{player.id}", player.team] for player in self.game.players]
        self.player_action = [[] for _ in self.game.players]
        self.options = [self.game.team_kickoff * 8]
    
    
    def step(self, actions: np.ndarray):
        for i, action in enumerate(actions):
            self.player_action[i].append(input_translate_js(action))
    
    
    def stop(self, save: bool = True):
        players_list = [[info, action] for info, action in zip(self.player_info, self.player_action)]
        if len(self.options) > 0:
            self.recording = [self.options[0], players_list]
            self.filename = self.generate_replay_name()
            if save: self.save(self.filename)
        
        self.recording = []
        self.player_info = []
        self.player_action = []
        self.options = []
    
    
    def save(self, file_name: str) -> None:
        with open(os.path.join(os.path.curdir, self.folder_rec, file_name), 'wb+') as f:
            encoded_recording = msgpack.packb(self.recording)
            f.write(encoded_recording)
