"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gateway = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const application_gateway_1 = require("@cdktf/provider-azurerm/lib/application-gateway");
const key_vault_access_policy_1 = require("@cdktf/provider-azurerm/lib/key-vault-access-policy");
const user_assigned_identity_1 = require("@cdktf/provider-azurerm/lib/user-assigned-identity");
const vnet = require("../../azure-virtualnetwork");
const core_azure_1 = require("../../core-azure");
// Define the class for Azure Application Gateway
class Gateway extends core_azure_1.AzureResource {
    /**
     * Constructs a new Azure Application Gateway.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The ID of this construct.
     * @param props - The properties for configuring the Azure Application Gateway. The properties include:
     *                - `name`: Required. Unique name for the Application Gateway within Azure.
     *                - `location`: Required. Azure Region for deployment.
     *                - `resourceGroup`: Optional. Reference to the resource group for deployment.
     *                - `skuTier`: Required. SKU tier of the Application Gateway (e.g., Standard, WAF).
     *                - `skuSize`: Required. Size of the SKU for the Application Gateway.
     *                - `capacity`: Required. Capacity (instance count) of the Application Gateway.
     *                - `backendAddressPools`: Required. Backend address pools for the Application Gateway.
     *                - `backendHttpSettings`: Required. Backend HTTP settings for the Application Gateway.
     *                - `httpListeners`: Required. HTTP listeners for the Application Gateway.
     *                - `requestRoutingRules`: Required. Request routing rules for the Application Gateway.
     *                - `publicIpAddress`: Optional. Public IP address for the frontend.
     *                - `privateIpAddress`: Optional. Private IP address for the frontend.
     *                - `privateIpAddressAllocation`: Optional. Allocation method for the private IP (Static, Dynamic).
     *                - `frontendPorts`: Optional. Frontend ports for the Application Gateway.
     *                - `subnet`: Optional. Subnet for the Application Gateway.
     *                - `enableHttp2`: Optional. Flag to enable HTTP2.
     *                - `fipsEnabled`: Optional. Flag to enable FIPS-compliant algorithms.
     *                - `firewallPolicyId`: Optional. ID of the firewall policy.
     *                - `forceFirewallPolicyAssociation`: Optional. Flag to enforce association of the firewall policy.
     *                - `tags`: Optional. Tags for resource management.
     *                - Additional optional properties as described in `IGatewayProps` interface.
     *
     * Example usage:
     * ```typescript
     * new Gateway(this, 'appGateway1', {
     *   name: 'gatewayEast',
     *   resourceGroup: resourceGroup,
        location: "eastus",
        skuTier: "Standard_v2",
        skuSize: "Standard_v2",
        capacity: 2,
        publicIpAddress: publicIp,
        subnet: subnet,
        backendAddressPools: [
          { name: "backend-address-pool-1" },
          {
            name: "backend-address-pool-2",
            ipAddresses: ["10.1.0.4", "10.1.0.5", "10.1.0.6"],
          },
        ],
        httpListeners: [
          {
            name: "http-listener",
            frontendPortName: "80",
            frontendIpConfigurationName: "Public-frontend-ip-configuration",
            protocol: "Http",
          },
        ],
        backendHttpSettings: [
          {
            name: "backend-http-setting",
            port: 80,
            protocol: "Http",
            requestTimeout: 20,
            cookieBasedAffinity: "Disabled",
          },
        ],
        requestRoutingRules: [
          {
            name: "request-routing-rule-1",
            httpListenerName: "http-listener",
            priority: 1,
            backendAddressPoolName: "backend-address-pool-1",
            backendHttpSettingsName: "backend-http-setting",
            ruleType: "Basic",
          },
        ],
     * });
     * ```
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        // Define the identity
        let identity;
        // Check if Azure Key Vault is used and no identity is provided
        if (props.keyVault && !props.identity) {
            // Create a managed identity and add it to identityIds
            const managedIdentity = new user_assigned_identity_1.UserAssignedIdentity(this, "ManagedIdentity", {
                name: `mi-${props.name}`,
                resourceGroupName: this.resourceGroup.name,
                location: props.location,
            });
            identity = {
                identityIds: [managedIdentity.id],
                type: "UserAssigned",
            };
            new key_vault_access_policy_1.KeyVaultAccessPolicyA(this, "policy", {
                keyVaultId: props.keyVault.id,
                tenantId: props.tenantId || "",
                objectId: managedIdentity.principalId,
                secretPermissions: ["Get", "List"],
            });
        }
        const defaults = {
            subnetId: props.subnet?.id ||
                new vnet.Network(this, "vnet", {
                    resourceGroup: this.resourceGroup,
                }).subnets.default.id,
            identity: props.identity || identity,
        };
        // WAF configuration
        let wafConfiguration = props.wafConfiguration
            ? {
                enabled: props.wafConfiguration.enabled,
                firewallMode: props.wafConfiguration.firewallMode || "Detection",
                ruleSetVersion: props.wafConfiguration.ruleSetVersion || "3.0",
            }
            : undefined;
        // Dynamically create frontend IP configurations
        let frontendIpConfigs = [];
        // Public IP configuration
        if (props.publicIpAddress) {
            frontendIpConfigs.push({
                name: "Public-frontend-ip-configuration",
                publicIpAddressId: props.publicIpAddress.id,
            });
        }
        // Private IP configuration
        if (props.privateIpAddress || props.privateIpAddressAllocation) {
            frontendIpConfigs.push({
                name: "Private-frontend-ip-configuration",
                subnetId: defaults.subnetId,
                privateIpAddress: props.privateIpAddress,
                privateIpAddressAllocation: props.privateIpAddressAllocation,
            });
        }
        // If no frontend ports are provided, use default dummy frontend ip configuration
        if (frontendIpConfigs.length == 0) {
            frontendIpConfigs.push({
                name: "Dummy-frontend-ip-configuration",
            });
        }
        // Set default frontend ports if not provided
        const defaultFrontendPorts = [
            { name: "80", port: 80 },
            { name: "443", port: 443 },
        ];
        const frontendPorts = props.frontendPorts && props.frontendPorts.length > 0
            ? props.frontendPorts
            : defaultFrontendPorts;
        // Create the Application Gateway
        const apgw = new application_gateway_1.ApplicationGateway(this, "ApplicationGateway", {
            name: props.name,
            resourceGroupName: this.resourceGroup.name,
            location: props.location,
            sslCertificate: props.sslCertificate,
            sslPolicy: props.sslPolicy,
            sslProfile: props.sslProfile,
            authenticationCertificate: props.authenticationCertificate,
            autoscaleConfiguration: props.autoscaleConfiguration,
            customErrorConfiguration: props.customErrorConfiguration,
            redirectConfiguration: props.redirectConfiguration,
            rewriteRuleSet: props.rewriteRuleSet,
            privateLinkConfiguration: props.privateLinkConfiguration,
            wafConfiguration: wafConfiguration,
            sku: {
                name: props.skuSize,
                tier: props.skuTier,
                capacity: props.capacity,
            },
            gatewayIpConfiguration: [
                {
                    subnetId: defaults.subnetId,
                    name: `${props.name}-configuration`,
                },
            ],
            frontendPort: frontendPorts,
            frontendIpConfiguration: frontendIpConfigs,
            backendAddressPool: props.backendAddressPools,
            backendHttpSettings: props.backendHttpSettings,
            httpListener: props.httpListeners,
            urlPathMap: props.urlPathMap,
            trustedRootCertificate: props.trustedRootCertificate,
            requestRoutingRule: props.requestRoutingRules,
            probe: props.probe,
            identity: defaults.identity,
            zones: props.zones,
            tags: props.tags,
        });
        this.id = apgw.id;
    }
}
exports.Gateway = Gateway;
_a = JSII_RTTI_SYMBOL_1;
Gateway[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_applicationgateway.Gateway", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2F0ZXdheS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9henVyZS1hcHBsaWNhdGlvbmdhdGV3YXkvbGliL2dhdGV3YXkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSx5RkFBcUY7QUFHckYsaUdBQTRGO0FBSTVGLCtGQUEwRjtBQUUxRixtREFBbUQ7QUFDbkQsaURBQWlEO0FBeU1qRCxpREFBaUQ7QUFDakQsTUFBYSxPQUFRLFNBQVEsMEJBQWE7SUFLeEM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQTJFRztJQUVILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0I7UUFDNUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztRQUNuQixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVwRCxzQkFBc0I7UUFDdEIsSUFBSSxRQUFRLENBQUM7UUFFYiwrREFBK0Q7UUFDL0QsSUFBSSxLQUFLLENBQUMsUUFBUSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNyQyxzREFBc0Q7WUFDdEQsTUFBTSxlQUFlLEdBQUcsSUFBSSw2Q0FBb0IsQ0FDOUMsSUFBSSxFQUNKLGlCQUFpQixFQUNqQjtnQkFDRSxJQUFJLEVBQUUsTUFBTSxLQUFLLENBQUMsSUFBSSxFQUFFO2dCQUN4QixpQkFBaUIsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUk7Z0JBQzFDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTthQUN6QixDQUNGLENBQUM7WUFFRixRQUFRLEdBQUc7Z0JBQ1QsV0FBVyxFQUFFLENBQUMsZUFBZSxDQUFDLEVBQUUsQ0FBQztnQkFDakMsSUFBSSxFQUFFLGNBQWM7YUFDckIsQ0FBQztZQUVGLElBQUksK0NBQXFCLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtnQkFDeEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRTtnQkFDN0IsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLElBQUksRUFBRTtnQkFDOUIsUUFBUSxFQUFFLGVBQWUsQ0FBQyxXQUFXO2dCQUNyQyxpQkFBaUIsRUFBRSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUM7YUFDbkMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFNLFFBQVEsR0FBRztZQUNmLFFBQVEsRUFDTixLQUFLLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQ2hCLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO29CQUM3QixhQUFhLEVBQUUsSUFBSSxDQUFDLGFBQWE7aUJBQ2xDLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDdkIsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLElBQUksUUFBUTtTQUNyQyxDQUFDO1FBRUYsb0JBQW9CO1FBQ3BCLElBQUksZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQjtZQUMzQyxDQUFDLENBQUM7Z0JBQ0UsT0FBTyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPO2dCQUN2QyxZQUFZLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVksSUFBSSxXQUFXO2dCQUNoRSxjQUFjLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWMsSUFBSSxLQUFLO2FBQy9EO1lBQ0gsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLGdEQUFnRDtRQUNoRCxJQUFJLGlCQUFpQixHQUNuQixFQUFFLENBQUM7UUFFTCwwQkFBMEI7UUFDMUIsSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQ3pCLGlCQUFpQixDQUFDLElBQUksQ0FBQztnQkFDckIsSUFBSSxFQUFFLGtDQUFrQztnQkFDeEMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFO2FBQzVDLENBQUMsQ0FBQztTQUNKO1FBRUQsMkJBQTJCO1FBQzNCLElBQUksS0FBSyxDQUFDLGdCQUFnQixJQUFJLEtBQUssQ0FBQywwQkFBMEIsRUFBRTtZQUM5RCxpQkFBaUIsQ0FBQyxJQUFJLENBQUM7Z0JBQ3JCLElBQUksRUFBRSxtQ0FBbUM7Z0JBQ3pDLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtnQkFDM0IsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtnQkFDeEMsMEJBQTBCLEVBQUUsS0FBSyxDQUFDLDBCQUEwQjthQUM3RCxDQUFDLENBQUM7U0FDSjtRQUVELGlGQUFpRjtRQUNqRixJQUFJLGlCQUFpQixDQUFDLE1BQU0sSUFBSSxDQUFDLEVBQUU7WUFDakMsaUJBQWlCLENBQUMsSUFBSSxDQUFDO2dCQUNyQixJQUFJLEVBQUUsaUNBQWlDO2FBQ3hDLENBQUMsQ0FBQztTQUNKO1FBRUQsNkNBQTZDO1FBQzdDLE1BQU0sb0JBQW9CLEdBQUc7WUFDM0IsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDeEIsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUU7U0FDM0IsQ0FBQztRQUVGLE1BQU0sYUFBYSxHQUNqQixLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxHQUFHLENBQUM7WUFDbkQsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhO1lBQ3JCLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQztRQUUzQixpQ0FBaUM7UUFDakMsTUFBTSxJQUFJLEdBQUcsSUFBSSx3Q0FBa0IsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDOUQsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGlCQUFpQixFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSTtZQUMxQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIseUJBQXlCLEVBQUUsS0FBSyxDQUFDLHlCQUF5QjtZQUMxRCxzQkFBc0IsRUFBRSxLQUFLLENBQUMsc0JBQXNCO1lBQ3BELHdCQUF3QixFQUFFLEtBQUssQ0FBQyx3QkFBd0I7WUFDeEQscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtZQUNsRCxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLHdCQUF3QjtZQUN4RCxnQkFBZ0IsRUFBRSxnQkFBZ0I7WUFDbEMsR0FBRyxFQUFFO2dCQUNILElBQUksRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDbkIsSUFBSSxFQUFFLEtBQUssQ0FBQyxPQUFPO2dCQUNuQixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7YUFDekI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEI7b0JBQ0UsUUFBUSxFQUFFLFFBQVEsQ0FBQyxRQUFRO29CQUMzQixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsSUFBSSxnQkFBZ0I7aUJBQ3BDO2FBQ0Y7WUFDRCxZQUFZLEVBQUUsYUFBYTtZQUMzQix1QkFBdUIsRUFBRSxpQkFBaUI7WUFDMUMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzlDLFlBQVksRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNqQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxrQkFBa0IsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzdDLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSztZQUNsQixRQUFRLEVBQUUsUUFBUSxDQUFDLFFBQVE7WUFDM0IsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO1lBQ2xCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLENBQUM7SUFDcEIsQ0FBQzs7QUF4TkgsMEJBeU5DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXBwbGljYXRpb25HYXRld2F5IH0gZnJvbSBcIkBjZGt0Zi9wcm92aWRlci1henVyZXJtL2xpYi9hcHBsaWNhdGlvbi1nYXRld2F5XCI7XG5pbXBvcnQgKiBhcyBhemFwZ3cgZnJvbSBcIkBjZGt0Zi9wcm92aWRlci1henVyZXJtL2xpYi9hcHBsaWNhdGlvbi1nYXRld2F5XCI7XG5pbXBvcnQgeyBLZXlWYXVsdCB9IGZyb20gXCJAY2RrdGYvcHJvdmlkZXItYXp1cmVybS9saWIva2V5LXZhdWx0XCI7XG5pbXBvcnQgeyBLZXlWYXVsdEFjY2Vzc1BvbGljeUEgfSBmcm9tIFwiQGNka3RmL3Byb3ZpZGVyLWF6dXJlcm0vbGliL2tleS12YXVsdC1hY2Nlc3MtcG9saWN5XCI7XG5pbXBvcnQgeyBQdWJsaWNJcCB9IGZyb20gXCJAY2RrdGYvcHJvdmlkZXItYXp1cmVybS9saWIvcHVibGljLWlwXCI7XG5pbXBvcnQgeyBSZXNvdXJjZUdyb3VwIH0gZnJvbSBcIkBjZGt0Zi9wcm92aWRlci1henVyZXJtL2xpYi9yZXNvdXJjZS1ncm91cFwiO1xuaW1wb3J0IHsgU3VibmV0IH0gZnJvbSBcIkBjZGt0Zi9wcm92aWRlci1henVyZXJtL2xpYi9zdWJuZXRcIjtcbmltcG9ydCB7IFVzZXJBc3NpZ25lZElkZW50aXR5IH0gZnJvbSBcIkBjZGt0Zi9wcm92aWRlci1henVyZXJtL2xpYi91c2VyLWFzc2lnbmVkLWlkZW50aXR5XCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0ICogYXMgdm5ldCBmcm9tIFwiLi4vLi4vYXp1cmUtdmlydHVhbG5ldHdvcmtcIjtcbmltcG9ydCB7IEF6dXJlUmVzb3VyY2UgfSBmcm9tIFwiLi4vLi4vY29yZS1henVyZVwiO1xuXG4vLyBEZWZpbmUgdGhlIGludGVyZmFjZSBmb3IgQXBwbGljYXRpb24gR2F0ZXdheSBwcm9wZXJ0aWVzXG5leHBvcnQgaW50ZXJmYWNlIElHYXRld2F5UHJvcHMge1xuICAvKipcbiAgICogT3B0aW9uYWwgcHVibGljIElQIGFkZHJlc3MgZm9yIHRoZSBmcm9udGVuZCBvZiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICovXG4gIHB1YmxpY0lwQWRkcmVzcz86IFB1YmxpY0lwO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBwcml2YXRlIElQIGFkZHJlc3MgZm9yIHRoZSBmcm9udGVuZCBvZiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICovXG4gIHByaXZhdGVJcEFkZHJlc3M/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFsbG9jYXRpb24gbWV0aG9kIGZvciB0aGUgcHJpdmF0ZSBJUCBhZGRyZXNzIChlLmcuLCBTdGF0aWMsIER5bmFtaWMpLlxuICAgKi9cbiAgcHJpdmF0ZUlwQWRkcmVzc0FsbG9jYXRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gd2hlcmUgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkgd2lsbCBiZSBkZXBsb3llZCAoZS5nLiwgcmVnaW9uKS5cbiAgICovXG4gIHJlYWRvbmx5IGxvY2F0aW9uOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIHJlZmVyZW5jZSB0byB0aGUgcmVzb3VyY2UgZ3JvdXAgaW4gd2hpY2ggdG8gZGVwbG95IHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKiBJZiBub3QgcHJvdmlkZWQsIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5IHdpbGwgYmUgZGVwbG95ZWQgaW4gdGhlIGRlZmF1bHQgcmVzb3VyY2UgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZUdyb3VwPzogUmVzb3VyY2VHcm91cDtcblxuICAvKipcbiAgICogVGhlIFNLVSB0aWVyIG9mIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5IChlLmcuLCBTdGFuZGFyZCwgV0FGKS5cbiAgICovXG4gIHJlYWRvbmx5IHNrdVRpZXI6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHNpemUgb2YgdGhlIFNLVSBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkuXG4gICAqL1xuICByZWFkb25seSBza3VTaXplOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBjYXBhY2l0eSAoaW5zdGFuY2UgY291bnQpIG9mIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKi9cbiAgcmVhZG9ubHkgY2FwYWNpdHk6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGJhY2tlbmQgYWRkcmVzcyBwb29scyBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkuXG4gICAqL1xuICByZWFkb25seSBiYWNrZW5kQWRkcmVzc1Bvb2xzOiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5QmFja2VuZEFkZHJlc3NQb29sW107XG5cbiAgLyoqXG4gICAqIFRoZSBiYWNrZW5kIEhUVFAgc2V0dGluZ3MgZm9yIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKi9cbiAgcmVhZG9ubHkgYmFja2VuZEh0dHBTZXR0aW5nczogYXphcGd3LkFwcGxpY2F0aW9uR2F0ZXdheUJhY2tlbmRIdHRwU2V0dGluZ3NbXTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgZnJvbnRlbmQgcG9ydHMgZm9yIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKi9cbiAgcmVhZG9ubHkgZnJvbnRlbmRQb3J0cz86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlGcm9udGVuZFBvcnRbXTtcblxuICAvKipcbiAgICogVGhlIEhUVFAgbGlzdGVuZXJzIGZvciB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICovXG4gIHJlYWRvbmx5IGh0dHBMaXN0ZW5lcnM6IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlIdHRwTGlzdGVuZXJbXTtcblxuICAvKipcbiAgICogVGhlIHJlcXVlc3Qgcm91dGluZyBydWxlcyBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkuXG4gICAqL1xuICByZWFkb25seSByZXF1ZXN0Um91dGluZ1J1bGVzOiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5UmVxdWVzdFJvdXRpbmdSdWxlW107XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIHN1Ym5ldCBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkuXG4gICAqL1xuICByZWFkb25seSBzdWJuZXQ/OiBTdWJuZXQ7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIHRlbmFudCBJRCBmb3IgdXNlIHdpdGggS2V5IFZhdWx0LCBpZiBhcHBsaWNhYmxlLlxuICAgKi9cbiAgcmVhZG9ubHkgdGVuYW50SWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZsYWcgdG8gZW5hYmxlIEhUVFAyLlxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlSHR0cDI/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBGbGFnIHRvIGVuYWJsZSBGSVBTLWNvbXBsaWFudCBhbGdvcml0aG1zLlxuICAgKi9cbiAgcmVhZG9ubHkgZmlwc0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBJRCBvZiB0aGUgZmlyZXdhbGwgcG9saWN5LlxuICAgKi9cbiAgcmVhZG9ubHkgZmlyZXdhbGxQb2xpY3lJZD86IHN0cmluZztcblxuICAvKipcbiAgICogRmxhZyB0byBlbmZvcmNlIGFzc29jaWF0aW9uIG9mIHRoZSBmaXJld2FsbCBwb2xpY3kuXG4gICAqL1xuICByZWFkb25seSBmb3JjZUZpcmV3YWxsUG9saWN5QXNzb2NpYXRpb24/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCB0YWdzIGZvciB0aGUgQXBwbGljYXRpb24gR2F0ZXdheSByZXNvdXJjZS5cbiAgICovXG4gIHJlYWRvbmx5IHRhZ3M/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhdmFpbGFiaWxpdHkgem9uZXMgZm9yIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKi9cbiAgcmVhZG9ubHkgem9uZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgS2V5IFZhdWx0IHJlc291cmNlIGZvciBzdG9yaW5nIFNTTCBjZXJ0aWZpY2F0ZXMuXG4gICAqL1xuICByZWFkb25seSBrZXlWYXVsdD86IEtleVZhdWx0O1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhdXRoZW50aWNhdGlvbiBjZXJ0aWZpY2F0ZXMgZm9yIG11dHVhbCBhdXRoZW50aWNhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGF1dGhlbnRpY2F0aW9uQ2VydGlmaWNhdGU/OiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5QXV0aGVudGljYXRpb25DZXJ0aWZpY2F0ZVtdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhdXRvc2NhbGUgY29uZmlndXJhdGlvbiBmb3IgZHluYW1pY2FsbHkgYWRqdXN0aW5nIHRoZSBjYXBhY2l0eSBvZiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICovXG4gIHJlYWRvbmx5IGF1dG9zY2FsZUNvbmZpZ3VyYXRpb24/OiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5QXV0b3NjYWxlQ29uZmlndXJhdGlvbjtcblxuICAvKipcbiAgICogT3B0aW9uYWwgY3VzdG9tIGVycm9yIGNvbmZpZ3VyYXRpb25zIHRvIHNwZWNpZnkgY3VzdG9tIGVycm9yIHBhZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgY3VzdG9tRXJyb3JDb25maWd1cmF0aW9uPzogYXphcGd3LkFwcGxpY2F0aW9uR2F0ZXdheUN1c3RvbUVycm9yQ29uZmlndXJhdGlvbltdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBpZGVudGl0eSBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXksIHVzZWQgZm9yIGFjY2Vzc2luZyBvdGhlciBBenVyZSByZXNvdXJjZXMuXG4gICAqL1xuICByZWFkb25seSBpZGVudGl0eT86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlJZGVudGl0eTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgY29uZmlndXJhdGlvbnMgZm9yIGVuYWJsaW5nIFByaXZhdGUgTGluayBvbiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICovXG4gIHJlYWRvbmx5IHByaXZhdGVMaW5rQ29uZmlndXJhdGlvbj86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlQcml2YXRlTGlua0NvbmZpZ3VyYXRpb25bXTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgcHJvYmVzIGZvciBoZWFsdGggY2hlY2tzIG9mIHRoZSBiYWNrZW5kIEhUVFAgc2V0dGluZ3MuXG4gICAqL1xuICByZWFkb25seSBwcm9iZT86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlQcm9iZVtdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBjb25maWd1cmF0aW9ucyBmb3IgcmVkaXJlY3QgcnVsZXMuXG4gICAqL1xuICByZWFkb25seSByZWRpcmVjdENvbmZpZ3VyYXRpb24/OiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5UmVkaXJlY3RDb25maWd1cmF0aW9uW107XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIHJld3JpdGUgcnVsZSBzZXRzIGZvciBtb2RpZnlpbmcgSFRUUCByZXF1ZXN0IGFuZCByZXNwb25zZSBoZWFkZXJzIGFuZCBib2RpZXMuXG4gICAqL1xuICByZWFkb25seSByZXdyaXRlUnVsZVNldD86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlSZXdyaXRlUnVsZVNldFtdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBTU0wgY2VydGlmaWNhdGVzIGZvciBlbmFibGluZyBIVFRQUyBvbiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICovXG4gIHJlYWRvbmx5IHNzbENlcnRpZmljYXRlPzogYXphcGd3LkFwcGxpY2F0aW9uR2F0ZXdheVNzbENlcnRpZmljYXRlW107XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIFNTTCBwb2xpY3kgY29uZmlndXJhdGlvbnMsIGRlZmluaW5nIHRoZSBwcm90b2NvbCBhbmQgY2lwaGVyIHN1aXRlcyB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3NsUG9saWN5PzogYXphcGd3LkFwcGxpY2F0aW9uR2F0ZXdheVNzbFBvbGljeTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgU1NMIHByb2ZpbGVzIGZvciBtYW5hZ2luZyBTU0wgdGVybWluYXRpb24gYW5kIHBvbGljeSBzZXR0aW5ncy5cbiAgICovXG4gIHJlYWRvbmx5IHNzbFByb2ZpbGU/OiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5U3NsUHJvZmlsZVtdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCB0aW1lb3V0IHNldHRpbmdzIGZvciB0aGUgQXBwbGljYXRpb24gR2F0ZXdheSByZXNvdXJjZXMuXG4gICAqL1xuICByZWFkb25seSB0aW1lb3V0cz86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlUaW1lb3V0cztcblxuICAvKipcbiAgICogT3B0aW9uYWwgdHJ1c3RlZCBjbGllbnQgY2VydGlmaWNhdGVzIGZvciBtdXR1YWwgYXV0aGVudGljYXRpb24uXG4gICAqL1xuICByZWFkb25seSB0cnVzdGVkQ2xpZW50Q2VydGlmaWNhdGU/OiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5VHJ1c3RlZENsaWVudENlcnRpZmljYXRlW107XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIHRydXN0ZWQgcm9vdCBjZXJ0aWZpY2F0ZXMgZm9yIGJhY2tlbmQgYXV0aGVudGljYXRpb24uXG4gICAqL1xuICByZWFkb25seSB0cnVzdGVkUm9vdENlcnRpZmljYXRlPzogYXphcGd3LkFwcGxpY2F0aW9uR2F0ZXdheVRydXN0ZWRSb290Q2VydGlmaWNhdGVbXTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgVVJMIHBhdGggbWFwIGZvciByb3V0aW5nIGJhc2VkIG9uIFVSTCBwYXRocy5cbiAgICovXG4gIHJlYWRvbmx5IHVybFBhdGhNYXA/OiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5VXJsUGF0aE1hcFtdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBXZWIgQXBwbGljYXRpb24gRmlyZXdhbGwgKFdBRikgY29uZmlndXJhdGlvbiB0byBwcm92aWRlIGVuaGFuY2VkIHNlY3VyaXR5LlxuICAgKi9cbiAgcmVhZG9ubHkgd2FmQ29uZmlndXJhdGlvbj86IGF6YXBndy5BcHBsaWNhdGlvbkdhdGV3YXlXYWZDb25maWd1cmF0aW9uO1xufVxuXG4vLyBEZWZpbmUgdGhlIGNsYXNzIGZvciBBenVyZSBBcHBsaWNhdGlvbiBHYXRld2F5XG5leHBvcnQgY2xhc3MgR2F0ZXdheSBleHRlbmRzIEF6dXJlUmVzb3VyY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvcHM6IElHYXRld2F5UHJvcHM7XG4gIHB1YmxpYyByZXNvdXJjZUdyb3VwOiBSZXNvdXJjZUdyb3VwO1xuICBwdWJsaWMgaWQ6IHN0cmluZztcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhIG5ldyBBenVyZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgLSBUaGUgc2NvcGUgaW4gd2hpY2ggdG8gZGVmaW5lIHRoaXMgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgSUQgb2YgdGhpcyBjb25zdHJ1Y3QuXG4gICAqIEBwYXJhbSBwcm9wcyAtIFRoZSBwcm9wZXJ0aWVzIGZvciBjb25maWd1cmluZyB0aGUgQXp1cmUgQXBwbGljYXRpb24gR2F0ZXdheS4gVGhlIHByb3BlcnRpZXMgaW5jbHVkZTpcbiAgICogICAgICAgICAgICAgICAgLSBgbmFtZWA6IFJlcXVpcmVkLiBVbmlxdWUgbmFtZSBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkgd2l0aGluIEF6dXJlLlxuICAgKiAgICAgICAgICAgICAgICAtIGBsb2NhdGlvbmA6IFJlcXVpcmVkLiBBenVyZSBSZWdpb24gZm9yIGRlcGxveW1lbnQuXG4gICAqICAgICAgICAgICAgICAgIC0gYHJlc291cmNlR3JvdXBgOiBPcHRpb25hbC4gUmVmZXJlbmNlIHRvIHRoZSByZXNvdXJjZSBncm91cCBmb3IgZGVwbG95bWVudC5cbiAgICogICAgICAgICAgICAgICAgLSBgc2t1VGllcmA6IFJlcXVpcmVkLiBTS1UgdGllciBvZiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheSAoZS5nLiwgU3RhbmRhcmQsIFdBRikuXG4gICAqICAgICAgICAgICAgICAgIC0gYHNrdVNpemVgOiBSZXF1aXJlZC4gU2l6ZSBvZiB0aGUgU0tVIGZvciB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICogICAgICAgICAgICAgICAgLSBgY2FwYWNpdHlgOiBSZXF1aXJlZC4gQ2FwYWNpdHkgKGluc3RhbmNlIGNvdW50KSBvZiB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICogICAgICAgICAgICAgICAgLSBgYmFja2VuZEFkZHJlc3NQb29sc2A6IFJlcXVpcmVkLiBCYWNrZW5kIGFkZHJlc3MgcG9vbHMgZm9yIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKiAgICAgICAgICAgICAgICAtIGBiYWNrZW5kSHR0cFNldHRpbmdzYDogUmVxdWlyZWQuIEJhY2tlbmQgSFRUUCBzZXR0aW5ncyBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkuXG4gICAqICAgICAgICAgICAgICAgIC0gYGh0dHBMaXN0ZW5lcnNgOiBSZXF1aXJlZC4gSFRUUCBsaXN0ZW5lcnMgZm9yIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKiAgICAgICAgICAgICAgICAtIGByZXF1ZXN0Um91dGluZ1J1bGVzYDogUmVxdWlyZWQuIFJlcXVlc3Qgcm91dGluZyBydWxlcyBmb3IgdGhlIEFwcGxpY2F0aW9uIEdhdGV3YXkuXG4gICAqICAgICAgICAgICAgICAgIC0gYHB1YmxpY0lwQWRkcmVzc2A6IE9wdGlvbmFsLiBQdWJsaWMgSVAgYWRkcmVzcyBmb3IgdGhlIGZyb250ZW5kLlxuICAgKiAgICAgICAgICAgICAgICAtIGBwcml2YXRlSXBBZGRyZXNzYDogT3B0aW9uYWwuIFByaXZhdGUgSVAgYWRkcmVzcyBmb3IgdGhlIGZyb250ZW5kLlxuICAgKiAgICAgICAgICAgICAgICAtIGBwcml2YXRlSXBBZGRyZXNzQWxsb2NhdGlvbmA6IE9wdGlvbmFsLiBBbGxvY2F0aW9uIG1ldGhvZCBmb3IgdGhlIHByaXZhdGUgSVAgKFN0YXRpYywgRHluYW1pYykuXG4gICAqICAgICAgICAgICAgICAgIC0gYGZyb250ZW5kUG9ydHNgOiBPcHRpb25hbC4gRnJvbnRlbmQgcG9ydHMgZm9yIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5LlxuICAgKiAgICAgICAgICAgICAgICAtIGBzdWJuZXRgOiBPcHRpb25hbC4gU3VibmV0IGZvciB0aGUgQXBwbGljYXRpb24gR2F0ZXdheS5cbiAgICogICAgICAgICAgICAgICAgLSBgZW5hYmxlSHR0cDJgOiBPcHRpb25hbC4gRmxhZyB0byBlbmFibGUgSFRUUDIuXG4gICAqICAgICAgICAgICAgICAgIC0gYGZpcHNFbmFibGVkYDogT3B0aW9uYWwuIEZsYWcgdG8gZW5hYmxlIEZJUFMtY29tcGxpYW50IGFsZ29yaXRobXMuXG4gICAqICAgICAgICAgICAgICAgIC0gYGZpcmV3YWxsUG9saWN5SWRgOiBPcHRpb25hbC4gSUQgb2YgdGhlIGZpcmV3YWxsIHBvbGljeS5cbiAgICogICAgICAgICAgICAgICAgLSBgZm9yY2VGaXJld2FsbFBvbGljeUFzc29jaWF0aW9uYDogT3B0aW9uYWwuIEZsYWcgdG8gZW5mb3JjZSBhc3NvY2lhdGlvbiBvZiB0aGUgZmlyZXdhbGwgcG9saWN5LlxuICAgKiAgICAgICAgICAgICAgICAtIGB0YWdzYDogT3B0aW9uYWwuIFRhZ3MgZm9yIHJlc291cmNlIG1hbmFnZW1lbnQuXG4gICAqICAgICAgICAgICAgICAgIC0gQWRkaXRpb25hbCBvcHRpb25hbCBwcm9wZXJ0aWVzIGFzIGRlc2NyaWJlZCBpbiBgSUdhdGV3YXlQcm9wc2AgaW50ZXJmYWNlLlxuICAgKlxuICAgKiBFeGFtcGxlIHVzYWdlOlxuICAgKiBgYGB0eXBlc2NyaXB0XG4gICAqIG5ldyBHYXRld2F5KHRoaXMsICdhcHBHYXRld2F5MScsIHtcbiAgICogICBuYW1lOiAnZ2F0ZXdheUVhc3QnLFxuICAgKiAgIHJlc291cmNlR3JvdXA6IHJlc291cmNlR3JvdXAsXG4gICAgICBsb2NhdGlvbjogXCJlYXN0dXNcIixcbiAgICAgIHNrdVRpZXI6IFwiU3RhbmRhcmRfdjJcIixcbiAgICAgIHNrdVNpemU6IFwiU3RhbmRhcmRfdjJcIixcbiAgICAgIGNhcGFjaXR5OiAyLFxuICAgICAgcHVibGljSXBBZGRyZXNzOiBwdWJsaWNJcCxcbiAgICAgIHN1Ym5ldDogc3VibmV0LFxuICAgICAgYmFja2VuZEFkZHJlc3NQb29sczogW1xuICAgICAgICB7IG5hbWU6IFwiYmFja2VuZC1hZGRyZXNzLXBvb2wtMVwiIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcImJhY2tlbmQtYWRkcmVzcy1wb29sLTJcIixcbiAgICAgICAgICBpcEFkZHJlc3NlczogW1wiMTAuMS4wLjRcIiwgXCIxMC4xLjAuNVwiLCBcIjEwLjEuMC42XCJdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIGh0dHBMaXN0ZW5lcnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiaHR0cC1saXN0ZW5lclwiLFxuICAgICAgICAgIGZyb250ZW5kUG9ydE5hbWU6IFwiODBcIixcbiAgICAgICAgICBmcm9udGVuZElwQ29uZmlndXJhdGlvbk5hbWU6IFwiUHVibGljLWZyb250ZW5kLWlwLWNvbmZpZ3VyYXRpb25cIixcbiAgICAgICAgICBwcm90b2NvbDogXCJIdHRwXCIsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgYmFja2VuZEh0dHBTZXR0aW5nczogW1xuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogXCJiYWNrZW5kLWh0dHAtc2V0dGluZ1wiLFxuICAgICAgICAgIHBvcnQ6IDgwLFxuICAgICAgICAgIHByb3RvY29sOiBcIkh0dHBcIixcbiAgICAgICAgICByZXF1ZXN0VGltZW91dDogMjAsXG4gICAgICAgICAgY29va2llQmFzZWRBZmZpbml0eTogXCJEaXNhYmxlZFwiLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHJlcXVlc3RSb3V0aW5nUnVsZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwicmVxdWVzdC1yb3V0aW5nLXJ1bGUtMVwiLFxuICAgICAgICAgIGh0dHBMaXN0ZW5lck5hbWU6IFwiaHR0cC1saXN0ZW5lclwiLFxuICAgICAgICAgIHByaW9yaXR5OiAxLFxuICAgICAgICAgIGJhY2tlbmRBZGRyZXNzUG9vbE5hbWU6IFwiYmFja2VuZC1hZGRyZXNzLXBvb2wtMVwiLFxuICAgICAgICAgIGJhY2tlbmRIdHRwU2V0dGluZ3NOYW1lOiBcImJhY2tlbmQtaHR0cC1zZXR0aW5nXCIsXG4gICAgICAgICAgcnVsZVR5cGU6IFwiQmFzaWNcIixcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAqIH0pO1xuICAgKiBgYGBcbiAgICovXG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IElHYXRld2F5UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgIHRoaXMucmVzb3VyY2VHcm91cCA9IHRoaXMuc2V0dXBSZXNvdXJjZUdyb3VwKHByb3BzKTtcblxuICAgIC8vIERlZmluZSB0aGUgaWRlbnRpdHlcbiAgICBsZXQgaWRlbnRpdHk7XG5cbiAgICAvLyBDaGVjayBpZiBBenVyZSBLZXkgVmF1bHQgaXMgdXNlZCBhbmQgbm8gaWRlbnRpdHkgaXMgcHJvdmlkZWRcbiAgICBpZiAocHJvcHMua2V5VmF1bHQgJiYgIXByb3BzLmlkZW50aXR5KSB7XG4gICAgICAvLyBDcmVhdGUgYSBtYW5hZ2VkIGlkZW50aXR5IGFuZCBhZGQgaXQgdG8gaWRlbnRpdHlJZHNcbiAgICAgIGNvbnN0IG1hbmFnZWRJZGVudGl0eSA9IG5ldyBVc2VyQXNzaWduZWRJZGVudGl0eShcbiAgICAgICAgdGhpcyxcbiAgICAgICAgXCJNYW5hZ2VkSWRlbnRpdHlcIixcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IGBtaS0ke3Byb3BzLm5hbWV9YCxcbiAgICAgICAgICByZXNvdXJjZUdyb3VwTmFtZTogdGhpcy5yZXNvdXJjZUdyb3VwLm5hbWUsXG4gICAgICAgICAgbG9jYXRpb246IHByb3BzLmxvY2F0aW9uLFxuICAgICAgICB9LFxuICAgICAgKTtcblxuICAgICAgaWRlbnRpdHkgPSB7XG4gICAgICAgIGlkZW50aXR5SWRzOiBbbWFuYWdlZElkZW50aXR5LmlkXSxcbiAgICAgICAgdHlwZTogXCJVc2VyQXNzaWduZWRcIixcbiAgICAgIH07XG5cbiAgICAgIG5ldyBLZXlWYXVsdEFjY2Vzc1BvbGljeUEodGhpcywgXCJwb2xpY3lcIiwge1xuICAgICAgICBrZXlWYXVsdElkOiBwcm9wcy5rZXlWYXVsdC5pZCxcbiAgICAgICAgdGVuYW50SWQ6IHByb3BzLnRlbmFudElkIHx8IFwiXCIsXG4gICAgICAgIG9iamVjdElkOiBtYW5hZ2VkSWRlbnRpdHkucHJpbmNpcGFsSWQsXG4gICAgICAgIHNlY3JldFBlcm1pc3Npb25zOiBbXCJHZXRcIiwgXCJMaXN0XCJdLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3QgZGVmYXVsdHMgPSB7XG4gICAgICBzdWJuZXRJZDpcbiAgICAgICAgcHJvcHMuc3VibmV0Py5pZCB8fFxuICAgICAgICBuZXcgdm5ldC5OZXR3b3JrKHRoaXMsIFwidm5ldFwiLCB7XG4gICAgICAgICAgcmVzb3VyY2VHcm91cDogdGhpcy5yZXNvdXJjZUdyb3VwLFxuICAgICAgICB9KS5zdWJuZXRzLmRlZmF1bHQuaWQsXG4gICAgICBpZGVudGl0eTogcHJvcHMuaWRlbnRpdHkgfHwgaWRlbnRpdHksXG4gICAgfTtcblxuICAgIC8vIFdBRiBjb25maWd1cmF0aW9uXG4gICAgbGV0IHdhZkNvbmZpZ3VyYXRpb24gPSBwcm9wcy53YWZDb25maWd1cmF0aW9uXG4gICAgICA/IHtcbiAgICAgICAgICBlbmFibGVkOiBwcm9wcy53YWZDb25maWd1cmF0aW9uLmVuYWJsZWQsXG4gICAgICAgICAgZmlyZXdhbGxNb2RlOiBwcm9wcy53YWZDb25maWd1cmF0aW9uLmZpcmV3YWxsTW9kZSB8fCBcIkRldGVjdGlvblwiLFxuICAgICAgICAgIHJ1bGVTZXRWZXJzaW9uOiBwcm9wcy53YWZDb25maWd1cmF0aW9uLnJ1bGVTZXRWZXJzaW9uIHx8IFwiMy4wXCIsXG4gICAgICAgIH1cbiAgICAgIDogdW5kZWZpbmVkO1xuXG4gICAgLy8gRHluYW1pY2FsbHkgY3JlYXRlIGZyb250ZW5kIElQIGNvbmZpZ3VyYXRpb25zXG4gICAgbGV0IGZyb250ZW5kSXBDb25maWdzOiBhemFwZ3cuQXBwbGljYXRpb25HYXRld2F5RnJvbnRlbmRJcENvbmZpZ3VyYXRpb25bXSA9XG4gICAgICBbXTtcblxuICAgIC8vIFB1YmxpYyBJUCBjb25maWd1cmF0aW9uXG4gICAgaWYgKHByb3BzLnB1YmxpY0lwQWRkcmVzcykge1xuICAgICAgZnJvbnRlbmRJcENvbmZpZ3MucHVzaCh7XG4gICAgICAgIG5hbWU6IFwiUHVibGljLWZyb250ZW5kLWlwLWNvbmZpZ3VyYXRpb25cIixcbiAgICAgICAgcHVibGljSXBBZGRyZXNzSWQ6IHByb3BzLnB1YmxpY0lwQWRkcmVzcy5pZCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIFByaXZhdGUgSVAgY29uZmlndXJhdGlvblxuICAgIGlmIChwcm9wcy5wcml2YXRlSXBBZGRyZXNzIHx8IHByb3BzLnByaXZhdGVJcEFkZHJlc3NBbGxvY2F0aW9uKSB7XG4gICAgICBmcm9udGVuZElwQ29uZmlncy5wdXNoKHtcbiAgICAgICAgbmFtZTogXCJQcml2YXRlLWZyb250ZW5kLWlwLWNvbmZpZ3VyYXRpb25cIixcbiAgICAgICAgc3VibmV0SWQ6IGRlZmF1bHRzLnN1Ym5ldElkLFxuICAgICAgICBwcml2YXRlSXBBZGRyZXNzOiBwcm9wcy5wcml2YXRlSXBBZGRyZXNzLFxuICAgICAgICBwcml2YXRlSXBBZGRyZXNzQWxsb2NhdGlvbjogcHJvcHMucHJpdmF0ZUlwQWRkcmVzc0FsbG9jYXRpb24sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBJZiBubyBmcm9udGVuZCBwb3J0cyBhcmUgcHJvdmlkZWQsIHVzZSBkZWZhdWx0IGR1bW15IGZyb250ZW5kIGlwIGNvbmZpZ3VyYXRpb25cbiAgICBpZiAoZnJvbnRlbmRJcENvbmZpZ3MubGVuZ3RoID09IDApIHtcbiAgICAgIGZyb250ZW5kSXBDb25maWdzLnB1c2goe1xuICAgICAgICBuYW1lOiBcIkR1bW15LWZyb250ZW5kLWlwLWNvbmZpZ3VyYXRpb25cIixcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIFNldCBkZWZhdWx0IGZyb250ZW5kIHBvcnRzIGlmIG5vdCBwcm92aWRlZFxuICAgIGNvbnN0IGRlZmF1bHRGcm9udGVuZFBvcnRzID0gW1xuICAgICAgeyBuYW1lOiBcIjgwXCIsIHBvcnQ6IDgwIH0sXG4gICAgICB7IG5hbWU6IFwiNDQzXCIsIHBvcnQ6IDQ0MyB9LFxuICAgIF07XG5cbiAgICBjb25zdCBmcm9udGVuZFBvcnRzID1cbiAgICAgIHByb3BzLmZyb250ZW5kUG9ydHMgJiYgcHJvcHMuZnJvbnRlbmRQb3J0cy5sZW5ndGggPiAwXG4gICAgICAgID8gcHJvcHMuZnJvbnRlbmRQb3J0c1xuICAgICAgICA6IGRlZmF1bHRGcm9udGVuZFBvcnRzO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBBcHBsaWNhdGlvbiBHYXRld2F5XG4gICAgY29uc3QgYXBndyA9IG5ldyBBcHBsaWNhdGlvbkdhdGV3YXkodGhpcywgXCJBcHBsaWNhdGlvbkdhdGV3YXlcIiwge1xuICAgICAgbmFtZTogcHJvcHMubmFtZSxcbiAgICAgIHJlc291cmNlR3JvdXBOYW1lOiB0aGlzLnJlc291cmNlR3JvdXAubmFtZSxcbiAgICAgIGxvY2F0aW9uOiBwcm9wcy5sb2NhdGlvbixcbiAgICAgIHNzbENlcnRpZmljYXRlOiBwcm9wcy5zc2xDZXJ0aWZpY2F0ZSxcbiAgICAgIHNzbFBvbGljeTogcHJvcHMuc3NsUG9saWN5LFxuICAgICAgc3NsUHJvZmlsZTogcHJvcHMuc3NsUHJvZmlsZSxcbiAgICAgIGF1dGhlbnRpY2F0aW9uQ2VydGlmaWNhdGU6IHByb3BzLmF1dGhlbnRpY2F0aW9uQ2VydGlmaWNhdGUsXG4gICAgICBhdXRvc2NhbGVDb25maWd1cmF0aW9uOiBwcm9wcy5hdXRvc2NhbGVDb25maWd1cmF0aW9uLFxuICAgICAgY3VzdG9tRXJyb3JDb25maWd1cmF0aW9uOiBwcm9wcy5jdXN0b21FcnJvckNvbmZpZ3VyYXRpb24sXG4gICAgICByZWRpcmVjdENvbmZpZ3VyYXRpb246IHByb3BzLnJlZGlyZWN0Q29uZmlndXJhdGlvbixcbiAgICAgIHJld3JpdGVSdWxlU2V0OiBwcm9wcy5yZXdyaXRlUnVsZVNldCxcbiAgICAgIHByaXZhdGVMaW5rQ29uZmlndXJhdGlvbjogcHJvcHMucHJpdmF0ZUxpbmtDb25maWd1cmF0aW9uLFxuICAgICAgd2FmQ29uZmlndXJhdGlvbjogd2FmQ29uZmlndXJhdGlvbixcbiAgICAgIHNrdToge1xuICAgICAgICBuYW1lOiBwcm9wcy5za3VTaXplLFxuICAgICAgICB0aWVyOiBwcm9wcy5za3VUaWVyLFxuICAgICAgICBjYXBhY2l0eTogcHJvcHMuY2FwYWNpdHksXG4gICAgICB9LFxuICAgICAgZ2F0ZXdheUlwQ29uZmlndXJhdGlvbjogW1xuICAgICAgICB7XG4gICAgICAgICAgc3VibmV0SWQ6IGRlZmF1bHRzLnN1Ym5ldElkLFxuICAgICAgICAgIG5hbWU6IGAke3Byb3BzLm5hbWV9LWNvbmZpZ3VyYXRpb25gLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIGZyb250ZW5kUG9ydDogZnJvbnRlbmRQb3J0cyxcbiAgICAgIGZyb250ZW5kSXBDb25maWd1cmF0aW9uOiBmcm9udGVuZElwQ29uZmlncyxcbiAgICAgIGJhY2tlbmRBZGRyZXNzUG9vbDogcHJvcHMuYmFja2VuZEFkZHJlc3NQb29scyxcbiAgICAgIGJhY2tlbmRIdHRwU2V0dGluZ3M6IHByb3BzLmJhY2tlbmRIdHRwU2V0dGluZ3MsXG4gICAgICBodHRwTGlzdGVuZXI6IHByb3BzLmh0dHBMaXN0ZW5lcnMsXG4gICAgICB1cmxQYXRoTWFwOiBwcm9wcy51cmxQYXRoTWFwLFxuICAgICAgdHJ1c3RlZFJvb3RDZXJ0aWZpY2F0ZTogcHJvcHMudHJ1c3RlZFJvb3RDZXJ0aWZpY2F0ZSxcbiAgICAgIHJlcXVlc3RSb3V0aW5nUnVsZTogcHJvcHMucmVxdWVzdFJvdXRpbmdSdWxlcyxcbiAgICAgIHByb2JlOiBwcm9wcy5wcm9iZSxcbiAgICAgIGlkZW50aXR5OiBkZWZhdWx0cy5pZGVudGl0eSxcbiAgICAgIHpvbmVzOiBwcm9wcy56b25lcyxcbiAgICAgIHRhZ3M6IHByb3BzLnRhZ3MsXG4gICAgfSk7XG5cbiAgICB0aGlzLmlkID0gYXBndy5pZDtcbiAgfVxufVxuIl19