"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("cdktf/lib/testing/adapters/jest");
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const log_analytics_workspace_1 = require("@cdktf/provider-azurerm/lib/log-analytics-workspace");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const public_ip_1 = require("@cdktf/provider-azurerm/lib/public-ip");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const subnet_1 = require("@cdktf/provider-azurerm/lib/subnet");
const virtual_network_1 = require("@cdktf/provider-azurerm/lib/virtual-network");
const cdktf_1 = require("cdktf");
const kv = require("../../azure-keyvault");
const testing_1 = require("../../testing");
const util = require("../../util/azureTenantIdHelpers");
const randomName_1 = require("../../util/randomName");
const apgw = require("../lib");
describe("Example of deploying Application Gateway", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "eastus",
        });
        const logAnalyticsWorkspace = new log_analytics_workspace_1.LogAnalyticsWorkspace(stack, "log_analytics", {
            location: "eastus",
            name: `la-${randomName}`,
            resourceGroupName: resourceGroup.name,
        });
        const vnet = new virtual_network_1.VirtualNetwork(stack, "vnet", {
            name: `vnet-${randomName}`,
            location: "eastus",
            resourceGroupName: resourceGroup.name,
            addressSpace: ["10.0.0.0/24"],
        });
        const subnet = new subnet_1.Subnet(stack, "subnet", {
            name: `subnet-${randomName}`,
            resourceGroupName: resourceGroup.name,
            virtualNetworkName: vnet.name,
            addressPrefixes: ["10.0.0.0/24"],
        });
        const publicIp = new public_ip_1.PublicIp(stack, "publicIp", {
            name: "testip",
            location: "eastus",
            resourceGroupName: resourceGroup.name,
            allocationMethod: "Static",
            sku: "Standard",
        });
        const publicIpwaf = new public_ip_1.PublicIp(stack, "publicIpwaf", {
            name: "testipwaf",
            location: "eastus",
            resourceGroupName: resourceGroup.name,
            allocationMethod: "Static",
            sku: "Standard",
        });
        const azureKeyVault = new kv.Vault(stack, "keyvault", {
            name: `kv-${randomName}`,
            resourceGroup: resourceGroup,
            location: "eastus",
            tenantId: util.getAzureTenantId(),
        });
        azureKeyVault.grantCustomAccess(clientConfig.objectId, {
            secretPermissions: ["Get", "List", "Set", "Delete"],
            keyPermissions: [
                "Backup",
                "Create",
                "Decrypt",
                "Delete",
                "Encrypt",
                "Get",
                "Import",
                "List",
                "Purge",
                "Recover",
                "Restore",
                "Sign",
                "UnwrapKey",
                "Update",
                "Verify",
                "WrapKey",
                "Release",
                "Rotate",
                "GetRotationPolicy",
                "SetRotationPolicy",
            ],
            certificatePermissions: [
                "Get",
                "List",
                "Create",
                "Delete",
                "GetIssuers",
                "ManageIssuers",
            ],
        });
        const cert = azureKeyVault.addSelfSignedCert("ssl-cert", "CN=contoso.com", [
            "internal.contoso.com",
        ]);
        const applicationGateway = new apgw.Gateway(stack, "appgw", {
            name: `apgw-${randomName}`,
            resourceGroup: resourceGroup,
            location: "eastus",
            skuTier: "Standard_v2",
            skuSize: "Standard_v2",
            capacity: 2,
            publicIpAddress: publicIp,
            subnet: subnet,
            backendAddressPools: [
                { name: "backend-address-pool-1" },
                {
                    name: "backend-address-pool-2",
                    ipAddresses: ["10.1.0.4", "10.1.0.5", "10.1.0.6"],
                },
            ],
            httpListeners: [
                {
                    name: "http-listener",
                    frontendPortName: "80",
                    frontendIpConfigurationName: "Public-frontend-ip-configuration",
                    protocol: "Http",
                },
            ],
            backendHttpSettings: [
                {
                    name: "backend-http-setting",
                    port: 80,
                    protocol: "Http",
                    requestTimeout: 20,
                    cookieBasedAffinity: "Disabled",
                },
            ],
            requestRoutingRules: [
                {
                    name: "request-routing-rule-1",
                    httpListenerName: "http-listener",
                    priority: 1,
                    backendAddressPoolName: "backend-address-pool-1",
                    backendHttpSettingsName: "backend-http-setting",
                    ruleType: "Basic",
                },
            ],
        });
        new apgw.Gateway(stack, "appgw_waf", {
            name: `apgw-${randomName}waf`,
            resourceGroup: resourceGroup,
            location: "eastus",
            skuTier: "WAF_v2",
            skuSize: "WAF_v2",
            capacity: 2,
            publicIpAddress: publicIpwaf,
            tenantId: util.getAzureTenantId(),
            subnet: subnet,
            keyVault: azureKeyVault.keyVault,
            backendAddressPools: [
                { name: "backend-address-pool-1" },
                {
                    name: "backend-address-pool-2",
                    ipAddresses: ["10.1.0.4", "10.1.0.5", "10.1.0.6"],
                },
            ],
            httpListeners: [
                {
                    name: "http-listener",
                    frontendPortName: "443",
                    frontendIpConfigurationName: "Public-frontend-ip-configuration",
                    protocol: "Https",
                    sslCertificateName: "internal.contoso.com",
                },
            ],
            backendHttpSettings: [
                {
                    name: "backend-http-setting",
                    port: 80,
                    protocol: "Http",
                    requestTimeout: 20,
                    cookieBasedAffinity: "Disabled",
                },
            ],
            requestRoutingRules: [
                {
                    name: "request-routing-rule-1",
                    httpListenerName: "http-listener",
                    priority: 1,
                    backendAddressPoolName: "backend-address-pool-1",
                    backendHttpSettingsName: "backend-http-setting",
                    ruleType: "Basic",
                },
            ],
            wafConfiguration: {
                enabled: true,
                firewallMode: "Detection",
                ruleSetType: "OWASP",
                ruleSetVersion: "3.0",
                disabledRuleGroup: [],
                fileUploadLimitMb: 100,
                requestBodyCheck: false,
                maxRequestBodySizeKb: 32,
            },
            probe: [
                {
                    name: "probe-8080",
                    interval: 3,
                    protocol: "Http",
                    path: "/status",
                    timeout: 15,
                    unhealthyThreshold: 3,
                    port: 8080,
                    host: "test",
                },
            ],
            redirectConfiguration: [
                {
                    name: "http-redirect",
                    redirectType: "Permanent",
                    targetListenerName: "http-listener",
                    includePath: true,
                    includeQueryString: true,
                },
            ],
            urlPathMap: [
                {
                    name: "url-path-map",
                    defaultRedirectConfigurationName: "http-redirect",
                    pathRule: [
                        {
                            name: "path-rule-01",
                            redirectConfigurationName: "http-redirect",
                            paths: ["/helloworld"],
                        },
                    ],
                },
            ],
            sslCertificate: [
                {
                    name: "internal.contoso.com",
                    keyVaultSecretId: cert.secretId,
                },
            ],
        });
        //Diag Settings
        applicationGateway.addDiagSettings({
            name: "diagsettings",
            logAnalyticsWorkspaceId: logAnalyticsWorkspace.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        //RBAC
        applicationGateway.addAccess(clientConfig.objectId, "Contributor");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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