"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const key_vault_1 = require("@cdktf/provider-azurerm/lib/key-vault");
const log_analytics_workspace_1 = require("@cdktf/provider-azurerm/lib/log-analytics-workspace");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const cdktf_1 = require("cdktf");
const testing_1 = require("../../testing");
const util = require("../../util/azureTenantIdHelpers");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
const appi = require("../lib");
describe("Example of deploying Application Insights", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", {
            features: {
                resourceGroup: {
                    preventDeletionIfContainsResources: false,
                },
            },
        });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "eastus",
        });
        const keyvault = new key_vault_1.KeyVault(stack, "key_vault", {
            name: `kv-${randomName}`,
            location: resourceGroup.location,
            resourceGroupName: resourceGroup.name,
            skuName: "standard",
            tenantId: util.getAzureTenantId(),
            purgeProtectionEnabled: true,
            softDeleteRetentionDays: 7,
            accessPolicy: [
                {
                    tenantId: util.getAzureTenantId(),
                    objectId: clientConfig.objectId,
                    secretPermissions: [
                        "Get",
                        "List",
                        "Set",
                        "Delete",
                        "Backup",
                        "Restore",
                        "Recover",
                        "Purge",
                    ],
                },
            ],
        });
        const logAnalyticsWorkspace = new log_analytics_workspace_1.LogAnalyticsWorkspace(stack, "log_analytics", {
            location: "eastus",
            name: `la-${randomName}`,
            resourceGroupName: resourceGroup.name,
        });
        const applicationInsights = new appi.AppInsights(stack, "testappi", {
            name: `appi-${randomName}`,
            location: "eastus",
            resourceGroup: resourceGroup,
            applicationType: "web",
            workspaceId: logAnalyticsWorkspace.id,
        });
        // Save Ikey to Key Vault as secret
        applicationInsights.saveIKeyToKeyVault(keyvault.id);
        applicationInsights.saveIKeyToKeyVault(keyvault.id, "customSecretName");
        //Diag Settings
        applicationInsights.addDiagSettings({
            name: "diagsettings",
            logAnalyticsWorkspaceId: logAnalyticsWorkspace.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        //RBAC
        applicationInsights.addAccess(clientConfig.objectId, "Contributor");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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