"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLake = void 0;
const filesystem_1 = require("./filesystem");
const sa = require("../../azure-storageaccount");
const core_azure_1 = require("../../core-azure");
class DataLake extends core_azure_1.AzureResource {
    /**
     * Represents an Azure Data Lake storage account, managing the storage and retrieval of data in a scalable, secure manner.
     *
     * This class encapsulates the properties and functionality of an Azure Data Lake, which is built on top of Azure Blob storage
     * with enhanced capabilities to handle big data analytics efficiently. The Data Lake enables batch and real-time data processing
     * and collaboration for analytics purposes. It provides a centralized place to store structured or unstructured data from multiple
     * sources so that it can be processed and analyzed.
     *
     * Properties:
     * - `accountReplicationType`: Specifies the type of data replication for durability and availability. Options include locally redundant storage (LRS),
     *   geo-redundant storage (GRS), and read-access geo-redundant storage (RAGRS).
     * - `accountTier`: Defines the performance tier (e.g., Standard, Premium) that affects the type of hardware used for the storage account.
     * - `location`: The Azure region where the storage account is located, influencing where data is stored geographically.
     * - `name`: A unique name for the storage account within Azure.
     * - `resourceGroup`: The resource group under which the storage account is categorized and billed.
     * - `tags`: Key-value pairs for resource categorization and operational management.
     * - `enableHttpsTrafficOnly`: Enforces HTTPS for data transfer, enhancing security by encrypting data in transit.
     * - `identity`: Managed Service Identity configuration for Azure Active Directory authentication management.
     * - `accessTier`: Specifies the data access tier (Hot, Cool) affecting cost and retrieval speeds.
     * - `isHnsEnabled`: Indicates if the Hierarchical Namespace is enabled, required for Azure Data Lake Storage Gen2.
     * - `minTlsVersion`: The minimum version of TLS required for securing connections, enhancing data security.
     * - `publicNetworkAccessEnabled`: Controls whether the storage account is accessible from the public internet.
     * - `accountKind`: Specifies the kind of storage account, which can influence supported features and pricing.
     *
     * @param scope - The scope in which this construct is defined, typically representing the cloud development kit (CDK) stack.
     * @param id - The unique identifier for this construct.
     * @param props - Properties for configuring the Data Lake, including location, name, security settings, etc.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.filesystems = new Map();
        // Default values
        this.props = {
            name: "test42348808",
            location: "eastus",
            ...props,
        };
        this.resourceGroup = this.setupResourceGroup(props);
        this.resourceGroupName = this.resourceGroup.name;
        this.storageAccount = new sa.Account(scope, id + "storageAccount", {
            ...this.props,
            accountTier: "Standard",
            accountKind: "StorageV2",
            isHnsEnabled: true,
            resourceGroup: this.resourceGroup,
        });
        this.id = this.storageAccount.id;
    }
    /**
     * Creates a new Data Lake Gen2 filesystem within the specified storage account.
     *
     * This method initializes a new filesystem in Azure Data Lake Storage Gen2, allowing for the storage of hierarchical data structures.
     * Each filesystem can contain multiple directories and files, organized in a hierarchical manner. This method ensures that the filesystem
     * name is unique within the storage account to prevent naming conflicts.
     *
     * @param name - The name of the new filesystem to create. This name must be unique within the storage account.
     * @param props - Configuration properties for the new filesystem, including metadata, encryption settings, and more.
     * @returns The newly created DataLakeFilesystem instance.
     * @throws Error if a filesystem with the same name already exists in the storage account.
     *
     * Example usage:
     * ```typescript
     * const filesystemConfig = {
     *   properties: { property1: 'value1' },  // Example properties
     *   defaultEncryptionScope: 'myEncryptionScope'
     * };
     * const newFilesystem = storageAccount.addDataLakeFilesystem('myNewFilesystem', filesystemConfig);
     * ```
     * This example creates a new filesystem named 'myNewFilesystem' with specified properties and encryption scope.
     */
    addDataLakeFilesystem(name, props) {
        if (this.filesystems.has(name)) {
            throw new Error(`Filesystem '${name}' already exists.`);
        }
        var config = {
            name: name,
            storageAccountId: this.storageAccount.id,
            ...props,
        };
        const newFilesystem = new filesystem_1.DataLakeFilesystem(this, name, config);
        this.filesystems.set(name, newFilesystem);
        return newFilesystem;
    }
}
exports.DataLake = DataLake;
//# sourceMappingURL=data:application/json;base64,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