"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeFilesystem = void 0;
const storage_data_lake_gen2_filesystem_1 = require("@cdktf/provider-azurerm/lib/storage-data-lake-gen2-filesystem");
const constructs_1 = require("constructs");
const path_1 = require("./path");
class DataLakeFilesystem extends constructs_1.Construct {
    /**
     * Represents a filesystem within Azure Data Lake Storage Gen2.
     *
     * This class manages the lifecycle and configuration of a filesystem in Azure Data Lake Gen2, which is used to store directories and files
     * in a hierarchical structure. This class allows for the creation, configuration, and management of access controls and properties associated with a filesystem.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the filesystem.
     * @param props - Configuration properties for the Data Lake Gen2 filesystem. These properties may include:
     *                - `name`: The name of the filesystem.
     *                - `storageAccountId`: The ID of the Azure Storage account that hosts the filesystem.
     *                - `defaultEncryptionScope`: Specifies the default encryption scope for the filesystem.
     *                - `properties`: A dictionary of strings representing properties to associate with the filesystem.
     *                - `ace`: An array of access control expressions to define permissions within the filesystem.
     *
     * Example usage:
     * ```typescript
     * const dataLakeFilesystem = new StorageDataLakeGen2Filesystem(this, 'myFilesystem', {
     *   name: 'mydatafilesystem',
     *   storageAccountId: 'storage_account_id',
     *   defaultEncryptionScope: 'my-encryption-scope',
     *   properties: {
     *     property1: 'value1',
     *     property2: 'value2'
     *   }
     * });
     * ```
     * This class initializes a filesystem with specified configurations such as name, storage account ID, and optional properties like encryption scope and custom properties.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.name = id;
        this.paths = new Map();
        this.storageAccountId = props.storageAccountId;
        this.filesystem = new storage_data_lake_gen2_filesystem_1.StorageDataLakeGen2Filesystem(this, id, props);
    }
    /**
     * Adds a new directory or file path to the existing Data Lake Gen2 filesystem.
     *
     * This method allows for the creation of a new path, which can be either a directory or a file, within the defined filesystem in Azure Data Lake Gen2.
     * Each path is configured with optional properties and access controls, making it possible to finely tune the permissions and settings for data stored at this path.
     *
     * @param name - The name of the new path to create within the filesystem. This name must be unique within the filesystem.
     * @param props - Configuration properties for the new path, including access controls, ownership details, and other relevant settings.
     * @returns The newly created DataLakePath instance representing the path within the filesystem.
     * @throws Error if a path with the same name already exists in the filesystem.
     *
     * Example usage:
     * ```typescript
     * const pathConfig = {
     *   group: 'dataGroup',
     *   owner: 'dataOwner',
     *   ace: [{ id: 'ace1', type: 'user', permissions: 'rwx' }]
     * };
     * const newPath = dataLakeFilesystem.addDataLakePath('newDataPath', pathConfig);
     * ```
     * This example demonstrates adding a new path 'newDataPath' to the filesystem with specified group, owner, and access control entries.
     */
    addDataLakePath(name, props) {
        if (this.paths.has(name)) {
            throw new Error(`Filesystem '${name}' already exists.`);
        }
        var config = {
            filesystemName: this.filesystem.name,
            storageAccountId: this.storageAccountId,
            path: name,
            resource: "directory",
            ...props,
        };
        const newPath = new path_1.DataLakePath(this, name, config);
        this.paths.set(name, newPath);
        return newPath;
    }
}
exports.DataLakeFilesystem = DataLakeFilesystem;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmlsZXN5c3RlbS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9henVyZS1kYXRhbGFrZS9saWIvZmlsZXN5c3RlbS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxSEFJdUU7QUFFdkUsMkNBQXVDO0FBQ3ZDLGlDQUEwRDtBQVMxRCxNQUFhLGtCQUFtQixTQUFRLHNCQUFTO0lBTy9DOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BNEJHO0lBQ0gsWUFDRSxLQUFnQixFQUNoQixFQUFVLEVBQ1YsS0FBMEM7UUFFMUMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQztRQUNmLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxHQUFHLEVBQXdCLENBQUM7UUFDN0MsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUMvQyxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksaUVBQTZCLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXFCRztJQUNJLGVBQWUsQ0FDcEIsSUFBWSxFQUNaLEtBQXlCO1FBRXpCLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxlQUFlLElBQUksbUJBQW1CLENBQUMsQ0FBQztTQUN6RDtRQUVELElBQUksTUFBTSxHQUFrQztZQUMxQyxjQUFjLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJO1lBQ3BDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQkFBZ0I7WUFDdkMsSUFBSSxFQUFFLElBQUk7WUFDVixRQUFRLEVBQUUsV0FBVztZQUNyQixHQUFHLEtBQUs7U0FDVCxDQUFDO1FBRUYsTUFBTSxPQUFPLEdBQUcsSUFBSSxtQkFBWSxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFckQsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzlCLE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7Q0FDRjtBQTNGRCxnREEyRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1xuICBTdG9yYWdlRGF0YUxha2VHZW4yRmlsZXN5c3RlbSxcbiAgU3RvcmFnZURhdGFMYWtlR2VuMkZpbGVzeXN0ZW1Db25maWcsXG4gIFN0b3JhZ2VEYXRhTGFrZUdlbjJGaWxlc3lzdGVtQWNlLFxufSBmcm9tIFwiQGNka3RmL3Byb3ZpZGVyLWF6dXJlcm0vbGliL3N0b3JhZ2UtZGF0YS1sYWtlLWdlbjItZmlsZXN5c3RlbVwiO1xuaW1wb3J0IHsgU3RvcmFnZURhdGFMYWtlR2VuMlBhdGhDb25maWcgfSBmcm9tIFwiQGNka3RmL3Byb3ZpZGVyLWF6dXJlcm0vbGliL3N0b3JhZ2UtZGF0YS1sYWtlLWdlbjItcGF0aFwiO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcbmltcG9ydCB7IERhdGFMYWtlUGF0aCwgRGF0YUxha2VQYXRoQ29uZmlnIH0gZnJvbSBcIi4vcGF0aFwiO1xuXG5leHBvcnQgaW50ZXJmYWNlIERhdGFMYWtlRmlsZXN5c3RlbUNvbmZpZyB7XG4gIHJlYWRvbmx5IGdyb3VwPzogc3RyaW5nO1xuICByZWFkb25seSBvd25lcj86IHN0cmluZztcbiAgcmVhZG9ubHkgcHJvcGVydGllcz86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG4gIHJlYWRvbmx5IGFjZT86IFN0b3JhZ2VEYXRhTGFrZUdlbjJGaWxlc3lzdGVtQWNlW107XG59XG5cbmV4cG9ydCBjbGFzcyBEYXRhTGFrZUZpbGVzeXN0ZW0gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZmlsZXN5c3RlbTogU3RvcmFnZURhdGFMYWtlR2VuMkZpbGVzeXN0ZW07XG4gIHB1YmxpYyByZWFkb25seSBwYXRoczogTWFwPHN0cmluZywgRGF0YUxha2VQYXRoPjtcblxuICBwcml2YXRlIHN0b3JhZ2VBY2NvdW50SWQ6IHN0cmluZztcblxuICAvKipcbiAgICogUmVwcmVzZW50cyBhIGZpbGVzeXN0ZW0gd2l0aGluIEF6dXJlIERhdGEgTGFrZSBTdG9yYWdlIEdlbjIuXG4gICAqXG4gICAqIFRoaXMgY2xhc3MgbWFuYWdlcyB0aGUgbGlmZWN5Y2xlIGFuZCBjb25maWd1cmF0aW9uIG9mIGEgZmlsZXN5c3RlbSBpbiBBenVyZSBEYXRhIExha2UgR2VuMiwgd2hpY2ggaXMgdXNlZCB0byBzdG9yZSBkaXJlY3RvcmllcyBhbmQgZmlsZXNcbiAgICogaW4gYSBoaWVyYXJjaGljYWwgc3RydWN0dXJlLiBUaGlzIGNsYXNzIGFsbG93cyBmb3IgdGhlIGNyZWF0aW9uLCBjb25maWd1cmF0aW9uLCBhbmQgbWFuYWdlbWVudCBvZiBhY2Nlc3MgY29udHJvbHMgYW5kIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGEgZmlsZXN5c3RlbS5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIC0gVGhlIHNjb3BlIGluIHdoaWNoIHRvIGRlZmluZSB0aGlzIGNvbnN0cnVjdCwgdHlwaWNhbGx5IHJlcHJlc2VudGluZyB0aGUgQ2xvdWQgRGV2ZWxvcG1lbnQgS2l0IChDREspIHN0YWNrLlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgdW5pcXVlIGlkZW50aWZpZXIgZm9yIHRoaXMgaW5zdGFuY2Ugb2YgdGhlIGZpbGVzeXN0ZW0uXG4gICAqIEBwYXJhbSBwcm9wcyAtIENvbmZpZ3VyYXRpb24gcHJvcGVydGllcyBmb3IgdGhlIERhdGEgTGFrZSBHZW4yIGZpbGVzeXN0ZW0uIFRoZXNlIHByb3BlcnRpZXMgbWF5IGluY2x1ZGU6XG4gICAqICAgICAgICAgICAgICAgIC0gYG5hbWVgOiBUaGUgbmFtZSBvZiB0aGUgZmlsZXN5c3RlbS5cbiAgICogICAgICAgICAgICAgICAgLSBgc3RvcmFnZUFjY291bnRJZGA6IFRoZSBJRCBvZiB0aGUgQXp1cmUgU3RvcmFnZSBhY2NvdW50IHRoYXQgaG9zdHMgdGhlIGZpbGVzeXN0ZW0uXG4gICAqICAgICAgICAgICAgICAgIC0gYGRlZmF1bHRFbmNyeXB0aW9uU2NvcGVgOiBTcGVjaWZpZXMgdGhlIGRlZmF1bHQgZW5jcnlwdGlvbiBzY29wZSBmb3IgdGhlIGZpbGVzeXN0ZW0uXG4gICAqICAgICAgICAgICAgICAgIC0gYHByb3BlcnRpZXNgOiBBIGRpY3Rpb25hcnkgb2Ygc3RyaW5ncyByZXByZXNlbnRpbmcgcHJvcGVydGllcyB0byBhc3NvY2lhdGUgd2l0aCB0aGUgZmlsZXN5c3RlbS5cbiAgICogICAgICAgICAgICAgICAgLSBgYWNlYDogQW4gYXJyYXkgb2YgYWNjZXNzIGNvbnRyb2wgZXhwcmVzc2lvbnMgdG8gZGVmaW5lIHBlcm1pc3Npb25zIHdpdGhpbiB0aGUgZmlsZXN5c3RlbS5cbiAgICpcbiAgICogRXhhbXBsZSB1c2FnZTpcbiAgICogYGBgdHlwZXNjcmlwdFxuICAgKiBjb25zdCBkYXRhTGFrZUZpbGVzeXN0ZW0gPSBuZXcgU3RvcmFnZURhdGFMYWtlR2VuMkZpbGVzeXN0ZW0odGhpcywgJ215RmlsZXN5c3RlbScsIHtcbiAgICogICBuYW1lOiAnbXlkYXRhZmlsZXN5c3RlbScsXG4gICAqICAgc3RvcmFnZUFjY291bnRJZDogJ3N0b3JhZ2VfYWNjb3VudF9pZCcsXG4gICAqICAgZGVmYXVsdEVuY3J5cHRpb25TY29wZTogJ215LWVuY3J5cHRpb24tc2NvcGUnLFxuICAgKiAgIHByb3BlcnRpZXM6IHtcbiAgICogICAgIHByb3BlcnR5MTogJ3ZhbHVlMScsXG4gICAqICAgICBwcm9wZXJ0eTI6ICd2YWx1ZTInXG4gICAqICAgfVxuICAgKiB9KTtcbiAgICogYGBgXG4gICAqIFRoaXMgY2xhc3MgaW5pdGlhbGl6ZXMgYSBmaWxlc3lzdGVtIHdpdGggc3BlY2lmaWVkIGNvbmZpZ3VyYXRpb25zIHN1Y2ggYXMgbmFtZSwgc3RvcmFnZSBhY2NvdW50IElELCBhbmQgb3B0aW9uYWwgcHJvcGVydGllcyBsaWtlIGVuY3J5cHRpb24gc2NvcGUgYW5kIGN1c3RvbSBwcm9wZXJ0aWVzLlxuICAgKi9cbiAgY29uc3RydWN0b3IoXG4gICAgc2NvcGU6IENvbnN0cnVjdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIHByb3BzOiBTdG9yYWdlRGF0YUxha2VHZW4yRmlsZXN5c3RlbUNvbmZpZyxcbiAgKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLm5hbWUgPSBpZDtcbiAgICB0aGlzLnBhdGhzID0gbmV3IE1hcDxzdHJpbmcsIERhdGFMYWtlUGF0aD4oKTtcbiAgICB0aGlzLnN0b3JhZ2VBY2NvdW50SWQgPSBwcm9wcy5zdG9yYWdlQWNjb3VudElkO1xuICAgIHRoaXMuZmlsZXN5c3RlbSA9IG5ldyBTdG9yYWdlRGF0YUxha2VHZW4yRmlsZXN5c3RlbSh0aGlzLCBpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgZGlyZWN0b3J5IG9yIGZpbGUgcGF0aCB0byB0aGUgZXhpc3RpbmcgRGF0YSBMYWtlIEdlbjIgZmlsZXN5c3RlbS5cbiAgICpcbiAgICogVGhpcyBtZXRob2QgYWxsb3dzIGZvciB0aGUgY3JlYXRpb24gb2YgYSBuZXcgcGF0aCwgd2hpY2ggY2FuIGJlIGVpdGhlciBhIGRpcmVjdG9yeSBvciBhIGZpbGUsIHdpdGhpbiB0aGUgZGVmaW5lZCBmaWxlc3lzdGVtIGluIEF6dXJlIERhdGEgTGFrZSBHZW4yLlxuICAgKiBFYWNoIHBhdGggaXMgY29uZmlndXJlZCB3aXRoIG9wdGlvbmFsIHByb3BlcnRpZXMgYW5kIGFjY2VzcyBjb250cm9scywgbWFraW5nIGl0IHBvc3NpYmxlIHRvIGZpbmVseSB0dW5lIHRoZSBwZXJtaXNzaW9ucyBhbmQgc2V0dGluZ3MgZm9yIGRhdGEgc3RvcmVkIGF0IHRoaXMgcGF0aC5cbiAgICpcbiAgICogQHBhcmFtIG5hbWUgLSBUaGUgbmFtZSBvZiB0aGUgbmV3IHBhdGggdG8gY3JlYXRlIHdpdGhpbiB0aGUgZmlsZXN5c3RlbS4gVGhpcyBuYW1lIG11c3QgYmUgdW5pcXVlIHdpdGhpbiB0aGUgZmlsZXN5c3RlbS5cbiAgICogQHBhcmFtIHByb3BzIC0gQ29uZmlndXJhdGlvbiBwcm9wZXJ0aWVzIGZvciB0aGUgbmV3IHBhdGgsIGluY2x1ZGluZyBhY2Nlc3MgY29udHJvbHMsIG93bmVyc2hpcCBkZXRhaWxzLCBhbmQgb3RoZXIgcmVsZXZhbnQgc2V0dGluZ3MuXG4gICAqIEByZXR1cm5zIFRoZSBuZXdseSBjcmVhdGVkIERhdGFMYWtlUGF0aCBpbnN0YW5jZSByZXByZXNlbnRpbmcgdGhlIHBhdGggd2l0aGluIHRoZSBmaWxlc3lzdGVtLlxuICAgKiBAdGhyb3dzIEVycm9yIGlmIGEgcGF0aCB3aXRoIHRoZSBzYW1lIG5hbWUgYWxyZWFkeSBleGlzdHMgaW4gdGhlIGZpbGVzeXN0ZW0uXG4gICAqXG4gICAqIEV4YW1wbGUgdXNhZ2U6XG4gICAqIGBgYHR5cGVzY3JpcHRcbiAgICogY29uc3QgcGF0aENvbmZpZyA9IHtcbiAgICogICBncm91cDogJ2RhdGFHcm91cCcsXG4gICAqICAgb3duZXI6ICdkYXRhT3duZXInLFxuICAgKiAgIGFjZTogW3sgaWQ6ICdhY2UxJywgdHlwZTogJ3VzZXInLCBwZXJtaXNzaW9uczogJ3J3eCcgfV1cbiAgICogfTtcbiAgICogY29uc3QgbmV3UGF0aCA9IGRhdGFMYWtlRmlsZXN5c3RlbS5hZGREYXRhTGFrZVBhdGgoJ25ld0RhdGFQYXRoJywgcGF0aENvbmZpZyk7XG4gICAqIGBgYFxuICAgKiBUaGlzIGV4YW1wbGUgZGVtb25zdHJhdGVzIGFkZGluZyBhIG5ldyBwYXRoICduZXdEYXRhUGF0aCcgdG8gdGhlIGZpbGVzeXN0ZW0gd2l0aCBzcGVjaWZpZWQgZ3JvdXAsIG93bmVyLCBhbmQgYWNjZXNzIGNvbnRyb2wgZW50cmllcy5cbiAgICovXG4gIHB1YmxpYyBhZGREYXRhTGFrZVBhdGgoXG4gICAgbmFtZTogc3RyaW5nLFxuICAgIHByb3BzOiBEYXRhTGFrZVBhdGhDb25maWcsXG4gICk6IERhdGFMYWtlUGF0aCB7XG4gICAgaWYgKHRoaXMucGF0aHMuaGFzKG5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEZpbGVzeXN0ZW0gJyR7bmFtZX0nIGFscmVhZHkgZXhpc3RzLmApO1xuICAgIH1cblxuICAgIHZhciBjb25maWc6IFN0b3JhZ2VEYXRhTGFrZUdlbjJQYXRoQ29uZmlnID0ge1xuICAgICAgZmlsZXN5c3RlbU5hbWU6IHRoaXMuZmlsZXN5c3RlbS5uYW1lLFxuICAgICAgc3RvcmFnZUFjY291bnRJZDogdGhpcy5zdG9yYWdlQWNjb3VudElkLFxuICAgICAgcGF0aDogbmFtZSxcbiAgICAgIHJlc291cmNlOiBcImRpcmVjdG9yeVwiLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfTtcblxuICAgIGNvbnN0IG5ld1BhdGggPSBuZXcgRGF0YUxha2VQYXRoKHRoaXMsIG5hbWUsIGNvbmZpZyk7XG5cbiAgICB0aGlzLnBhdGhzLnNldChuYW1lLCBuZXdQYXRoKTtcbiAgICByZXR1cm4gbmV3UGF0aDtcbiAgfVxufVxuIl19