"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const eventhub_1 = require("@cdktf/provider-azurerm/lib/eventhub");
const cdktf = require("cdktf");
const constructs_1 = require("constructs");
const authorization_1 = require("./authorization");
const consumer_1 = require("./consumer");
const kusto_connection_1 = require("./kusto-connection");
class Instance extends constructs_1.Construct {
    /**
     * Constructs a new Event Hub instance.
     *
     * This class creates an Azure Event Hub instance within a specified namespace. Event Hubs is a highly scalable
     * data streaming platform and event ingestion service, capable of receiving and processing millions of events per second.
     * Event Hubs can process and store events, data, or telemetry produced by distributed software and devices.
     *
     * @param scope - The scope in which to define this construct, typically used for managing lifecycles and creation order.
     * @param name - The unique name for this instance of the Event Hub.
     * @param ehInstanceProps - The properties for configuring the Event Hub instance. The properties include:
     *                - `name`: Required. The name of the Event Hub instance.
     *                - `partitionCount`: Optional. The number of partitions for the Event Hub. Defaults to 2. Max value is 32 for shared namespaces.
     *                - `messageRetention`: Optional. The number of days to retain the messages. Defaults to 1.
     *                - `status`: Optional. Specifies the operational status of the Event Hub (Active, Disabled, SendDisabled). Defaults to "Active".
     *                - `resourceGroup`: Required. The name of the resource group in which the Event Hub's parent namespace exists.
     *                - `namespaceName`: Required. The name of the Event Hub Namespace where this instance will be created.
     *
     * Example usage:
     * ```typescript
     * const eventHubInstance = new Instance(this, 'myEventHubInstance', {
     *   name: 'exampleEventHub',
     *   namespaceName: 'exampleNamespace',
     *   resourceGroup: resourceGroup,
     *   partitionCount: 4,
     *   messageRetention: 7,
     *   status: 'Active'
     * });
     * ```
     */
    constructor(scope, name, ehInstanceProps) {
        super(scope, name);
        this.ehInstanceProps = ehInstanceProps;
        const defaults = {
            partitionCount: ehInstanceProps.partitionCount || 2,
            messageRetention: ehInstanceProps.messageRetention || 1,
            status: ehInstanceProps.status || "Active",
        };
        const eventhubInstance = new eventhub_1.Eventhub(this, `ehinstance-${ehInstanceProps.name}`, {
            name: ehInstanceProps.name,
            resourceGroupName: ehInstanceProps.resourceGroup.name,
            namespaceName: ehInstanceProps.namespaceName,
            ...defaults,
        });
        // Outputs
        this.id = eventhubInstance.id;
        this.partitionIds = eventhubInstance.partitionIds;
        this.name = eventhubInstance.name;
        const cdktfTerraformOutputEventhubInstanceId = new cdktf.TerraformOutput(this, "id", {
            value: eventhubInstance.id,
        });
        const cdktfTerraformOutputEventhubInstancePartitionIds = new cdktf.TerraformOutput(this, "partition_ids", {
            value: eventhubInstance.partitionIds,
        });
        cdktfTerraformOutputEventhubInstanceId.overrideLogicalId("id");
        cdktfTerraformOutputEventhubInstancePartitionIds.overrideLogicalId("partition_ids");
    }
    /**
     * Adds an Authorization Rule to an Event Hub instance.
     *
     * This method creates a new Authorization Rule associated with the specified Event Hub,
     * granting specified permissions such as 'listen', 'send', and 'manage' based on the properties provided.
     * The rule determines the access level granted to users and applications for the Event Hub.
     *
     * @param props - The properties for the Authorization Rule, which include:
     *                - `name`: Required. A unique identifier for the Authorization Rule within the Event Hub.
     *                - `listen`: Optional. Specifies if the rule allows listening to the Event Hub. Defaults to false.
     *                - `send`: Optional. Specifies if the rule allows sending events to the Event Hub. Defaults to false.
     *                - `manage`: Optional. Specifies if the rule allows managing the Event Hub. When set to true,
     *                            both 'listen' and 'send' are implicitly enabled. Defaults to false.
     *
     * @returns An instance of the AuthorizationRule class, configured with the specified permissions and associated
     *          with the Event Hub specified in the enclosing construct's properties.
     *
     * Example usage:
     * ```typescript
     * const eventHubAuthRule = eventHubInstance.addAuthorizationRule({
     *   name: 'myCustomAuthRule',
     *   listen: true,
     *   send: true,
     *   manage: false // Only listening and sending permissions are granted.
     * });
     * ```
     */
    addAuthorizationRule(props) {
        return new authorization_1.AuthorizationRule(this, `ehauthrule-${props.name}`, {
            resourceGroupName: this.ehInstanceProps.resourceGroup.name,
            namespaceName: this.ehInstanceProps.namespaceName,
            eventhubName: this.name,
            ...props,
        });
    }
    /**
     * Adds a Consumer Group to an existing Event Hub instance.
     *
     * This method creates a new Consumer Group for the specified Event Hub. Consumer groups represent a view of the entire Event Hub,
     * allowing consumer applications to have separate, independent views of the event stream. They read the stream at their own pace
     * and maintain their own sequence point or offset. This enables a single Event Hub to support multiple consumer applications.
     *
     * @param name - The name of the Consumer Group to be added. This name must be unique within the Event Hub namespace.
     * @param userMetadata - Optional. User-defined metadata for the Consumer Group. This metadata is useful for storing additional
     *                       information about the consumer group, such as its purpose or operational details.
     *
     * @returns An instance of the ConsumerGroup class, configured with the specified properties and associated with the Event Hub
     *          specified in the enclosing construct's properties.
     *
     * Example usage:
     * ```typescript
     * const myConsumerGroup = eventHubInstance.addConsumerGroup('myConsumerGroupName', 'Metadata about this consumer group');
     * ```
     *
     * @remarks
     * Each consumer group can have multiple concurrent readers, but each partition in the Event Hub can only have one active consumer
     * from a specific consumer group at a time. Multiple consumer groups enable multiple consuming applications to each have a separate
     * view of the event stream, and to read the stream independently at their own pace and with their own offsets.
     */
    addConsumerGroup(name, userMetadata) {
        return new consumer_1.ConsumerGroup(this, `ehconsumergroup-${name}`, {
            resourceGroup: this.ehInstanceProps.resourceGroup,
            namespaceName: this.ehInstanceProps.namespaceName,
            eventhubName: this.name,
            name: name,
            userMetadata: userMetadata,
        });
    }
    /**
     * Adds a Kusto Data Connection to an existing Kusto Cluster and Database for ingesting data from an EventHub.
     *
     * This method configures a new Kusto Data Connection linked to the specified EventHub. It facilitates the ingestion of streaming data
     * into the Kusto database, allowing for real-time analytics on streamed data. This connection specifies how data from EventHub
     * is to be ingested into tables within the Kusto Database.
     *
     * @param props - The properties for the Kusto Data Connection, derived from BaseKustoDataConnectionProps, which include:
     *                - `name`: Required. The name of the data connection to create.
     *                - `location`: Required. The Azure region where the data connection will be created.
     *                - `kustoResourceGroup`: Required. The Resource Group where the Kusto database exists.
     *                - `kustoClusterName`: Required. The name of the Kusto Cluster to which this data connection will be added.
     *                - `kustoDatabaseName`: Required. The name of the Kusto Database to which this data connection will be added.
     *                - `consumerGroup`: Optional. The EventHub consumer group used for ingestion. Defaults to "$Default".
     *                - `tableName`: Optional. The target table name in the Kusto database used for data ingestion.
     *                - `identityId`: Optional. The resource ID of a managed identity used for authentication with EventHub.
     *                - `mappingRuleName`: Optional. The mapping rule name used for data ingestion.
     *                - `dataFormat`: Optional. Specifies the data format of EventHub messages. Defaults to "JSON".
     *                - `databaseRoutingType`: Optional. Indicates the routing type for the database. Defaults to "Single".
     *                - `compression`: Optional. Specifies the compression type for the data connection. Defaults to "None".
     *
     * @returns An instance of the KustoDataConnection class, configured with the specified properties and linked to the EventHub
     *          specified in the enclosing construct's properties.
     *
     * Example usage:
     * ```typescript
     * const kustoConnection = kustoInstance.addKustoDataConnection({
     *   name: 'myKustoDataConnection',
     *   location: 'West US',
     *   kustoResourceGroup: resourceGroup,
     *   kustoClusterName: 'myCluster',
     *   kustoDatabaseName: 'myDatabase',
     *   tableName: 'IngestionTable',
     *   consumerGroup: '$Default',
     *   dataFormat: 'JSON'
     * });
     * ```
     */
    addKustoDataConnection(props) {
        return new kusto_connection_1.KustoDataConnection(this, `ehkustodataconnection-${this.ehInstanceProps.name}-${props.name}`, {
            eventhubId: this.id,
            ...props,
        });
    }
}
exports.Instance = Instance;
_a = JSII_RTTI_SYMBOL_1;
Instance[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_eventhub.Instance", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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