"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Vault = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const key_vault_1 = require("@cdktf/provider-azurerm/lib/key-vault");
const cdktf = require("cdktf");
const certificate_1 = require("./certificate");
const key_1 = require("./key");
const policy_1 = require("./policy");
const secret_1 = require("./secret");
const lib_1 = require("../../core-azure/lib");
class Vault extends lib_1.AzureResource {
    /**
     * Constructs a new Azure Key Vault resource.
     *
     * This class creates and configures an Azure Key Vault, a secure store for managing secrets, keys, certificates, and other sensitive data.
     * It supports advanced configurations such as access policies, network rules, and data retention policies.
     *
     * @param scope - The scope in which to define this construct, usually representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the Key Vault.
     * @param props - The properties for creating the Key Vault as defined in VaultProps. These include settings for location, SKU, tenant ID, etc.
     *
     * Example usage:
     * ```typescript
     * new Vault(this, 'MyKeyVault', {
     *   name: 'mySecureVault',
     *   location: 'East US',
     *   resourceGroup: myResourceGroup,
     *   sku: 'premium',
     *   tenantId: 'my-tenant-id',
     *   softDeleteRetentionDays: 90,
     *   purgeProtection: true,
     *   tags: {
     *     project: 'My Application'
     *   }
     * });
     * ```
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.accessPolicies = [];
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        // Provide default values
        const purgeProtection = props.purgeProtection ?? true;
        const sku = props.sku ?? "standard";
        const softDeleteRetentionDays = props.softDeleteRetentionDays ?? 90;
        const azurermKeyVault = new key_vault_1.KeyVault(this, "key_vault", {
            name: props.name,
            location: props.location,
            resourceGroupName: this.resourceGroup.name,
            tags: props.tags,
            skuName: sku,
            tenantId: props.tenantId,
            networkAcls: props.networkAcls,
            purgeProtectionEnabled: purgeProtection,
            softDeleteRetentionDays: softDeleteRetentionDays,
        });
        this.id = azurermKeyVault.id;
        this.keyVault = azurermKeyVault;
        // Terraform Outputs
        const cdktfTerraformOutputKeyVaultid = new cdktf.TerraformOutput(this, "id", {
            value: azurermKeyVault.id,
        });
        /*This allows the Terraform resource name to match the original name. You can remove the call if you don't need them to match.*/
        cdktfTerraformOutputKeyVaultid.overrideLogicalId("id");
        const cdktfTerraformOutputKeyVaultname = new cdktf.TerraformOutput(this, "key_vault_name", {
            value: azurermKeyVault.name,
        });
        /*This allows the Terraform resource name to match the original name. You can remove the call if you don't need them to match.*/
        cdktfTerraformOutputKeyVaultname.overrideLogicalId("key_vault_name");
    }
    /**
     * Grants read-only access to secrets stored in the Key Vault to a specified Azure AD group.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive read access to secrets.
     */
    grantSecretReaderAccess(azureAdGroupId) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            secretPermissions: ["Get", "List"],
        };
        const policy = new policy_1.AccessPolicy(this, `kv_secret_reader_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Grants administrative access to secrets stored in the Key Vault to a specified Azure AD group.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive administrative access to secrets.
     */
    grantSecretAdminAccess(azureAdGroupId) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            secretPermissions: [
                "Get",
                "List",
                "Set",
                "Delete",
                "Backup",
                "Restore",
                "Recover",
            ],
        };
        const policy = new policy_1.AccessPolicy(this, `kv_secret_admin_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Grants administrative access to certificates stored in the Key Vault to a specified Azure AD group.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive administrative access to certificates.
     */
    grantCertAdminAccess(azureAdGroupId) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            certificatePermissions: [
                "Get",
                "List",
                "Set",
                "Delete",
                "Backup",
                "Restore",
                "Recover",
            ],
        };
        const policy = new policy_1.AccessPolicy(this, `kv_cert_admin_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Grants read-only access to certificates stored in the Key Vault to a specified Azure AD group.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive read access to certificates.
     */
    grantCertReaderAccess(azureAdGroupId) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            certificatePermissions: ["Get", "List"],
        };
        const policy = new policy_1.AccessPolicy(this, `kv_cert_reader_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Grants administrative access to keys stored in the Key Vault to a specified Azure AD group.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive administrative access to keys.
     */
    grantKeyAdminAccess(azureAdGroupId) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            keyPermissions: [
                "Get",
                "List",
                "Set",
                "Delete",
                "Backup",
                "Restore",
                "Recover",
            ],
        };
        const policy = new policy_1.AccessPolicy(this, `kv_key_admin_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Grants read-only access to keys stored in the Key Vault to a specified Azure AD group.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive read access to keys.
     */
    grantKeyReaderAccess(azureAdGroupId) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            keyPermissions: ["Get", "List"],
        };
        const policy = new policy_1.AccessPolicy(this, `kv_key_reader_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Grants custom access based on specified options to an Azure AD group in the Key Vault.
     *
     * @param azureAdGroupId - The Azure Active Directory group ID that will receive the custom access.
     * @param options - Custom access options specifying various permissions for secrets, keys, certificates, and storage.
     */
    grantCustomAccess(azureAdGroupId, options) {
        const policyProps = {
            keyVaultId: this,
            tenantId: this.props.tenantId,
            objectId: azureAdGroupId,
            ...options,
        };
        const policy = new policy_1.AccessPolicy(this, `kv_custom_policy_access_${azureAdGroupId}`, policyProps);
        this.accessPolicies.push(policy);
    }
    /**
     * Creates a new secret within the Azure Key Vault.
     *
     * This method facilitates the storage of sensitive information in the form of a secret within the Key Vault.
     * Secrets are protected items such as passwords, database connection strings, or any other piece of information
     * that needs to be securely stored and accessed. This method allows setting additional properties such as
     * expiration date and content type for better management and compliance.
     *
     * @param keyVaultSecretName - The unique name for the secret within the Key Vault.
     * @param secretValue - The sensitive information or data that needs to be securely stored as a secret.
     * @param expirationDate - Optional. The expiration date of the secret in ISO 8601 format (YYYY-MM-DD or YYYY-MM-DDTHH:MM:SSZ).
     *                         If provided, the secret will no longer be valid after this date.
     * @param contentType - Optional. A description of the type of information the secret contains (e.g., 'password', 'connectionString').
     *                      This can be used by applications to handle the secret appropriately.
     *
     * Example usage:
     * ```typescript
     * vault.addSecret(
     *   'myDatabasePassword',
     *   'p@ssw0rd123!',
     *   '2030-01-01',
     *   'databasePassword'
     * );
     * ```
     * This method does not return a value. It creates a secret within the Key Vault with the specified properties.
     */
    addSecret(keyVaultSecretName, secretValue, expirationDate, contentType) {
        const secretProps = {
            keyVaultId: this,
            name: keyVaultSecretName,
            value: secretValue,
            expirationDate: expirationDate,
            contentType: contentType,
            accessPolicies: this.accessPolicies,
        };
        new secret_1.Secret(this, keyVaultSecretName, secretProps);
    }
    /**
     * Creates an RSA cryptographic key within the Azure Key Vault.
     *
     * This method facilitates the creation of an RSA key, which is useful for a variety of cryptographic operations such as
     * encryption, decryption, digital signature verification, and more. The RSA key created by this method is configurable
     * with an optional expiration date and a default key size of 2048 bits. The key operations allowed include decryption,
     * encryption, signing, verifying signatures, and key wrapping/unwrapping.
     *
     * @param keyVaultKeyName - The unique name for the RSA key within the Key Vault.
     * @param expirationDate - Optional. The expiration date of the key in ISO 8601 format (YYYY-MM-DD or YYYY-MM-DDTHH:MM:SSZ).
     *                         If provided, the key will no longer be valid after this date.
     * @returns A KeyVaultKey object representing the newly created RSA key within the vault.
     *
     * Example usage:
     * ```typescript
     * const rsaKey = vault.addRSAKey(
     *   'myRSAKey',
     *   '2030-01-01'
     * );
     * ```
     * This method returns the created KeyVaultKey object, allowing further operations or references to the key.
     */
    addRSAKey(keyVaultKeyName, expirationDate) {
        const keyProps = {
            keyVaultId: this,
            name: keyVaultKeyName,
            keyType: "RSA",
            keySize: 2048,
            keyOpts: ["decrypt", "encrypt", "sign", "unwrapKey", "verify", "wrapKey"],
            expires: expirationDate,
            accessPolicies: this.accessPolicies,
        };
        const key = new key_1.Key(this, keyVaultKeyName, keyProps);
        return key.vaultKey;
    }
    /**
     * Creates a cryptographic key within the Azure Key Vault.
     *
     * This method allows the creation of a cryptographic key of specified type and size within the Key Vault. The key can be
     * configured with specific operations it can perform, such as encryption, decryption, signing, etc. An optional expiration
     * date can also be set to control the key's lifecycle. This method is flexible, supporting various key types and sizes,
     * making it suitable for a wide range of cryptographic needs.
     *
     * @param keyVaultKeyName - The unique name for the cryptographic key within the Key Vault.
     * @param keyType - The type of cryptographic key to create (e.g., 'RSA', 'EC', 'oct-HSM').
     * @param keySize - The size of the cryptographic key in bits (e.g., 2048, 3072, 4096 for RSA).
     * @param keyOpts - A list of cryptographic operations that the key is allowed to perform. Possible values might include
     *                  'encrypt', 'decrypt', 'sign', 'verify', 'wrapKey', 'unwrapKey'.
     * @param expirationDate - Optional. The expiration date of the key in ISO 8601 format (YYYY-MM-DD or YYYY-MM-DDTHH:MM:SSZ).
     *                         If provided, the key will no longer be valid after this date, aligning with best practices for key management.
     * @returns A KeyVaultKey object representing the newly created cryptographic key within the vault.
     *
     * Example usage:
     * ```typescript
     * const myKey = vault.addKey(
     *   'myKey',
     *   'RSA',
     *   2048,
     *   ['encrypt', 'decrypt', 'sign', 'verify'],
     *   '2030-12-31'
     * );
     * ```
     * This method returns the created KeyVaultKey object, enabling immediate use within the application for cryptographic operations.
     */
    addKey(keyVaultKeyName, keyType, keySize, keyOpts, expirationDate) {
        const keyProps = {
            keyVaultId: this,
            name: keyVaultKeyName,
            keyType: keyType,
            keySize: keySize,
            keyOpts: keyOpts,
            expires: expirationDate,
            accessPolicies: this.accessPolicies,
        };
        const key = new key_1.Key(this, keyVaultKeyName, keyProps);
        return key.vaultKey;
    }
    /**
     * Creates a self-signed certificate within the Azure Key Vault.
     *
     * This method facilitates the creation of a self-signed certificate, which is a digital certificate that is signed by
     * its own creator rather than a trusted authority. Self-signed certificates can be useful for testing, internal
     * communications, or any scenario where public trust is not required. The method allows specifying subject details,
     * DNS names for the certificate, and managing its lifecycle with action types and expiry.
     *
     * @param certName - The unique name for the certificate within the Key Vault.
     * @param subject - The subject name of the certificate, typically formatted as an X.500 Distinguished Name (e.g., "CN=example.com").
     * @param dnsNames - An array of DNS names that should be associated with this certificate. This is useful for certificates
     *                   that need to be valid for multiple hostnames.
     * @param actionType - Optional. Specifies the action to be performed with the certificate, such as 'create' or 'renew'.
     * @param daysBeforeExpiry - Optional. Number of days before expiry when an action should be taken, useful for auto-renewal scenarios.
     * @returns A KeyVaultCertificate object representing the newly created self-signed certificate.
     *
     * Example usage:
     * ```typescript
     * const myCertificate = vault.addSelfSignedCert(
     *   'myCert',
     *   'CN=mydomain.com',
     *   ['mydomain.com', 'www.mydomain.com'],
     *   'create',
     *   30
     * );
     * ```
     * This method returns the KeyVaultCertificate object, enabling it to be used immediately within the application or stored for future use.
     */
    addSelfSignedCert(certName, subject, dnsNames, actionType, daysBeforeExpiry) {
        const keyProps = {
            keyVaultId: this,
            name: certName,
            subject: subject,
            dnsNames: dnsNames,
            actionType: actionType,
            daysBeforeExpiry: daysBeforeExpiry,
            accessPolicies: this.accessPolicies,
        };
        const cert = new certificate_1.SelfSignedCertificate(this, certName, keyProps);
        return cert.certificate;
    }
    /**
     * Adds a certificate issuer to the Azure Key Vault.
     *
     * This method configures a certificate issuer within the Key Vault, allowing the Key Vault to issue certificates
     * through external providers. Configuring an issuer is essential for enabling automated certificate management
     * processes, such as issuance and renewal, directly through the Key Vault with a specified Certificate Authority (CA).
     *
     * @param name - The unique name for the certificate issuer within the Key Vault.
     * @param provider - The name of the external provider that will issue the certificates, such as 'DigiCert' or 'GlobalSign'.
     *
     * Example usage:
     * ```typescript
     * vault.addCertIssuer(
     *   'myCertIssuer',
     *   'DigiCert'
     * );
     * ```
     * This method configures a certificate issuer but does not return any value. The issuer details, including provider name
     * and any necessary credentials (managed externally or through additional method parameters), are set up in the Key Vault
     * for future certificate operations.
     */
    addCertIssuer(name, provider) {
        new certificate_1.CertificateIssuer(this, name, {
            name: name,
            providerName: provider,
            keyVaultId: this,
            accessPolicies: this.accessPolicies,
        });
    }
}
exports.Vault = Vault;
_a = JSII_RTTI_SYMBOL_1;
Vault[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_keyvault.Vault", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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