"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const log_analytics_workspace_1 = require("@cdktf/provider-azurerm/lib/log-analytics-workspace");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const cdktf_1 = require("cdktf");
const kv = require("..");
const testing_1 = require("../../testing");
const util = require("../../util/azureTenantIdHelpers");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
describe("Example of deploying a Key Vault", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "eastus",
        });
        const azureKeyVault = new kv.Vault(stack, "kv", {
            name: `kv-${randomName}`,
            location: "eastus",
            sku: "standard",
            resourceGroup: resourceGroup,
            tenantId: util.getAzureTenantId(),
            networkAcls: {
                bypass: "AzureServices",
                defaultAction: "Allow",
                ipRules: ["0.0.0.0/0"],
            },
            softDeleteRetentionDays: 7,
        });
        const logAnalyticsWorkspace = new log_analytics_workspace_1.LogAnalyticsWorkspace(stack, "log_analytics", {
            location: "eastus",
            name: `la-${randomName}`,
            resourceGroupName: resourceGroup.name,
        });
        //Diag Settings
        azureKeyVault.addDiagSettings({
            name: "diagsettings",
            logAnalyticsWorkspaceId: logAnalyticsWorkspace.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        //RBAC
        azureKeyVault.addAccess(clientConfig.objectId, "Contributor");
        // Access Policy
        azureKeyVault.grantSecretAdminAccess("bc26a701-6acb-4117-93e0-e44054e22d60");
        azureKeyVault.grantCustomAccess(clientConfig.objectId, {
            storagePermissions: ["Get", "List", "Set", "Delete"],
            secretPermissions: ["Get", "List", "Set", "Delete"],
            keyPermissions: [
                "Backup",
                "Create",
                "Decrypt",
                "Delete",
                "Encrypt",
                "Get",
                "Import",
                "List",
                "Purge",
                "Recover",
                "Restore",
                "Sign",
                "UnwrapKey",
                "Update",
                "Verify",
                "WrapKey",
                "Release",
                "Rotate",
                "GetRotationPolicy",
                "SetRotationPolicy",
            ],
            certificatePermissions: [
                "Get",
                "List",
                "Create",
                "Delete",
                "GetIssuers",
                "ManageIssuers",
            ],
        });
        // Create Secret
        azureKeyVault.addSecret("secret1", "password", "2033-08-23T15:23:17Z", "application/x-pkcs12");
        azureKeyVault.addSecret("customSecretName", "password", "2033-08-23T15:23:17Z", "application/x-pkcs12");
        // Create Key
        azureKeyVault.addRSAKey("key1", "2033-08-23T15:23:17Z");
        azureKeyVault.addKey("key2", "RSA", 2048, ["encrypt", "decrypt", "sign", "verify", "wrapKey", "unwrapKey"], "2033-08-23T15:23:17Z");
        // Create Certificate
        azureKeyVault.addSelfSignedCert("cert1", "CN=contoso.com", [
            "internal.contoso.com",
            "domain.hello.world",
        ]);
        azureKeyVault.addCertIssuer("issuer1", "SslAdminV2");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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