"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Database = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kusto_database_1 = require("@cdktf/provider-azurerm/lib/kusto-database");
const kusto_database_principal_assignment_1 = require("@cdktf/provider-azurerm/lib/kusto-database-principal-assignment");
const kusto_script_1 = require("@cdktf/provider-azurerm/lib/kusto-script");
const cdktf = require("cdktf");
const constructs_1 = require("constructs");
const ts_md5_1 = require("ts-md5");
class Database extends constructs_1.Construct {
    /**
     * Represents a Kusto Database within an Azure Kusto Cluster.
     *
     * This class is responsible for the creation and management of a database in Azure Data Explorer (Kusto),
     * which stores data tables and provides a query engine. A Kusto database is a logical group of tables
     * and is associated with a specific Kusto cluster. The database supports configurations such as
     * hot cache period and soft delete period to optimize performance and data retention according to
     * specific workload requirements.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the Kusto database.
     * @param props - The properties required to configure the Kusto database. These include:
     *                       - `kusto`: Reference to the Kusto cluster to which the database will belong.
     *                       - `name`: The name of the database to be created within the Kusto cluster.
     *                       - `hotCachePeriod`: Optional. Specifies the duration that data should be kept in cache for faster query performance.
     *                                           Expressed in ISO 8601 duration format.
     *                       - `softDeletePeriod`: Optional. Specifies the duration that data should be retained before it stops being accessible.
     *                                             Expressed in ISO 8601 duration format.
     *
     * Example usage:
     * ```typescript
     * const myDatabase = new Database(this, 'MyKustoDatabase', {
     *   kusto: myKustoCluster,
     *   name: 'AnalyticsDB',
     *   hotCachePeriod: 'P30D',  // 30 days
     *   softDeletePeriod: 'P365D' // 365 days
     * });
     * ```
     *
     * This class sets up the database configurations and integrates it within the specified Kusto cluster,
     * providing capabilities to manage and query large datasets effectively.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const kustoDatabase = new kusto_database_1.KustoDatabase(this, `kusto-db-${props.name}`, {
            name: props.name,
            location: props.kustoCluster.location,
            resourceGroupName: props.kustoCluster.resourceGroupName,
            clusterName: props.kustoCluster.name,
        });
        if (props.hotCachePeriod) {
            kustoDatabase.addOverride("hot_cache_period", props.hotCachePeriod);
        }
        if (props.softDeletePeriod) {
            kustoDatabase.addOverride("soft_delete_period", props.softDeletePeriod);
        }
        // Outputs
        this.kustoDatabase = kustoDatabase;
        this.databaseName = props.name;
        const cdktfTerraformOutputKustoDbId = new cdktf.TerraformOutput(this, "id", {
            value: this.kustoDatabase.id,
        });
        cdktfTerraformOutputKustoDbId.overrideLogicalId("id");
    }
    /**
     * Adds a database principal assignment in the Kusto cluster, assigning specified access rights to a principal.
     *
     * This method is used to grant access permissions to a specific user, group, or service principal within an Azure Active Directory.
     * These permissions determine the level of access that the principal has over the Kusto database, such as viewing, ingesting, or managing data.
     * The assignment is made by creating a KustoDatabasePrincipalAssignment resource, specifying the principal details and the type of role
     * they should assume.
     *
     * @param kustoDatabaseAccessProps - The properties defining the principal assignment. This includes:
     *   - `name`: A unique name identifying this principal assignment.
     *   - `tenantId`: The Azure Active Directory tenant ID where the principal resides.
     *   - `principalId`: The object ID of the principal (user, group, or service principal) in Azure AD.
     *   - `principalType`: The type of principal (e.g., User, Group, App).
     *   - `role`: The database role assigned to the principal. Roles can include Admin, User, Viewer, among others.
     *
     * Example usage:
     * ```typescript
     * myDatabase.addPermission({
     *   name: 'DataScienceTeamAccess',
     *   tenantId: 'tenant-id',
     *   principalId: 'principal-id',
     *   principalType: 'Group',
     *   role: 'Viewer'
     * });
     * ```
     * This method creates a new principal assignment, enabling specified access controls for the principal
     * on the Kusto database based on the role assigned. It is crucial for managing security and access
     * governance within the Kusto environment.
     */
    addPermission(kustoDatabaseAccessProps) {
        new kusto_database_principal_assignment_1.KustoDatabasePrincipalAssignment(this, `kusto-db-${kustoDatabaseAccessProps.name}-access`, {
            name: kustoDatabaseAccessProps.name,
            resourceGroupName: this.kustoDatabase.resourceGroupName,
            clusterName: this.kustoDatabase.clusterName,
            databaseName: this.kustoDatabase.name,
            tenantId: kustoDatabaseAccessProps.tenantId,
            principalId: kustoDatabaseAccessProps.principalId,
            principalType: kustoDatabaseAccessProps.principalType,
            role: kustoDatabaseAccessProps.role,
        });
    }
    /**
     * Adds a new table to an existing Azure Kusto database.
     *
     * This method creates a table within the specified Kusto database using a given schema. Tables in Kusto store structured data with
     * defined columns and types, which are crucial for storing and querying large datasets efficiently. The method constructs a Kusto
     * Data Explorer control command to create the table and then executes this command within the context of the database.
     *
     * @param tableName - The name of the table to create, which must be unique within the database.
     * @param tableSchema - An array of schema properties defining the columns of the table, including column names and their data types.
     *
     * Example usage:
     * ```typescript
     * myDatabase.addTable('SalesData', [
     *   { columnName: 'TransactionId', columnType: 'int' },
     *   { columnName: 'TransactionDate', columnType: 'datetime' },
     *   { columnName: 'Amount', columnType: 'real' }
     * ]);
     * ```
     * This method constructs the command to create the table and applies it directly within the Kusto database,
     * ensuring the table is ready for data ingestion and querying.
     */
    addTable(tableName, tableSchema) {
        const schemaContent = tableSchema
            .map((column) => {
            return `${column.columnName}:${column.columnType}`;
        })
            .join(", ");
        const scriptContent = `.create table ${tableName} ( ${schemaContent} )`;
        new kusto_script_1.KustoScript(this, `kusto-db-${this.databaseName}-table-${tableName}`, {
            name: tableName,
            databaseId: this.kustoDatabase.id,
            scriptContent: scriptContent,
            continueOnErrorsEnabled: false,
            forceAnUpdateWhenValueChanged: ts_md5_1.Md5.hashStr(scriptContent),
        });
    }
    /**
     * Adds and executes a control command or script within the Kusto database.
     *
     * This method facilitates the execution of Kusto Query Language (KQL) scripts or control commands within the specified
     * Kusto database. Scripts can perform a variety of functions, from schema modifications, like adding new tables or altering
     * existing ones, to data management operations, such as data ingestion or cleanup tasks. Each script is executed as a
     * KustoScript resource, which ensures that the script is applied correctly and atomically to the database.
     *
     * @param scriptName - A unique name for the script, which helps in identifying the script resource within the deployment.
     * @param scriptContent - The KQL script or control command to be executed. This should be a valid KQL command string.
     *
     * Example usage:
     * ```typescript
     * myDatabase.addScript('InitializeSalesTable', `
     *   .create table SalesData (TransactionId: int, TransactionDate: datetime, Amount: real)
     *   .alter-merge table SalesData policy retentionsoftdelete = 365d
     * `);
     * ```
     * This method will create a `KustoScript` resource that encapsulates the command, ensuring it is executed against the
     * database, and is tracked as part of the resource management within Azure.
     */
    addScript(scriptName, scriptContent) {
        new kusto_script_1.KustoScript(this, `kusto-db-${this.databaseName}-script-${scriptName}`, {
            name: `script-${scriptName}`,
            databaseId: this.kustoDatabase.id,
            scriptContent: scriptContent,
            continueOnErrorsEnabled: false,
            forceAnUpdateWhenValueChanged: ts_md5_1.Md5.hashStr(scriptContent),
        });
    }
}
exports.Database = Database;
_a = JSII_RTTI_SYMBOL_1;
Database[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_kusto.Database", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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