"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const eventhub_namespace_1 = require("@cdktf/provider-azurerm/lib/eventhub-namespace");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const storage_account_1 = require("@cdktf/provider-azurerm/lib/storage-account");
const cdktf_1 = require("cdktf");
const la = require("..");
const testing_1 = require("../../testing");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
describe("Example of deploying Log Analytics", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "eastus",
        });
        const namespace = new eventhub_namespace_1.EventhubNamespace(stack, "ehns", {
            name: `ehns-${randomName}`,
            resourceGroupName: resourceGroup.name,
            location: resourceGroup.location,
            sku: "Standard",
        });
        const storage = new storage_account_1.StorageAccount(stack, "storage", {
            name: `sta${randomName}88t97`,
            resourceGroupName: resourceGroup.name,
            location: resourceGroup.location,
            accountReplicationType: "LRS",
            accountTier: "Standard",
            minTlsVersion: "TLS1_2",
            publicNetworkAccessEnabled: false,
            networkRules: {
                bypass: ["AzureServices"],
                defaultAction: "Deny",
            },
        });
        const logAnalyticsWorkspace = new la.Workspace(stack, "la", {
            name: `la-${randomName}`,
            location: "eastus",
            retention: 90,
            sku: "PerGB2018",
            resourceGroup: resourceGroup,
            functions: [
                {
                    name: "function_name_1",
                    displayName: "Example function 1",
                    query: "Event | where EventLevelName != 'Informational' | where TimeGenerated > ago(24h)",
                    functionAlias: "function_name_1",
                    functionParameters: [],
                },
                {
                    name: "function_name_2",
                    displayName: "Example function 2",
                    query: "Event | where EventLevelName != 'Informational' | where TimeGenerated > ago(24h)",
                    functionAlias: "function_name_2",
                    functionParameters: ["typeArg:string=mail", "tagsArg:string=dc"],
                },
            ],
            dataExport: [
                {
                    name: "export-test",
                    exportDestinationId: namespace.id,
                    tableNames: ["Heartbeat"],
                    enabled: true,
                },
            ],
        });
        // Test RBAC
        logAnalyticsWorkspace.addAccess(clientConfig.objectId, "Contributor");
        logAnalyticsWorkspace.addAccess(clientConfig.objectId, "Monitoring Reader");
        // Test Diag Settings
        logAnalyticsWorkspace.addDiagSettings({
            storageAccountId: storage.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        // Test Metric Alert
        logAnalyticsWorkspace.addMetricAlert({
            name: "metricAlert-test",
            criteria: [
                {
                    metricName: "Heartbeat",
                    metricNamespace: "Microsoft.operationalinsights/workspaces",
                    aggregation: "Average",
                    operator: "LessThan",
                    threshold: 0,
                },
            ],
        });
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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