"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroupAssociations = exports.SecurityGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const network_interface_security_group_association_1 = require("@cdktf/provider-azurerm/lib/network-interface-security-group-association");
const network_security_group_1 = require("@cdktf/provider-azurerm/lib/network-security-group");
const network_security_rule_1 = require("@cdktf/provider-azurerm/lib/network-security-rule");
const subnet_network_security_group_association_1 = require("@cdktf/provider-azurerm/lib/subnet-network-security-group-association");
const constructs_1 = require("constructs");
const lib_1 = require("../../core-azure/lib");
class SecurityGroup extends lib_1.AzureResource {
    /**
     * Represents an Azure Network Security Group (NSG).
     *
     * This class is responsible for the creation and management of an Azure Network Security Group, which acts as a virtual firewall
     * for virtual network resources. A Network Security Group contains a list of security rules that allow or deny network traffic
     * to resources connected to Azure Virtual Networks (VNet). Each rule specifies a combination of source and destination, port,
     * and protocol, and an action (allow or deny) based on those combinations. This class allows for detailed configuration of these
     * rules to enforce security policies for inbound and outbound network traffic.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the security group.
     * @param props - The properties required to configure the Network Security Group, as defined in the SecurityGroupProps interface. These include:
     *                - `resourceGroup`: Optional. Reference to the resource group for deployment.
     *                - `location`: The Azure region where the NSG will be created.
     *                - `name`: The name of the NSG, which must be unique within the resource group.
     *                - `rules`: A list of rules that define the security policies for traffic control.
     *
     * Example usage:
     * ```typescript
     * new SecurityGroup(this, 'MySecurityGroup', {
     *   resourceGroup: myResourceGroup,
     *   location: 'East US',
     *   name: 'myNsg',
     *   rules: [{
     *     name: 'AllowSSH',
     *     priority: 100,
     *     direction: 'Inbound',
     *     access: 'Allow',
     *     protocol: 'Tcp',
     *     sourcePortRange: '*',
     *     destinationPortRange: '22',
     *     sourceAddressPrefix: '*',
     *     destinationAddressPrefix: '*'
     *   }]
     * });
     * ```
     * This class initializes a Network Security Group with specified rules, handling network security management tasks efficiently.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        // Create a network security group
        const nsg = new network_security_group_1.NetworkSecurityGroup(this, "nsg", {
            name: props.name,
            resourceGroupName: this.resourceGroup.name,
            location: props.location,
        });
        // Create security rules within the network security group
        for (const ruleConfig of props.rules) {
            new network_security_rule_1.NetworkSecurityRule(this, ruleConfig.name, {
                name: ruleConfig.name,
                resourceGroupName: this.resourceGroup.name,
                networkSecurityGroupName: nsg.name,
                priority: ruleConfig.priority,
                direction: ruleConfig.direction,
                access: ruleConfig.access,
                protocol: ruleConfig.protocol,
                sourcePortRange: ruleConfig.sourcePortRange,
                destinationPortRange: ruleConfig.destinationPortRange,
                sourceAddressPrefix: ruleConfig.sourceAddressPrefix,
                destinationAddressPrefix: ruleConfig.destinationAddressPrefix,
            });
        }
        this.id = nsg.id;
        this.name = nsg.name;
    }
    /**
     * Associates this Network Security Group with a specified subnet.
     *
     * This method facilitates the attachment of the security group to a subnet, applying the security group's rules to all
     * resources within the subnet. This is crucial for managing network access and security policies at the subnet level.
     *
     * @param subnet - The subnet object to which this network security group will be associated.
     *
     * Example usage:
     * ```typescript
     * const mySubnet = { id: 'subnet-123', name: 'SubnetA' };
     * mySecurityGroup.associateToSubnet(mySubnet);
     * ```
     * This operation ensures that the security rules defined in the network security group are enforced on all network interfaces
     * attached to the specified subnet.
     */
    associateToSubnet(subnet) {
        new SecurityGroupAssociations(this, subnet.name, {
            subnetId: subnet.id,
            networkSecurityGroupId: this.id,
        });
    }
    /**
     * Associates this Network Security Group with a specified network interface.
     *
     * This method attaches the security group to a network interface, applying the security group's rules to the network interface.
     * This allows for fine-grained control of network traffic to and from the specific network interface.
     *
     * @param networkInterface - The network interface object to which this network security group will be associated.
     *
     * Example usage:
     * ```typescript
     * const myNetworkInterface = { id: 'nic-456', name: 'NetworkInterfaceA' };
     * mySecurityGroup.associateToNetworkInterface(myNetworkInterface);
     * ```
     * This operation ensures that the security rules defined in the network security group are applied directly to the specified
     * network interface, controlling access in a more targeted manner.
     */
    associateToNetworkInterface(networkInterface) {
        new SecurityGroupAssociations(this, networkInterface.name, {
            networkInterfaceId: networkInterface.id,
            networkSecurityGroupId: this.id,
        });
    }
}
exports.SecurityGroup = SecurityGroup;
_a = JSII_RTTI_SYMBOL_1;
SecurityGroup[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_networksecuritygroup.SecurityGroup", version: "0.0.3-pre.12" };
class SecurityGroupAssociations extends constructs_1.Construct {
    /**
     * Manages the associations of Azure Network Security Groups with subnets and network interfaces.
     *
     * This class provides the functionality to associate a network security group with either subnets or network interfaces
     * within the Azure environment. By managing these associations, it helps enforce security rules at both the subnet level
     * and the network interface level, enhancing security configurations and compliance.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for the association instance.
     * @param props - The properties for the association. Includes the network security group ID and optionally a subnet ID or network interface ID.
     *
     * Example usage:
     * ```typescript
     * new SecurityGroupAssociations(this, 'MyAssociations', {
     *   networkSecurityGroupId: 'nsg-123',
     *   subnetId: 'subnet-123',
     *   networkInterfaceId: 'nic-456',
     * });
     * ```
     * Depending on the properties provided, this class will create the appropriate associations to apply the network security group
     * to the specified subnet or network interface.
     */
    constructor(scope, id, props) {
        super(scope, id);
        // If subnetId is provided, create a SubnetNetworkSecurityGroupAssociation
        if (props.subnetId) {
            new subnet_network_security_group_association_1.SubnetNetworkSecurityGroupAssociation(this, "subassociation", {
                subnetId: props.subnetId,
                networkSecurityGroupId: props.networkSecurityGroupId,
            });
        }
        // If networkInterfaceId is provided, create a NetworkInterfaceSecurityGroupAssociation
        if (props.networkInterfaceId) {
            new network_interface_security_group_association_1.NetworkInterfaceSecurityGroupAssociation(this, "nicassociation", {
                networkInterfaceId: props.networkInterfaceId,
                networkSecurityGroupId: props.networkSecurityGroupId,
            });
        }
    }
}
exports.SecurityGroupAssociations = SecurityGroupAssociations;
_b = JSII_RTTI_SYMBOL_1;
SecurityGroupAssociations[_b] = { fqn: "@microsoft/terraform-cdk-constructs.azure_networksecuritygroup.SecurityGroupAssociations", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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