"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueryRuleAlert = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const monitor_scheduled_query_rules_alert_v2_1 = require("@cdktf/provider-azurerm/lib/monitor-scheduled-query-rules-alert-v2");
const cdktf = require("cdktf");
const constructs_1 = require("constructs");
const moment = require("moment");
class QueryRuleAlert extends constructs_1.Construct {
    /**
     * Represents an Azure Monitor Scheduled Query Rule Alert.
     *
     * This class is responsible for the creation and management of a Scheduled Query Rule Alert in Azure Monitor.
     * Scheduled Query Rule Alerts execute specified queries at regular intervals over the data collected in Log Analytics
     * Workspaces or Application Insights, and alert based on the results of these queries. These alerts can be used to monitor
     * application health, infrastructure changes, or compliance with certain conditions.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the Scheduled Query Rule Alert.
     * @param props - The properties required to configure the Scheduled Query Rule Alert, as defined in the AzureQueryRuleAlertProps interface.
     *                These include:
     *                - `name`: The name of the Scheduled Query Rule.
     *                - `resourceGroup`: The Azure Resource Group under which the alert will be created.
     *                - `location`: The Azure region where the alert will be deployed.
     *                - `criteriaQuery`: The query to execute. The results of this query determine whether an alert should be triggered.
     *                - `evaluationFrequency`: How often the query should be run.
     *                - `windowDuration`: The time period over which data is collected for each execution of the query.
     *                - `severity`: The severity of the alert.
     *                - `actionActionGroupId`: The action group to invoke when the alert criteria are met.
     *                - `enabled`: Indicates whether the alert rule is enabled.
     *
     * Example usage:
     * ```typescript
     * new QueryRuleAlert(this, 'MyAlertRule', {
     *   name: 'HighErrorRateAlert',
     *   resourceGroup: myResourceGroup,
     *   location: 'West US 2',
     *   criteriaQuery: 'Heartbeat | summarize AggregatedValue = count() by bin(TimeGenerated, 5m)',
     *   evaluationFrequency: 'PT5M',
     *   windowDuration: 'PT1H',
     *   severity: 3,
     *   actionActionGroupId: ['/subscriptions/sub-id/resourceGroups/rg/providers/microsoft.insights/actionGroups/myActionGroup'],
     *   enabled: true
     * });
     * ```
     * This class sets up the alert rule and ensures it is ready to trigger actions based on the specified criteria and schedule.
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Properties validation
        const evaluationFrequencyOption = [
            "PT1M",
            "PT5M",
            "PT10M",
            "PT15M",
            "PT30M",
            "PT45M",
            "PT1H",
            "PT2H",
            "PT3H",
            "PT4H",
            "PT5H",
            "PT6H",
            "P1D",
        ];
        const windowDurationOption = [
            "PT1M",
            "PT5M",
            "PT10M",
            "PT15M",
            "PT30M",
            "PT45M",
            "PT1H",
            "PT2H",
            "PT3H",
            "PT4H",
            "PT5H",
            "PT6H",
            "P1D",
            "P2D",
        ];
        const muteActionsAfterAlertDurationOption = [
            "PT5M",
            "PT10M",
            "PT15M",
            "PT30M",
            "PT45M",
            "PT1H",
            "PT2H",
            "PT3H",
            "PT4H",
            "PT5H",
            "PT6H",
            "P1D",
            "P2D",
        ];
        const queryTimeRangeOverrideOption = [
            "PT5M",
            "PT10M",
            "PT15M",
            "PT20M",
            "PT30M",
            "PT45M",
            "PT1H",
            "PT2H",
            "PT3H",
            "PT4H",
            "PT5H",
            "PT6H",
            "P1D",
            "P2D",
        ];
        // Validate evaluationFrequency
        if (!evaluationFrequencyOption.includes(props.evaluationFrequency)) {
            throw new Error("invalid evaluationFrequency");
        }
        // Validate windowDuration
        if (!windowDurationOption.includes(props.windowDuration)) {
            throw new Error("invalid windowDuration");
        }
        // Validate muteActionsAfterAlertDuration
        if (props.muteActionsAfterAlertDuration &&
            !muteActionsAfterAlertDurationOption.includes(props.muteActionsAfterAlertDuration)) {
            throw new Error("invalid muteActionsAfterAlertDuration");
        }
        // Validate queryTimeRangeOverride
        if (props.queryTimeRangeOverride &&
            !queryTimeRangeOverrideOption.includes(props.queryTimeRangeOverride)) {
            throw new Error("invalid queryTimeRangeOverride");
        }
        // The query look back which is windowDuration * numberOfEvaluationPeriods cannot exceed 48 hours.
        if (props.criteriaFailNumberOfEvaluationPeriods) {
            const windowDurationHours = moment
                .duration(props.windowDuration)
                .asHours();
            const numberOfEvaluationPeriods = props.criteriaFailNumberOfEvaluationPeriods;
            if (windowDurationHours * numberOfEvaluationPeriods > 48) {
                throw new Error("queryTimeRangeOverride cannot exceed 48 hours");
            }
        }
        this.queryRuleAlertProps = props;
        this.resourceGroup = props.resourceGroup;
        // Properties with default values
        const defaults = {
            autoMitigationEnabled: props.autoMitigationEnabled || false,
            workspaceAlertsStorageEnabled: props.workspaceAlertsStorageEnabled || false,
            enabled: props.enabled || true,
            skipQueryValidation: props.skipQueryValidation || true,
            metricMeasureColumn: props.criteriaMetricMeasureColumn || undefined,
        };
        const criteriaFailingPeriods = props.criteriaFailMinimumFailingPeriodsToTriggerAlert !== undefined &&
            props.criteriaFailNumberOfEvaluationPeriods !== undefined
            ? {
                minimumFailingPeriodsToTriggerAlert: props.criteriaFailMinimumFailingPeriodsToTriggerAlert,
                numberOfEvaluationPeriods: props.criteriaFailNumberOfEvaluationPeriods,
            }
            : undefined;
        const dimension = props.criteriaDimensionName &&
            props.criteriaDimensionOperator &&
            props.criteriaDimensionValues
            ? [
                {
                    name: props.criteriaDimensionName,
                    operator: props.criteriaDimensionOperator,
                    values: props.criteriaDimensionValues,
                },
            ]
            : undefined;
        const azurermMonitorQueryRuleAlert = new monitor_scheduled_query_rules_alert_v2_1.MonitorScheduledQueryRulesAlertV2(this, "queryrulealert", {
            ...defaults,
            name: props.name,
            resourceGroupName: props.resourceGroup.name,
            location: props.location,
            scopes: props.scopes,
            windowDuration: props.windowDuration,
            evaluationFrequency: props.evaluationFrequency,
            severity: props.severity,
            criteria: [
                {
                    operator: props.criteriaOperator,
                    query: props.criteriaQuery,
                    threshold: props.criteriaThreshold,
                    timeAggregationMethod: props.criteriatimeAggregationMethod,
                    dimension: dimension,
                    failingPeriods: criteriaFailingPeriods,
                },
            ],
            action: props.actionActionGroupId
                ? { actionGroups: props.actionActionGroupId }
                : undefined,
            tags: props.tags,
        });
        // Output
        this.id = azurermMonitorQueryRuleAlert.id;
        const cdktfTerraformOutputQueryRuleAlertId = new cdktf.TerraformOutput(this, "id", {
            value: azurermMonitorQueryRuleAlert.id,
        });
        cdktfTerraformOutputQueryRuleAlertId.overrideLogicalId("query_rule_alert_id");
    }
}
exports.QueryRuleAlert = QueryRuleAlert;
_a = JSII_RTTI_SYMBOL_1;
QueryRuleAlert[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_queryrulealert.QueryRuleAlert", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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