"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Account = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const storage_account_1 = require("@cdktf/provider-azurerm/lib/storage-account");
const storage_account_network_rules_1 = require("@cdktf/provider-azurerm/lib/storage-account-network-rules");
const container_1 = require("./container");
const fileshare_1 = require("./fileshare");
const queue_1 = require("./queue");
const table_1 = require("./table");
const lib_1 = require("../../core-azure/lib");
class Account extends lib_1.AzureResourceWithAlert {
    /**
     * Represents an Azure Storage Account within a Terraform deployment.
     *
     * This class is responsible for the creation and management of an Azure Storage Account, which is a scalable and secure service
     * for storing large amounts of unstructured data that can be accessed from anywhere in the world over HTTP or HTTPS. Common uses
     * of the Azure Storage Account include storing of blobs (objects), file shares, tables, and queues. This class provides methods
     * to manage storage resources, configure network rules, and integrate with Azure Active Directory for secure access management.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the storage account.
     * @param props - The properties required to configure the Azure Storage Account, as defined in the AccountProps interface. These include:
     *                - `name`: The name of the storage account, which must be unique within the Azure region.
     *                - `location`: The Azure region where the storage account will be created.
     *                - `resourceGroup`: The Azure Resource Group under which the storage account will be deployed.
     *                - `accountReplicationType`: The type of data replication to ensure data durability and availability.
     *                - `accountTier`: The performance tier that affects the type of hardware used for the storage account.
     *                - `tags`: A dictionary of tags to apply to the storage account for organizational purposes.
     *
     * Example usage:
     * ```typescript
     * const storageAccount = new Account(this, 'MyStorageAccount', {
     *   location: 'East US',
     *   name: 'myStorageAccount',
     *   resourceGroup: myResourceGroup,
     *   accountReplicationType: 'LRS',
     *   accountTier: 'Standard',
     *   enableHttpsTrafficOnly: true,
     *   tags: {
     *     environment: 'production'
     *   }
     * });
     * ```
     * This class sets up the storage account with the specified configurations, handles resource allocation, and applies security
     * settings based on the properties provided.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        this.containers = new Map();
        this.shares = new Map();
        this.tables = new Map();
        this.name = props.name;
        this.location = props.location;
        // default Storage Account Settings
        const defaults = {
            accountReplicationType: "LRS",
            accountTier: "Standard",
            enableHttpsTrafficOnly: true,
            accessTier: "Hot",
            isHnsEnabled: true,
            minTlsVersion: "TLS1_2",
            publicnetworkAccessEnabled: false,
            sharedAccessKeyEnabled: true,
            ...props,
        };
        // Create the storage account
        const storageAccount = new storage_account_1.StorageAccount(this, "storageaccount", {
            ...defaults,
            name: props.name,
            resourceGroupName: this.resourceGroup.name,
            location: props.location,
            tags: props.tags,
        });
        this.id = storageAccount.id;
        this.name = storageAccount.name;
        this.accountKind = storageAccount.accountKind;
        this.accountTier = storageAccount.accountTier;
    }
    /**
     * Adds a new container to the storage account.
     * @param name The name of the container. It must be unique within the storage account.
     * @param containerAccessType The level of public access to the container. Defaults to 'private'.
     * @param metadata Optional metadata for the container as key-value pairs.
     * @returns The created Container instance.
     * @throws Error if a container with the same name already exists within the storage account.
     *
     * This method creates a new container within the Azure storage account, allowing for the specification of access
     * level and metadata. If the container already exists, it throws an error to prevent duplication.
     *
     * Example usage:
     * ```typescript
     * const container = storageAccount.addContainer('myContainer', 'private', { owner: 'IT' });
     * ```
     */
    addContainer(name, containerAccessType, metadata) {
        if (this.containers.has(name)) {
            throw new Error(`Container '${name}' already exists.`);
        }
        const newContainer = new container_1.Container(this, name, {
            name: name,
            storageAccountName: this.name,
            containerAccessType: containerAccessType || "private",
            metadata: metadata || {},
        });
        this.containers.set(name, newContainer);
        return newContainer;
    }
    /**
     * Adds a new file share to the storage account.
     * @param name The name of the file share. Must be unique within the storage account.
     * @param props Optional properties for configuring the file share, such as quota and access tier.
     * @returns The created FileShare instance.
     * @throws Error if a file share with the same name already exists.
     *
     * This method facilitates the addition of a file share to the storage account, with optional settings for
     * capacity (quota) and data access frequency (access tier). If a file share with the same name exists, an error is thrown.
     *
     * Example usage:
     * ```typescript
     * const fileShare = storageAccount.addFileShare('myFileShare', { quota: 1024, accessTier: 'Hot' });
     * ```
     */
    addFileShare(name, props) {
        if (this.shares.has(name)) {
            throw new Error(`Share '${name}' already exists.`);
        }
        const defaults = {
            quota: props?.quota || 1024,
            accessTier: props?.accessTier || "Hot",
            enabledProtocol: props?.enabledProtocol || "SMB",
            acl: props?.acl || [],
            metadata: props?.metadata || {},
        };
        const newShare = new fileshare_1.FileShare(this, name, {
            ...defaults,
            name: name,
            storageAccountName: this.name,
        });
        this.shares.set(name, newShare);
        return newShare;
    }
    /**
     * Adds a new table to the storage account.
     * @param name The name of the table. Must be unique within the storage account.
     * @param acl Optional access control list for the table, specifying permissions.
     * @returns The created Table instance.
     * @throws Error if a table with the same name already exists.
     *
     * This method creates a new table within the storage account, optionally allowing for access control configurations.
     * It throws an error if a table with the same name already exists, ensuring uniqueness within the account.
     *
     * Example usage:
     * ```typescript
     * const table = storageAccount.addTable('myTable', [{ id: 'policy1', type: 'read' }]);
     * ```
     */
    addTable(name, acl) {
        if (this.tables.has(name)) {
            throw new Error(`Table '${name}' already exists.`);
        }
        const newTable = new table_1.Table(this, name, {
            name: name,
            storageAccountName: this.name,
            acl: acl,
        });
        this.tables.set(name, newTable);
        return newTable;
    }
    /**
     * Adds a new queue to the storage account.
     * @param name The name of the queue. Must be unique within the storage account.
     * @param metadata Optional metadata for the queue as key-value pairs.
     * @returns The created Queue instance.
     *
     * This method creates a new queue in the storage account, with optional metadata. It is useful for message queuing
     * in applications, enabling asynchronous task processing and inter-service communication.
     *
     * Example usage:
     * ```typescript
     * const queue = storageAccount.addQueue('myQueue', { priority: 'high' });
     * ```
     */
    addQueue(name, metadata) {
        return new queue_1.Queue(this, name, {
            name: name,
            storageAccountName: this.name,
            metadata: metadata,
        });
    }
    /**
     * Adds network rules to the storage account to control access based on IP and virtual network settings.
     * @param props Configuration properties for the network rules, including allowed IPs and virtual network subnet IDs.
     * @returns The configured network rules.
     *
     * This method configures network rules for the storage account, specifying which IPs and virtual networks can access
     * the storage resources. It allows detailed control over data security and access management.
     *
     * Example usage:
     * ```typescript
     * storageAccount.addNetworkRules({
     *   bypass: ['AzureServices'],
     *   defaultAction: 'Deny',
     *   ipRules: ['1.2.3.4/32'],
     *   virtualNetworkSubnetIds: ['subnetId'],
     * });
     * ```
     */
    addNetworkRules(props) {
        return new storage_account_network_rules_1.StorageAccountNetworkRulesA(this, "rules", {
            storageAccountId: this.id,
            bypass: props.bypass,
            defaultAction: props.defaultAction,
            ipRules: props.ipRules,
            virtualNetworkSubnetIds: props.virtualNetworkSubnetIds,
            privateLinkAccess: props.privateLinkAccess,
        });
    }
}
exports.Account = Account;
_a = JSII_RTTI_SYMBOL_1;
Account[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_storageaccount.Account", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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