"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Blob = exports.Container = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const storage_blob_1 = require("@cdktf/provider-azurerm/lib/storage-blob");
const storage_container_1 = require("@cdktf/provider-azurerm/lib/storage-container");
const constructs_1 = require("constructs");
class Container extends constructs_1.Construct {
    /**
     * Represents an Azure Storage Container within a specific Azure Storage Account.
     *
     * This class is designed for the creation and management of an Azure Storage Container, which serves as a unit of storage
     * that houses data objects, known as blobs. Containers are analogous to directories in a file system, and are used to organize
     * sets of blobs within a storage account. This class allows for granular control over blob storage, providing functionalities
     * such as setting access levels, managing metadata, and implementing security measures like encryption scopes.
     *
     * @param scope - The scope in which to define this construct, typically a reference to the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the container.
     * @param props - Configuration properties for the Azure Storage Container, based on the StorageContainerConfig interface. These include:
     *                - `name`: The name of the storage container, which must be unique within the associated storage account.
     *                - `storageAccountName`: The name of the storage account where this container is located.
     *                - `containerAccessType`: Defines the level of public access to the container. Options include 'container',
     *                  'blob', or 'none', controlling how the blobs within the container can be accessed.
     *                - `defaultEncryptionScope`: Specifies the default encryption scope for the container, enhancing data security.
     *                - `encryptionScopeOverrideEnabled`: Allows or prevents overriding the encryption scope on a per-blob basis.
     *                - `metadata`: Key-value pairs that provide additional information about the container, which can be used for
     *                  organizational, search, or billing purposes.
     *                - `timeouts`: Specifies custom timeout settings for CRUD operations on the container, providing control over
     *                  operation durations and retries.
     *
     * Example usage:
     * ```typescript
     * const storageContainer = new Container(this, 'MyContainer', {
     *   name: 'mycontainer',
     *   storageAccountName: 'mystorageaccount',
     *   containerAccessType: 'blob',
     *   defaultEncryptionScope: 'myEncryptionScope',
     *   encryptionScopeOverrideEnabled: true,
     *   metadata: {
     *     department: 'Finance'
     *   }
     * });
     * ```
     * This instantiation sets up a storage container with specified properties including access types and metadata. It is ready
     * to house blobs and provides methods to manage these blobs effectively.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.blobs = new Map();
        // Create a storage container
        const container = new storage_container_1.StorageContainer(this, "container", {
            name: props.name,
            storageAccountName: props.storageAccountName,
            containerAccessType: props.containerAccessType,
            metadata: props.metadata,
        });
        this.name = props.name;
        this.storageAccountName = props.storageAccountName;
        this.storageContainerName = container.name;
    }
    /**
     * Adds a blob to this Azure Storage Container.
     *
     * This method facilitates the addition of a blob to an Azure Storage Container managed by this class. It handles the creation and
     * configuration of the blob, including setting its type, source content, and metadata. This is useful for uploading various types
     * of unstructured data, such as images, videos, documents, or other binary files, into a cloud-based storage solution.
     *
     * @param blobName - The name of the blob to be added, which will be used as the blob's unique identifier within the container.
     * @param filePath - The file path or URL for the source of the blob's content. This specifies the location of the file to be uploaded.
     * @param props - Optional configuration properties for the blob, which include:
     *                - `type`: The type of the blob (e.g., 'Block', 'Append', 'Page'). Default is 'Block'.
     *                - `contentType`: The MIME type of the blob's content, such as 'application/octet-stream' for binary data. This helps browsers
     *                  and other clients handle the file correctly when it's downloaded or accessed.
     *                - `metadata`: A dictionary of key-value pairs to store as metadata with the blob. Metadata is typically used to store additional
     *                  details about the blob, such as tags, descriptions, or other attributes.
     *
     * @returns The newly created Blob object, which represents the blob added to the storage container.
     *
     * Example usage:
     * ```typescript
     * const storageBlob = storageContainer.addBlob('exampleBlob', './path/to/local/file.txt', {
     *   type: 'Block',
     *   contentType: 'text/plain',
     *   metadata: { customKey: 'customValue' }
     * });
     * ```
     * In this example, a new blob named 'exampleBlob' is added to the storage container. The content of the blob is sourced
     * from a local file specified by `filePath`. The blob is configured as a 'Block' type with 'text/plain' content type and
     * custom metadata. The method returns the blob instance for further use or reference.
     */
    addBlob(blobName, filePath, props) {
        const newStorageBlob = new Blob(this, blobName, {
            name: blobName,
            storageAccountName: this.storageAccountName,
            storageContainerName: this.storageContainerName,
            type: props?.type || "Block",
            source: filePath,
            contentType: props?.contentType || "application/octet-stream",
            metadata: props?.metadata || {},
        });
        this.blobs.set(blobName, newStorageBlob);
        return newStorageBlob;
    }
}
exports.Container = Container;
_a = JSII_RTTI_SYMBOL_1;
Container[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_storageaccount.Container", version: "0.0.3-pre.12" };
class Blob extends constructs_1.Construct {
    /**
     * Represents a blob within an Azure Storage Container.
     *
     * This class is responsible for the creation and management of a blob in an Azure Storage Container. Blobs are unstructured
     * data objects, which can include files like images, documents, videos, or any other file type. The Blob class provides a way
     * to manage these files in the cloud, allowing for scalable, durable, and accessible data storage. This class supports various
     * blob types such as block blobs for text and binary data, append blobs for log files, and page blobs for large volumes of
     * random access data.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) stack.
     * @param id - The unique identifier for this instance of the blob.
     * @param props - Configuration properties for the Azure Storage Blob. These properties may include:
     *                - `name`: The name of the blob, which must be unique within the container.
     *                - `storageAccountName`: The name of the storage account in which the blob is stored.
     *                - `storageContainerName`: The name of the storage container in which the blob resides.
     *                - `type`: The type of the blob (e.g., 'Block', 'Append', 'Page').
     *                - `source`: The source of the blob's content, which could be a path to a file or a URL.
     *                - `contentType`: The MIME type of the blob's content, such as 'application/octet-stream' for binary data.
     *                - `metadata`: A dictionary of strings that represents metadata to associate with the blob.
     *                - `accessTier`: The data access tier, affecting storage costs and data retrieval speeds.
     *                - `cacheControl`, `contentEncoding`, `contentDisposition`, `contentMd5`: Optional parameters for controlling the caching behavior,
     *                  encoding, content disposition, and integrity check of the blob.
     *
     * Example usage:
     * ```typescript
     * const myBlob = new Blob(this, 'MyBlob', {
     *   name: 'exampleblob',
     *   storageAccountName: 'mystorageaccount',
     *   storageContainerName: 'mycontainer',
     *   type: 'Block',
     *   source: './path/to/file.jpg',
     *   contentType: 'image/jpeg',
     *   metadata: {
     *     author: 'John Doe'
     *   }
     * });
     * ```
     * This class initializes a blob with the specified configurations and handles the uploading of content from the specified source.
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Create a storage container
        new storage_blob_1.StorageBlob(this, "blob", {
            name: props.name,
            storageAccountName: props.storageAccountName,
            storageContainerName: props.storageContainerName,
            type: props.type,
            source: props.source,
            contentType: props.contentType,
            metadata: props.metadata,
        });
        this.name = props.name;
    }
}
exports.Blob = Blob;
_b = JSII_RTTI_SYMBOL_1;
Blob[_b] = { fqn: "@microsoft/terraform-cdk-constructs.azure_storageaccount.Blob", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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