"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxVM = exports.WindowsVM = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const linux_virtual_machine_1 = require("@cdktf/provider-azurerm/lib/linux-virtual-machine");
const network_interface_1 = require("@cdktf/provider-azurerm/lib/network-interface");
const public_ip_1 = require("@cdktf/provider-azurerm/lib/public-ip");
const virtual_machine_extension_1 = require("@cdktf/provider-azurerm/lib/virtual-machine-extension");
const windows_virtual_machine_1 = require("@cdktf/provider-azurerm/lib/windows-virtual-machine");
const image_references_1 = require("./image-references");
const network_1 = require("../../azure-virtualnetwork/lib/network");
const lib_1 = require("../../core-azure/lib");
class WindowsVM extends lib_1.AzureResource {
    /**
     * Represents a Windows-based Virtual Machine (VM) within Microsoft Azure.
     *
     * This class is designed to provision and manage a Windows VM in Azure, allowing for detailed configuration including
     * the VM's size, the operating system image, network settings, and administrative credentials. It supports customization
     * of the OS disk, networking setup, and optional features like custom data scripts and boot diagnostics.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) application.
     * @param id - The unique identifier for this instance of the Windows VM, used within the scope for reference.
     * @param props - Configuration properties for the Windows Virtual Machine, derived from the WindowsVMProps interface. These include:
     *                - `location`: The geographic location where the VM will be hosted (e.g., "eastus").
     *                - `name`: The name of the VM, which must be unique within the resource group.
     *                - `resourceGroup`: Optional. Reference to the resource group for deployment.
     *                - `size`: The size specification of the VM (e.g., "Standard_B2s").
     *                - `adminUsername`: The administrator username for accessing the VM.
     *                - `adminPassword`: The administrator password for accessing the VM.
     *                - `sourceImageReference`: A reference to the specific Windows image to be used for the VM.
     *                - `sourceImageId`: The identifier for a custom image to use for the VM.
     *                - `tags`: Key-value pairs for resource tagging.
     *                - `osDisk`: Configuration for the VM's operating system disk.
     *                - `subnet`: Specifies the subnet within which the VM will be placed.
     *                - `publicIPAllocationMethod`: The method used to allocate a public IP address to the VM.
     *                - `customData`: Scripts or commands passed to the VM at startup.
     *                - `bootstrapCustomData`: Custom data used to trigger the Azure Custom Script Extension for VM setup tasks.
     *                - `bootDiagnosticsStorageURI`: URI for storage where VM boot diagnostics are collected.
     *
     * Example usage:
     * ```typescript
     * const vm = new WindowsVM(this, 'MyWindowsVM', {
     *   resourceGroup: myResourceGroup,
     *   name: 'myVM',
     *   size: 'Standard_DS1_v2',
     *   adminUsername: 'adminuser',
     *   adminPassword: 'securepassword123',
     *   sourceImageReference: { publisher: 'MicrosoftWindowsServer', offer: 'WindowsServer', sku: '2019-Datacenter', version: 'latest' },
     *   osDisk: { caching: 'ReadWrite', storageAccountType: 'Standard_LRS' },
     *   subnet: mySubnet,
     *   tags: { environment: 'production' }
     * });
     * ```
     * This class initializes a Windows VM with the specified configurations, handling details like network setup, OS installation,
     * and security settings, thus providing a robust infrastructure for hosting applications on Windows Server environments.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        // Default configurations for the virtual machine.
        const defaults = {
            name: props.name || this.node.path.split("/")[0],
            location: props.location || "eastus",
            size: props.size || "Standard_B2s",
            osDisk: props.osDisk || {
                caching: "ReadWrite",
                storageAccountType: "Standard_LRS",
            },
            sourceImageReference: props.sourceImageReference ||
                image_references_1.WindowsImageReferences.windowsServer2022DatacenterCore,
            subnet: props.subnet ||
                new network_1.Network(this, "vnet", {
                    resourceGroup: this.resourceGroup,
                }).subnets.default,
        };
        // Create Public IP if specified.
        let publicIpId;
        if (props.publicIPAllocationMethod) {
            const azurermPublicIp = new public_ip_1.PublicIp(this, "public-ip", {
                name: `pip-${defaults.name}`,
                resourceGroupName: this.resourceGroup.name,
                location: defaults.location,
                allocationMethod: props.publicIPAllocationMethod,
                tags: props.tags,
            });
            publicIpId = azurermPublicIp.id;
            this.publicIp = azurermPublicIp.ipAddress;
        }
        // Create the Network Interface.
        const azurermNetworkInterface = new network_interface_1.NetworkInterface(this, "nic", {
            ...defaults,
            name: `nic-${defaults.name}`,
            resourceGroupName: this.resourceGroup.name,
            ipConfiguration: [
                {
                    name: "internal",
                    subnetId: defaults.subnet.id,
                    privateIpAddressAllocation: "Dynamic",
                    publicIpAddressId: publicIpId,
                },
            ],
            tags: props.tags,
        });
        // Base64 encode custom data if provided.
        const customData = props.customData || props.boostrapCustomData;
        const base64CustomData = customData
            ? Buffer.from(customData).toString("base64")
            : undefined;
        // Create the Windows Virtual Machine.
        const azurermWindowsVirtualMachine = new windows_virtual_machine_1.WindowsVirtualMachine(this, "vm", {
            ...defaults,
            resourceGroupName: this.resourceGroup.name,
            adminUsername: props.adminUsername,
            adminPassword: props.adminPassword,
            tags: props.tags,
            networkInterfaceIds: [azurermNetworkInterface.id],
            sourceImageId: props.sourceImageId,
            customData: base64CustomData,
            bootDiagnostics: { storageAccountUri: props.bootDiagnosticsStorageURI },
            lifecycle: props.lifecycle,
        });
        this.id = azurermWindowsVirtualMachine.id;
        this.name = azurermWindowsVirtualMachine.name;
        // Bootstrap VM with custom script extension if bootstrap custom data is provided.
        if (props.boostrapCustomData) {
            new virtual_machine_extension_1.VirtualMachineExtension(this, "script-ext", {
                name: `${this.name}-script-ext`,
                virtualMachineId: this.id,
                publisher: "Microsoft.Compute",
                type: "CustomScriptExtension",
                typeHandlerVersion: "1.10",
                protectedSettings: '{"commandToExecute": "rename  C:\\\\AzureData\\\\CustomData.bin  postdeploy.ps1 & powershell -ExecutionPolicy Unrestricted -File C:\\\\AzureData\\\\postdeploy.ps1"}',
            });
        }
    }
}
exports.WindowsVM = WindowsVM;
_a = JSII_RTTI_SYMBOL_1;
WindowsVM[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_virtualmachine.WindowsVM", version: "0.0.3-pre.12" };
class LinuxVM extends lib_1.AzureResource {
    /**
     * Represents a Linux-based Virtual Machine (VM) within Microsoft Azure.
     *
     * This class is designed to provision and manage a Linux VM in Azure, facilitating detailed configuration including
     * VM size, the operating system image, network settings, and administrative credentials. It supports custom data scripts,
     * SSH configurations, and optional features like managed identity and boot diagnostics.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) application.
     * @param id - The unique identifier for this instance of the Linux VM, used within the scope for reference.
     * @param props - Configuration properties for the Linux Virtual Machine, derived from the LinuxVMProps interface. These include:
     *                - `location`: The geographic location where the VM will be hosted (e.g., "eastus").
     *                - `name`: The name of the VM, which must be unique within the resource group.
     *                - `resourceGroup`: Optional. Reference to the resource group for deployment.
     *                - `size`: The size specification of the VM (e.g., "Standard_B2s").
     *                - `availabilitySetId`: The ID of the availability set in which to include the VM.
     *                - `userData`: Custom data scripts to pass to the VM upon creation.
     *                - `adminSshKey`: SSH keys for secure access to the VM.
     *                - `zone`: The availability zone for deploying the VM.
     *                - `identity`: Managed identity settings for accessing other Azure services.
     *                - `additionalCapabilities`: Special capabilities like Ultra Disk support.
     *                - `sourceImageReference`: A reference to the specific Linux image to be used for the VM.
     *                - `sourceImageId`: The identifier for a custom image to use for the VM.
     *                - `tags`: Key-value pairs for resource tagging.
     *                - `osDisk`: Configuration for the VM's operating system disk.
     *                - `subnet`: Specifies the subnet within which the VM will be placed.
     *                - `publicIPAllocationMethod`: Method used to allocate a public IP address.
     *                - `customData`: Additional scripts or commands passed to the VM at startup.
     *                - `enableSshAzureADLogin`: Option to enable Azure AD login for SSH.
     *                - `bootDiagnosticsStorageURI`: URI for storage where VM boot diagnostics are collected.
     *
     * Example usage:
     * ```typescript
     * const linuxVM = new LinuxVM(this, 'MyLinuxVM', {
     *   resourceGroup: myResourceGroup,
     *   name: 'myVM',
     *   size: 'Standard_DS1_v2',
     *   adminUsername: 'adminuser',
     *   adminSshKey: [{ publicKey: 'ssh-rsa AAAAB...' }],
     *   sourceImageReference: { publisher: 'Canonical', offer: 'UbuntuServer', sku: '18.04-LTS', version: 'latest' },
     *   osDisk: { caching: 'ReadWrite', storageAccountType: 'Standard_LRS' },
     *   subnet: mySubnet,
     *   tags: { environment: 'production' }
     * });
     * ```
     * This class initializes a Linux VM with the specified configurations, handling details like network setup, OS installation,
     * and security settings, thus providing a robust infrastructure for hosting applications on Linux environments.
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Assigning the properties
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        // Extracting the name from the node path
        const pathName = this.node.path.split("/")[0];
        // Setting default configurations for the virtual machine
        const defaults = {
            name: props.name || pathName,
            adminUsername: props.adminUsername || `admin${pathName}`,
            location: props.location || "eastus",
            size: props.size || "Standard_B2s",
            osDisk: props.osDisk || {
                caching: "ReadWrite",
                storageAccountType: "Standard_LRS",
            },
            sourceImageReference: props.sourceImageReference ||
                image_references_1.WindowsImageReferences.windowsServer2022DatacenterCore,
            subnet: props.subnet ||
                new network_1.Network(this, "vnet", {
                    resourceGroup: this.resourceGroup,
                }).subnets.default,
        };
        // Create Public IP if specified
        let publicIpId;
        if (props.publicIPAllocationMethod) {
            const azurermPublicIp = new public_ip_1.PublicIp(this, "public-ip", {
                name: `pip-${defaults.name}`,
                resourceGroupName: this.resourceGroup.name,
                location: defaults.location,
                allocationMethod: props.publicIPAllocationMethod,
                tags: props.tags,
            });
            publicIpId = azurermPublicIp.id;
            this.publicIp = azurermPublicIp.ipAddress;
        }
        // Create the Network Interface
        const azurermNetworkInterface = new network_interface_1.NetworkInterface(this, "nic", {
            ...defaults,
            name: `nic-${defaults.name}`,
            resourceGroupName: this.resourceGroup.name,
            ipConfiguration: [
                {
                    name: "internal",
                    subnetId: defaults.subnet.id,
                    privateIpAddressAllocation: "Dynamic",
                    publicIpAddressId: publicIpId,
                },
            ],
            tags: props.tags,
        });
        // Create the Linux Virtual Machine
        const azurermLinuxVirtualMachine = new linux_virtual_machine_1.LinuxVirtualMachine(this, "vm", {
            ...defaults,
            resourceGroupName: this.resourceGroup.name,
            adminPassword: props.adminPassword,
            tags: props.tags,
            networkInterfaceIds: [azurermNetworkInterface.id],
            sourceImageId: props.sourceImageId,
            customData: props.customData
                ? Buffer.from(props.customData).toString("base64")
                : undefined,
            userData: props.userData
                ? Buffer.from(props.userData).toString("base64")
                : undefined,
            availabilitySetId: props.availabilitySetId,
            adminSshKey: props.adminSshKey,
            bootDiagnostics: { storageAccountUri: props.bootDiagnosticsStorageURI },
            zone: props.zone,
            identity: props.identity,
            additionalCapabilities: props.additionalCapabilities,
            secret: props.secret,
            disablePasswordAuthentication: props.adminPassword ? false : true,
            lifecycle: props.lifecycle,
        });
        // Assigning the VM's ID and name to the class properties
        this.id = azurermLinuxVirtualMachine.id;
        this.name = azurermLinuxVirtualMachine.name;
        // Enable SSH Azure AD Login if specified
        if (props.enableSshAzureADLogin) {
            new virtual_machine_extension_1.VirtualMachineExtension(this, "AADSSHlogin", {
                name: "AADSSHLoginForLinux",
                virtualMachineId: this.id,
                publisher: "Microsoft.Azure.ActiveDirectory",
                type: "AADSSHLoginForLinux",
                typeHandlerVersion: "1.0",
                tags: props.tags,
            });
        }
    }
}
exports.LinuxVM = LinuxVM;
_b = JSII_RTTI_SYMBOL_1;
LinuxVM[_b] = { fqn: "@microsoft/terraform-cdk-constructs.azure_virtualmachine.LinuxVM", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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