"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const storage_account_1 = require("@cdktf/provider-azurerm/lib/storage-account");
const subnet_1 = require("@cdktf/provider-azurerm/lib/subnet");
const virtual_network_1 = require("@cdktf/provider-azurerm/lib/virtual-network");
const cdktf_1 = require("cdktf");
const vm = require("..");
const testing_1 = require("../../testing");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
describe("Example of deploying a Linux Virtual Machine", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "southcentralus",
        });
        const vnet = new virtual_network_1.VirtualNetwork(stack, "vnet", {
            name: `vnet-${randomName}`,
            location: resourceGroup.location,
            resourceGroupName: resourceGroup.name,
            addressSpace: ["10.0.0.0/16"],
        });
        const subnet = new subnet_1.Subnet(stack, "subnet1", {
            name: "subnet1",
            resourceGroupName: resourceGroup.name,
            virtualNetworkName: vnet.name,
            addressPrefixes: ["10.0.1.0/24"],
        });
        const storage = new storage_account_1.StorageAccount(stack, "storage", {
            name: `sta${randomName}08l98`,
            resourceGroupName: resourceGroup.name,
            location: resourceGroup.location,
            accountReplicationType: "LRS",
            accountTier: "Standard",
            minTlsVersion: "TLS1_2",
            publicNetworkAccessEnabled: false,
            networkRules: {
                bypass: ["AzureServices"],
                defaultAction: "Deny",
            },
        });
        const linuxVm = new vm.LinuxVM(stack, "vm", {
            name: randomName,
            location: "southcentralus",
            resourceGroup: resourceGroup,
            size: "Standard_D2as_v4",
            adminUsername: "testadmin",
            adminPassword: "Password1234!",
            osDisk: {
                caching: "ReadWrite",
                storageAccountType: "Standard_LRS",
            },
            sourceImageReference: {
                publisher: "Canonical",
                offer: "0001-com-ubuntu-server-jammy",
                sku: "22_04-lts-gen2",
                version: "latest",
            },
            subnet: subnet,
            publicIPAllocationMethod: "Static",
            userData: "#!/bin/bash\nsudo apt-get install -y apache2",
            enableSshAzureADLogin: true,
            identity: {
                type: "SystemAssigned",
            },
            bootDiagnosticsStorageURI: storage.primaryBlobEndpoint,
            lifecycle: {
                ignoreChanges: ["tags", "identity"],
            },
        });
        // Diag Settings
        linuxVm.addDiagSettings({
            name: "diagsettings",
            storageAccountId: storage.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        // RBAC
        linuxVm.addAccess(clientConfig.objectId, "Contributor");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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