"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const storage_account_1 = require("@cdktf/provider-azurerm/lib/storage-account");
const subnet_1 = require("@cdktf/provider-azurerm/lib/subnet");
const virtual_network_1 = require("@cdktf/provider-azurerm/lib/virtual-network");
const cdktf_1 = require("cdktf");
const vm = require("..");
const testing_1 = require("../../testing");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
describe("Example of deploying a Windows Virtual Machine", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "southcentralus",
        });
        const vnet = new virtual_network_1.VirtualNetwork(stack, "vnet", {
            name: `vnet-${randomName}`,
            location: resourceGroup.location,
            resourceGroupName: resourceGroup.name,
            addressSpace: ["10.0.0.0/16"],
        });
        const subnet = new subnet_1.Subnet(stack, "subnet1", {
            name: "subnet1",
            resourceGroupName: resourceGroup.name,
            virtualNetworkName: vnet.name,
            addressPrefixes: ["10.0.1.0/24"],
        });
        const storage = new storage_account_1.StorageAccount(stack, "storage", {
            name: `sta${randomName}87u98`,
            resourceGroupName: resourceGroup.name,
            location: resourceGroup.location,
            accountReplicationType: "LRS",
            accountTier: "Standard",
            minTlsVersion: "TLS1_2",
            publicNetworkAccessEnabled: false,
            networkRules: {
                bypass: ["AzureServices"],
                defaultAction: "Deny",
            },
        });
        const winVm = new vm.WindowsVM(stack, "vm", {
            name: randomName,
            location: "southcentralus",
            resourceGroup: resourceGroup,
            size: "Standard_D2as_v4",
            adminUsername: "testadmin",
            adminPassword: "Password1234!",
            osDisk: {
                caching: "ReadWrite",
                storageAccountType: "Standard_LRS",
            },
            sourceImageReference: {
                publisher: "MicrosoftWindowsServer",
                offer: "WindowsServer",
                sku: "2019-Datacenter",
                version: "latest",
            },
            subnet: subnet,
            publicIPAllocationMethod: "Static",
            boostrapCustomData: "Install-WindowsFeature -Name Web-Server; $website = '<h1>Hello World!</h1>'; Set-Content \"C:\\inetpub\\wwwroot\\iisstart.htm\" $website",
            bootDiagnosticsStorageURI: storage.primaryBlobEndpoint,
            lifecycle: {
                ignoreChanges: ["tags", "identity"],
            },
        });
        // Diag Settings
        winVm.addDiagSettings({
            name: "diagsettings",
            storageAccountId: storage.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        // RBAC
        winVm.addAccess(clientConfig.objectId, "Contributor");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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