"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const storage_account_1 = require("@cdktf/provider-azurerm/lib/storage-account");
const subnet_1 = require("@cdktf/provider-azurerm/lib/subnet");
const virtual_network_1 = require("@cdktf/provider-azurerm/lib/virtual-network");
const cdktf_1 = require("cdktf");
const vmss = require("..");
const testing_1 = require("../../testing");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
describe("Example of deploying a Linux VMSS Cluster", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(12);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "southcentralus",
        });
        const vnet = new virtual_network_1.VirtualNetwork(stack, "vnet", {
            name: `vnet-${randomName}`,
            location: resourceGroup.location,
            resourceGroupName: resourceGroup.name,
            addressSpace: ["10.0.0.0/16"],
        });
        const subnet = new subnet_1.Subnet(stack, "subnet1", {
            name: "subnet1",
            resourceGroupName: resourceGroup.name,
            virtualNetworkName: vnet.name,
            addressPrefixes: ["10.0.1.0/24"],
        });
        const storage = new storage_account_1.StorageAccount(stack, "storage", {
            name: `sta${randomName}88r93`,
            resourceGroupName: resourceGroup.name,
            location: resourceGroup.location,
            accountReplicationType: "LRS",
            accountTier: "Standard",
            minTlsVersion: "TLS1_2",
            publicNetworkAccessEnabled: false,
            networkRules: {
                bypass: ["AzureServices"],
                defaultAction: "Deny",
            },
        });
        const vm = new vmss.LinuxCluster(stack, "vm", {
            name: randomName,
            location: "southcentralus",
            resourceGroup: resourceGroup,
            sku: "Standard_D2as_v4",
            adminUsername: "testadmin",
            adminPassword: "Password1234!",
            osDisk: {
                caching: "ReadWrite",
                storageAccountType: "Standard_LRS",
            },
            sourceImageReference: {
                publisher: "Canonical",
                offer: "0001-com-ubuntu-server-jammy",
                sku: "22_04-lts-gen2",
                version: "latest",
            },
            subnet: subnet,
            publicIPAddress: [
                {
                    name: "testpublicip",
                },
            ],
            userData: "#!/bin/bash\nsudo apt-get install -y apache2",
            enableSshAzureADLogin: true,
            identity: {
                type: "SystemAssigned",
            },
            bootDiagnosticsStorageURI: storage.primaryBlobEndpoint,
            lifecycle: {
                ignoreChanges: ["tags", "identity"],
            },
        });
        // Diag Settings
        vm.addDiagSettings({
            name: "diagsettings",
            storageAccountId: storage.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        // RBAC
        vm.addAccess(clientConfig.objectId, "Contributor");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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