"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const data_azurerm_client_config_1 = require("@cdktf/provider-azurerm/lib/data-azurerm-client-config");
const provider_1 = require("@cdktf/provider-azurerm/lib/provider");
const resource_group_1 = require("@cdktf/provider-azurerm/lib/resource-group");
const storage_account_1 = require("@cdktf/provider-azurerm/lib/storage-account");
const subnet_1 = require("@cdktf/provider-azurerm/lib/subnet");
const virtual_network_1 = require("@cdktf/provider-azurerm/lib/virtual-network");
const cdktf_1 = require("cdktf");
const vmss = require("..");
const testing_1 = require("../../testing");
const randomName_1 = require("../../util/randomName");
require("cdktf/lib/testing/adapters/jest");
describe("Example of deploying a Windows VMSS Cluster", () => {
    let stack;
    let fullSynthResult;
    const streamOutput = process.env.STREAM_OUTPUT !== "false";
    beforeEach(() => {
        const app = cdktf_1.Testing.app();
        stack = new cdktf_1.TerraformStack(app, "test");
        const randomName = (0, randomName_1.generateRandomName)(9);
        const clientConfig = new data_azurerm_client_config_1.DataAzurermClientConfig(stack, "CurrentClientConfig", {});
        new provider_1.AzurermProvider(stack, "azureFeature", { features: {} });
        // Create a resource group
        const resourceGroup = new resource_group_1.ResourceGroup(stack, "rg", {
            name: `rg-${randomName}`,
            location: "southcentralus",
        });
        const vnet = new virtual_network_1.VirtualNetwork(stack, "vnet", {
            name: `vnet-${randomName}`,
            location: resourceGroup.location,
            resourceGroupName: resourceGroup.name,
            addressSpace: ["10.0.0.0/16"],
        });
        const subnet = new subnet_1.Subnet(stack, "subnet1", {
            name: "subnet1",
            resourceGroupName: resourceGroup.name,
            virtualNetworkName: vnet.name,
            addressPrefixes: ["10.0.1.0/24"],
        });
        const storage = new storage_account_1.StorageAccount(stack, "storage", {
            name: `sta${randomName}88s96`,
            resourceGroupName: resourceGroup.name,
            location: resourceGroup.location,
            accountReplicationType: "LRS",
            accountTier: "Standard",
            minTlsVersion: "TLS1_2",
            publicNetworkAccessEnabled: false,
            networkRules: {
                bypass: ["AzureServices"],
                defaultAction: "Deny",
            },
        });
        const vm = new vmss.WindowsCluster(stack, "vmss", {
            name: randomName,
            location: "southcentralus",
            resourceGroup: resourceGroup,
            sku: "Standard_D2as_v4",
            adminUsername: "testadmin",
            adminPassword: "Password1234!",
            osDisk: {
                caching: "ReadWrite",
                storageAccountType: "Standard_LRS",
            },
            sourceImageReference: {
                publisher: "MicrosoftWindowsServer",
                offer: "WindowsServer",
                sku: "2019-Datacenter",
                version: "latest",
            },
            subnet: subnet,
            publicIPAddress: [
                {
                    name: "testpublicip",
                },
            ],
            boostrapCustomData: "Install-WindowsFeature -Name Web-Server; $website = '<h1>Hello World!</h1>'; Set-Content \"C:\\inetpub\\wwwroot\\iisstart.htm\" $website",
            bootDiagnosticsStorageURI: storage.primaryBlobEndpoint,
            lifecycle: {
                ignoreChanges: ["tags", "identity"],
            },
        });
        // Diag Settings
        vm.addDiagSettings({
            name: "diagsettings",
            storageAccountId: storage.id,
            metric: [
                {
                    category: "AllMetrics",
                },
            ],
        });
        // RBAC
        vm.addAccess(clientConfig.objectId, "Contributor");
        fullSynthResult = cdktf_1.Testing.fullSynth(stack); // Save the result for reuse
    });
    afterEach(() => {
        try {
            (0, testing_1.TerraformDestroy)(fullSynthResult, streamOutput);
        }
        catch (error) {
            console.error("Error during Terraform destroy:", error);
        }
    });
    it("check if stack can be deployed", () => {
        (0, testing_1.TerraformApplyAndCheckIdempotency)(fullSynthResult, streamOutput); // Set to true to stream output
    });
});
//# sourceMappingURL=data:application/json;base64,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