"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Network = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const subnet_1 = require("@cdktf/provider-azurerm/lib/subnet");
const virtual_network_1 = require("@cdktf/provider-azurerm/lib/virtual-network");
const peering_1 = require("./peering");
const lib_1 = require("../../core-azure/lib");
class Network extends lib_1.AzureResource {
    /**
     * Represents an Azure Virtual Network (VNet) within Microsoft Azure.
     *
     * This class is responsible for the creation and management of a virtual network, which provides an isolated environment
     * where Azure resources, such as VMs and databases, can securely communicate with each other, the internet, and on-premises
     * networks. It supports configurations such as multiple address spaces and subnets, enabling complex networking scenarios.
     *
     * @param scope - The scope in which to define this construct, typically representing the Cloud Development Kit (CDK) application.
     * @param id - The unique identifier for this instance of the network, used within the scope for reference.
     * @param props - Configuration properties for the Azure Virtual Network, derived from the NetworkProps interface. These include:
     *                - `resourceGroup`: Optional. Reference to the resource group for deployment.
     *                - `name`: Optional. The name of the virtual network. If not provided, a default name will be assigned.
     *                - `location`: Optional. The Azure region where the virtual network will be deployed. Defaults to the resource group's region.
     *                - `addressSpace`: Optional. A list of CIDR blocks that define the address spaces of the virtual network.
     *                - `subnets`: Optional. An array of subnets to be created within the virtual network, each defined by a name and a CIDR block.
     *
     * Example usage:
     * ```typescript
     * const network = new Network(this, 'MyVirtualNetwork', {
     *   resourceGroup: myResourceGroup,
     *   name: 'myVNet',
     *   location: 'West US',
     *   addressSpace: ['10.0.0.0/16'],
     *   subnets: [{ name: 'subnet1', addressPrefixes: ['10.0.1.0/24'] }]
     * });
     * ```
     * This class initializes a virtual network with the specified configurations and handles the provisioning of subnets
     * within the network, providing a foundational networking layer for hosting cloud resources.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.subnets = {}; // Map of subnet name to Subnet object
        this.props = props;
        this.resourceGroup = this.setupResourceGroup(props);
        const defaults = {
            name: props.name || `vnet-${this.node.path.split("/")[0]}`,
            location: props.location || "eastus",
            addressSpace: props.addressSpace || ["10.0.0.0/16"],
            subnets: props.subnets || [
                {
                    name: "default",
                    addressPrefixes: ["10.1.0.0/24"],
                },
            ],
        };
        // Create a virtual network
        const vnet = new virtual_network_1.VirtualNetwork(this, "vnet", {
            ...defaults,
            resourceGroupName: this.resourceGroup.name,
        });
        this.name = vnet.name;
        this.id = vnet.id;
        this.resourceGroup = this.resourceGroup;
        this.virtualNetwork = vnet;
        // Create subnets within the virtual network
        for (const subnetConfig of defaults.subnets) {
            const subnet = new subnet_1.Subnet(this, subnetConfig.name, {
                name: subnetConfig.name,
                resourceGroupName: this.resourceGroup.name,
                virtualNetworkName: vnet.name,
                addressPrefixes: subnetConfig.addressPrefixes,
            });
            this.subnets[subnetConfig.name] = subnet; // Populate the subnetsMap
        }
    }
    /**
     * Establishes a peering connection between this virtual network and another remote virtual network.
     *
     * This method configures a two-way peering connection, allowing resources in both virtual networks to communicate
     * seamlessly. It sets up peering settings such as network access, traffic forwarding, and gateway transit based on
     * provided configurations.
     *
     * @param remoteVirtualNetwork - The remote virtual network with which to establish a peering connection.
     * @param localPeerSettings - Optional settings applied from this virtual network to the remote virtual network.
     *                             Controls aspects like virtual network access, traffic forwarding, and use of gateways.
     * @param remotePeerSettings - Optional settings applied from the remote virtual network to this virtual network.
     *                             Allows customization of how the remote network interacts with this one.
     *
     * Example usage:
     * ```typescript
     * // Assuming 'this' is a reference to a local virtual network instance.
     * const partnerVNet = new Network(this, 'PartnerVNet', { ... });
     * this.addVnetPeering(partnerVNet, {
     *   allowVirtualNetworkAccess: true,
     *   allowForwardedTraffic: false,
     *   allowGatewayTransit: true,
     *   useRemoteGateways: false
     * }, {
     *   allowVirtualNetworkAccess: true,
     *   allowForwardedTraffic: true,
     *   allowGatewayTransit: false,
     *   useRemoteGateways: false
     * });
     * ```
     * This method invokes the `Peer` class to create a peering between 'this' virtual network and 'partnerVNet'.
     * The settings control traffic behavior and access permissions in both directions of the peering.
     */
    addVnetPeering(remoteVirtualNetwork, localPeerSettings, remotePeerSettings) {
        const vnetPeerProps = {
            virtualNetwork: this,
            remoteVirtualNetwork: remoteVirtualNetwork,
            localToRemoteSettings: localPeerSettings,
            remoteToLocalSettings: remotePeerSettings,
        };
        new peering_1.Peer(this, remoteVirtualNetwork.id, vnetPeerProps);
    }
}
exports.Network = Network;
_a = JSII_RTTI_SYMBOL_1;
Network[_a] = { fqn: "@microsoft/terraform-cdk-constructs.azure_virtualnetwork.Network", version: "0.0.3-pre.12" };
//# sourceMappingURL=data:application/json;base64,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